/*  Small compiler
 *
 *  Binary code generation
 */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>     /* for macro max() */
#include <string.h>
#include <ctype.h>
#include "sc.h"

#pragma pack(1) /* file structures *must* be byte-aligned */
#pragma option -a-

typedef void (*OPCODE_PROC)(FILE *fbin,char *params,int opcode);

typedef struct {
  int opcode;
  char *name;
  int segment;          /* _incseg=parse in cseg, _indseg=parse in dseg */
  OPCODE_PROC func;
} OPCODE;

/* apparently, strtol() does not work correctly on very large (unsigned)
 * hexadecimal values */
static unsigned long hex2long(char *s,char **n)
{
  unsigned long result=0L;
  int digit;

  /* NB. only works for lower case letters */
  for ( ;; ) {
    assert(!(*s>='A' && *s<='Z'));      /* lower case only */
    if (*s>='0' && *s<='9')
      digit=*s-'0';
    else if (*s>='a' && *s<='f')
      digit=*s-'a' + 10;
    else
      break;
    result=(result<<4) | digit;
    s++;
  } /* for */
  if (n!=NULL)
    *n=s;
  return result;
}

#if defined BIG_ENDIAN
static unsigned long *align16(unsigned short *v)
{
  unsigned char *s = (unsigned char *)v;
  unsigned char t;

  /* swap two bytes */
  t=s[0];
  s[0]=s[1];
  s[1]=t;
  return v;
}

static unsigned long *align32(unsigned long *v)
{
  unsigned char *s = (unsigned char *)v;
  unsigned char t;

  /* swap outer two bytes */
  t=s[0];
  s[0]=s[3];
  s[3]=t;
  /* swap inner two bytes */
  t=s[1];
  s[1]=s[2];
  s[2]=t;
  return v;
}
#else
#  define align16(v)    (v)
#  define align32(v)    (v)
#endif

#pragma argsused
static void noop(FILE *fbin,char *params,int opcode)
{
}

#pragma argsused
static void parm0(FILE *fbin,char *params,int opcode)
{
  fwrite(&opcode,1,1,fbin);
}

static void parm1(FILE *fbin,char *params,int opcode)
{
  unsigned long p=hex2long(params,NULL);
  fwrite(&opcode,1,1,fbin);
  fwrite(align32(&p),4,1,fbin);
}

static void parm2(FILE *fbin,char *params,int opcode)
{
  unsigned long p1,p2;

  p1=hex2long(params,&params);
  p2=hex2long(params,NULL);
  fwrite(&opcode,1,1,fbin);
  fwrite(align32(&p1),4,1,fbin);
  fwrite(align32(&p2),4,1,fbin);
}

#pragma argsused
static void do_dump(FILE *fbin,char *params,int opcode)
{
  unsigned long p;

  while (*params!='\0') {
    p=hex2long(params,&params);
    fwrite(align32(&p),4,1,fbin);
    while (isspace(*params))
      params++;
  } /* while */
}

static void do_call(FILE *fbin,char *params,int opcode)
{
  char name[_namemax+1];
  int i;
  symbol *sym;
  unsigned long p;

  for (i=0; !isspace(*params); i++,params++) {
    assert(*params!='\0');
    assert(i<_namemax);
    name[i]=*params;
  } /* for */
  name[i]='\0';

  /* look up the function address */
  sym=findglb(name);        /* already in symbol table? */
  assert(sym!=NULL);
  assert(sym->ident==_functn || sym->ident==_reffunc);
  assert(sym->vclass==_global);

  p=sym->addr;
  fwrite(&opcode,1,1,fbin);
  fwrite(align32(&p),sizeof(cell),1,fbin);
}

static void do_jump(FILE *fbin,char *params,int opcode)
{
  int i;
  unsigned long p;

  i=(int)hex2long(params,NULL);
  assert(i>=0 && i<_numlabels);

  p=lbltab[i];
  fwrite(&opcode,1,1,fbin);
  fwrite(align32(&p),sizeof(cell),1,fbin);
}

static void do_file(FILE *fbin,char *params,int opcode)
{
  char *endptr;
  unsigned long p;
  int i;

  for (endptr=params; !isspace(*endptr) && endptr!='\0'; endptr++)
    /* nothing */;
  assert(*endptr==' ');

  i=(int)(endptr-params);
  assert(i>0 && i<256);
  p=hex2long(endptr,NULL);

  fwrite(&opcode,1,1,fbin);
  fwrite(&i,1,1,fbin);
  fwrite(params,1,i,fbin);
  fwrite(align32(&p),sizeof(cell),1,fbin);
}

static void do_symbol(FILE *fbin,char *params,int opcode)
{
  char *endptr;
  unsigned long offset;
  int i,mclass,type;

  for (endptr=params; !isspace(*endptr) && endptr!='\0'; endptr++)
    /* nothing */;
  assert(*endptr==' ');

  i=(int)(endptr-params);
  assert(i>0 && i<_namemax);
  offset=hex2long(endptr,&endptr);
  mclass=(int)hex2long(endptr,&endptr);
  type=(int)hex2long(endptr,NULL);

  fwrite(&opcode,1,1,fbin);
  fwrite(&i,1,1,fbin);
  fwrite(params,1,i,fbin);
  fwrite(align32(&offset),sizeof(cell),1,fbin);
  fwrite(&mclass,1,1,fbin);
  fwrite(&type,1,1,fbin);
}

static OPCODE opcodes[] = {
  /* special "directives" (no opcodes) */
  {  0, "code",       0,       noop },
  {  0, "data",       0,       noop },
  {  0, "export",     0,       noop },
  {  0, "stksize",    0,       noop },
  {  0, "dump",       _indseg, do_dump },

  /* loading from memory and storing into memory */
  {  1, "load.pri",   _incseg, parm1 },
  {  2, "load.alt",   _incseg, parm1 },
  {  3, "load.s.pri", _incseg, parm1 },
  {  4, "load.s.alt", _incseg, parm1 },
  {  5, "load.i",     _incseg, parm0 },
  {  6, "lref.pri",   _incseg, parm1 },
  {  7, "lref.alt",   _incseg, parm1 },
  {  8, "lref.s.pri", _incseg, parm1 },
  {  9, "lref.s.alt", _incseg, parm1 },
  { 10, "const.pri",  _incseg, parm1 },
  { 11, "const.alt",  _incseg, parm1 },
  { 12, "addr.pri",   _incseg, parm1 },
  { 13, "addr.alt",   _incseg, parm1 },
  { 14, "stor.pri",   _incseg, parm1 },
  { 15, "stor.alt",   _incseg, parm1 },
  { 16, "stor.s.pri", _incseg, parm1 },
  { 17, "stor.s.alt", _incseg, parm1 },
  { 18, "stor.i",     _incseg, parm0 },
  { 19, "sref.pri",   _incseg, parm1 },
  { 20, "sref.alt",   _incseg, parm1 },
  { 21, "sref.s.pri", _incseg, parm1 },
  { 22, "sref.s.alt", _incseg, parm1 },
  /* loading and storing of (special) registers */
  { 23, "lidx",       _incseg, parm0 },
  { 24, "idxaddr",    _incseg, parm0 },
  { 25, "lcrtl",      _incseg, parm1 },
  { 26, "scrtl",      _incseg, parm1 },
  { 27, "move.pri",   _incseg, parm0 },
  { 28, "move.alt",   _incseg, parm0 },
  { 29, "xchg",       _incseg, parm0 },

  /* stack and heap manipulation */
  { 30, "push.pri",   _incseg, parm0 },
  { 31, "push.alt",   _incseg, parm0 },
  { 32, "push.c",     _incseg, parm1 },
  { 33, "push",       _incseg, parm1 },
  { 34, "push.s",     _incseg, parm1 },
  { 35, "pop.pri",    _incseg, parm0 },
  { 36, "pop.alt",    _incseg, parm0 },
  { 37, "stack",      _incseg, parm1 },
  { 38, "heap",       _incseg, parm1 },

  /* jumps, function calls (and returns) */
  { 39, "proc",       _incseg, parm0 },
  { 40, "ret",        _incseg, parm0 },
  { 41, "retn",       _incseg, parm0 },
  { 42, "call",       _incseg, do_call },
  { 43, "call.i",     _incseg, parm1 },  /* always a number */
  { 44, "jump",       _incseg, do_jump },
  { 45, "jrel",       _incseg, parm1 },  /* always a number */
  { 46, "jzer",       _incseg, do_jump },
  { 47, "jnz",        _incseg, do_jump },
  { 48, "jeq",        _incseg, do_jump },
  { 49, "jneq",       _incseg, do_jump },
  { 50, "jless",      _incseg, do_jump },
  { 51, "jleq",       _incseg, do_jump },
  { 52, "jgrtr",      _incseg, do_jump },
  { 53, "jgeq",       _incseg, do_jump },
  { 54, "jsless",     _incseg, do_jump },
  { 55, "jsleq",      _incseg, do_jump },
  { 56, "jsgrtr",     _incseg, do_jump },
  { 57, "jsgeq",      _incseg, do_jump },

  /* shift instructions */
  { 58, "shl",        _incseg, parm0 },
  { 59, "shr",        _incseg, parm0 },
  { 60, "sshr",       _incseg, parm0 },
  { 61, "shl.c.pri",  _incseg, parm1 },
  { 62, "shl.c.alt",  _incseg, parm1 },
  { 63, "shr.c.pri",  _incseg, parm1 },
  { 64, "shr.c.alt",  _incseg, parm1 },

  /* arithmetic and bitwise instructions */
  { 65, "smul",       _incseg, parm0 },
  { 66, "sdiv",       _incseg, parm0 },
  { 67, "sdiv.alt",   _incseg, parm0 },
  { 68, "umul",       _incseg, parm0 },
  { 69, "udiv",       _incseg, parm0 },
  { 70, "udiv.alt",   _incseg, parm0 },
  { 71, "add",        _incseg, parm0 },
  { 72, "sub",        _incseg, parm0 },
  { 73, "sub.alt",    _incseg, parm0 },
  { 74, "and",        _incseg, parm0 },
  { 75, "or",         _incseg, parm0 },
  { 76, "xor",        _incseg, parm0 },
  { 77, "not",        _incseg, parm0 },
  { 78, "neg",        _incseg, parm0 },
  { 79, "invert",     _incseg, parm0 },
  { 80, "add.c",      _incseg, parm1 },
  { 81, "smul.c",     _incseg, parm1 },
  { 82, "zero.pri",   _incseg, parm0 },
  { 83, "zero.alt",   _incseg, parm0 },
  { 84, "zero",       _incseg, parm1 },
  { 85, "zero.s",     _incseg, parm1 },
  { 86, "sign.pri",   _incseg, parm0 },
  { 87, "sign.alt",   _incseg, parm0 },

  /* relational operators */
  { 88, "eq",         _incseg, parm0 },
  { 89, "neq",        _incseg, parm0 },
  { 90, "less",       _incseg, parm0 },
  { 91, "leq",        _incseg, parm0 },
  { 92, "grtr",       _incseg, parm0 },
  { 93, "geq",        _incseg, parm0 },
  { 94, "sless",      _incseg, parm0 },
  { 95, "sleq",       _incseg, parm0 },
  { 96, "sgrtr",      _incseg, parm0 },
  { 97, "sgeq",       _incseg, parm0 },
  { 98, "eq.c.pri",   _incseg, parm1 },
  { 99, "eq.c.alt",   _incseg, parm1 },

  /* increment/decrement */
  {100, "inc.pri",    _incseg, parm0 },
  {101, "inc.alt",    _incseg, parm0 },
  {102, "inc",        _incseg, parm1 },
  {103, "inc.s",      _incseg, parm1 },
  {104, "inc.i",      _incseg, parm1 },
  {105, "dec.pri",    _incseg, parm0 },
  {106, "dec.alt",    _incseg, parm0 },
  {107, "dec",        _incseg, parm1 },
  {108, "dec.s",      _incseg, parm1 },
  {109, "dec.i",      _incseg, parm1 },

  /* special instructions */
  {110, "movs",       _incseg, parm1 },
  {111, "cmps",       _incseg, parm1 },
  {112, "fill",       _incseg, parm1 },
  {113, "halt",       _incseg, parm1 },
  {114, "bounds",     _incseg, parm1 },
  {115, "sysreq.pri", _incseg, parm0 },
  {116, "sysreq.c",   _incseg, parm1 },

  /* debugging opcodes */
  {117, "file",       _incseg, do_file },
  {118, "line",       _incseg, parm2 },
  {119, "symbol",     _incseg, do_symbol },

  /* terminator */
  {  0, NULL,         0,       noop }
};

typedef struct {
  long size;    /* size of the "file" */
  short magic;  /* signature */
  char file_version;    /* file format version */
  char amx_version;     /* required version of the AMX */
  short flags;
  short defsize;
  long cod;     /* initial value of COD - code block */
  long dat;     /* initial value of DAT - data block */
  long hea;     /* initial value of HEA - start of the heap */
  long stp;     /* initial value of STP - stack top */
  long cip;     /* initial valie of CIP - the instruction pointer */
  short num_exports;    /* number of items in the "exported functions" table */
  long exports;         /* offset to the "exported functions" table */
  short num_natives;    /* number of items in the "native functions" table */
  long natives;         /* offset to the "native functions" table */
  short num_modules;    /* number of items in the "module table" */
  long modules;         /* offset to the "module table" */
} AMX_HEADER;

typedef struct {
  cell address;
  char name[_namemax+1];
} FUNCSTUB;

void assemble(FILE *fout,FILE *fin)
{
  AMX_HEADER hdr;
  FUNCSTUB func;
  int numexports,numnatives,nummodules;
  char line[256],*instr,*params;
  int i,pass;
  symbol *sym;
  cell mainaddr;
  constval *mod;

  /* verify the opcode table */
  #if !defined NDEBUG
    pass=0;
    for (i=0; opcodes[i].name!=NULL; i++)
      if (opcodes[i].opcode!=0)
        assert(opcodes[i].opcode==++pass);
  #endif

  numexports=0;
  numnatives=0;
  mainaddr=0;
  /* count number of exported and native functions */
  for (sym=glbtab.next; sym!=NULL; sym=sym->next) {
    if (sym->ident==_functn) {
      if ((sym->usage & _external)!=0 && (sym->usage & _refer)!=0)
        numnatives++;
      if ((sym->usage & _exported)!=0 && (sym->usage & _define)!=0)
        numexports++;
      if (strcmp(sym->name,"main")==0) {
        assert(sym->vclass==_global);
        mainaddr=sym->addr;
      } /* if */
    } /* if */
  } /* for */
  assert(numnatives==ext_funcid);

  /* Relocate the natives table, modules with a zero usage count
   * are not written to the output file.
   */
  {
    int cur_id=0;
    int mod_id=0;
    for (mod=mod_tab.next; mod!=NULL; mod=mod->next) {
      if (mod->value > 0) {
        for (sym=glbtab.next; sym!=NULL; sym=sym->next)
          if (sym->ident==_functn && (sym->usage & _external)!=0 && sym->x.mod_id==cur_id)
            sym->x.mod_id=mod_id;
        mod_id++;
      } /* if */
      cur_id++;
    } /* for */
  }

  nummodules=0;
  for (mod=mod_tab.next; mod!=NULL; mod=mod->next)
    if (mod->value>0)
      nummodules++;

  /* write the abstract machine header */
  memset(&hdr, 0, sizeof hdr);
  hdr.magic=(short)0xF1E0;
  hdr.file_version=0;
  hdr.amx_version=0;
  hdr.flags=0;
  hdr.defsize=_namemax+1+sizeof(cell);
  hdr.num_exports=numexports;
  hdr.num_natives=numnatives;
  hdr.num_modules=nummodules;
  hdr.exports=sizeof hdr; /* export table starts right after the header */
  hdr.natives=hdr.exports + numexports*sizeof(FUNCSTUB);
  hdr.modules=hdr.natives + numnatives*sizeof(FUNCSTUB);
  hdr.cod=hdr.modules + nummodules*sizeof(FUNCSTUB);
  hdr.dat=hdr.cod+code_idx;
  hdr.hea=hdr.dat+glb_declared*sizeof(cell);
  hdr.stp=hdr.hea+stksize*sizeof(cell);
  hdr.cip=mainaddr;
  hdr.size=hdr.hea;
  #if defined BIG_ENDIAN
    align32(&hdr.size);
    align16(&hdr.magic);
    align16(&hdr.flags);;
    align16(&hdr.defsize);
    align16(&hdr.num_exports);
    align16(&hdr.num_natives);
    align16(&hdr.num_modules);
    align32(&hdr.exports);
    align32(&hdr.natives);
    align32(&hdr.modules);
    align32(&hdr.cod);
    align32(&hdr.dat);
    align32(&hdr.hea);
    align32(&hdr.stp);
    align32(&hdr.cip);
  #endif
  fwrite(&hdr,1,sizeof hdr,fout);

  /* write the export table */
  for (sym=glbtab.next; sym!=NULL; sym=sym->next) {
    if (sym->ident==_functn
        && (sym->usage & _exported)!=0 && (sym->usage & _define)!=0)
    {
      memset(&func, 0, sizeof func);
      strcpy(func.name,sym->name);
      assert(sym->vclass==_global);
      func.address=sym->addr;
      #if defined BIG_ENDIAN
        align32(&func.address);
      #endif
      fwrite(&func,1,sizeof func,fout);
    } /* if */
  } /* for */

  /* write the natives table */
  for (i=0; i<numnatives; i++) {
    for (sym=glbtab.next; sym!=NULL; sym=sym->next) {
      if (sym->ident==_functn && sym->addr==i
          && (sym->usage & _external)!=0 && (sym->usage & _refer)!=0)
      {
        memset(&func, 0, sizeof func);
        strcpy(func.name,sym->name);
        assert(sym->vclass==_global);
        func.address=sym->x.mod_id;
        #if defined BIG_ENDIAN
          align32(&func.address);
        #endif
        fwrite(&func,1,sizeof func,fout);
      } /* if */
    } /* for */
  } /* for */

  /* write the modules table */
  for (mod=mod_tab.next; mod!=NULL; mod=mod->next) {
    if (mod->value>0) {
      memset(&func, 0, sizeof func);
      assert(strlen(mod->name)>0 && strlen(mod->name)<=_namemax);
      strcpy(func.name,mod->name);
      func.address=0;   /* no need to swap this */
      fwrite(&func,1,sizeof func,fout);
    } /* if */
  } /* for */

  for (pass=_incseg; pass<=_indseg; pass++) {
    rewind(fin);
    while (fgets(line,256,fin)!=NULL) {
      strlwr(line);
      if ((instr=strchr(line,';'))!=NULL)
        *instr='\0';    /* erase comments */
      for (instr=line; isspace(*instr); instr++)
        /* nothing */;
      /* ignore empty lines and labels (labels have a special syntax, so these
       * must be parsed separately) */
      if (*instr=='\0' || *instr=='l' && *(instr+1)=='.')
        continue;
      /* get to the end of the instruction (make use of the '\n' that fgets()
       * added at the end of the line; this way we will *always* drop on a
       * whitespace character) */
      for (params=instr; !isspace(*params); params++)
        /* nothing */;
      assert(params>instr);
      for (i=0; opcodes[i].name!=NULL && strncmp(opcodes[i].name,instr,max((int)strlen(opcodes[i].name),(int)(params-instr)))!=0; i++)
        /* nothing */;
      assert(opcodes[i].name!=NULL);
      while (isspace(*params))
        params++;
      if (opcodes[i].segment==pass)
        opcodes[i].func(fout,params,opcodes[i].opcode);
    } /* while */
  } /* for */
}

