/*  Small compiler
 *
 *  Staging buffer and optimizer.
 *
 *  The staging buffer
 *  ------------------
 *  The staging buffer allows buffered output of generated code, deletion
 *  of redundant code, optimization by a tinkering process and reversing
 *  the ouput of evaluated expressions (which is used for the reversed
 *  evaluation of arguments in functions).
 *  Initially, stgwrite() writes to the file directly, but after a call to
 *  stgset(_yes), output is redirected to the buffer. After a call to
 *  stgset(_no), stgwrite()'s output is directed to the file again. Thus
 *  only one routine is used for writing to the output, which can be
 *  buffered output or direct output.
 *
 *  staging buffer variables:   stgbuf  - the buffer
 *                              stgidx  - current index in the staging buffer
 *                              staging - if true, write to the staging buffer;
 *                                        if false, write to file directly.
 */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>     /* for atoi() */
#include <string.h>
#include <ctype.h>
#include "sc.h"

static char stgbuf[_stgmax+1];
/* the variables "stgidx" and "staging" are declared in "sc.c" */

/*  stgmark
 *
 *  Copies a mark into the staging buffer. At this moment there are three
 *  possible marks:
 *     _startreverse    identifies the beginning of a series of expression
 *                      strings that must be written to the output file in
 *                      reversed order
 *    _endreverse       identifies the end of 'reverse evaluation'
 *    _exprstart        only valid within a block that is evaluated in
 *                      reversed oreder, it identifies the start of an
 *                      expression
 *
 *  Global references: stgidx  (altered)
 *                     stgbuf  (altered)
 *                     staging (referred to only)
 */
void stgmark(char mark)
{
  if (staging){
    if (stgidx<_stgmax){
      stgbuf[stgidx]=mark;
      stgidx+=1;
    } else {
      error(101,"staging buffer");       /* staging buffer overflow */
    } /* endif */
  } /* endif */
}

/*  stgwrite
 *
 *  Writes the string "st" to the staging buffer or to the output file. In the
 *  case of writing to the staging buffer, the terminating byte of zero is
 *  copied too, but... the optimizer can only work on complete lines (not on
 *  fractions of it. Therefore if the string is staged, if the last character
 *  written to the buffer is a '\0' and the previous-to-last is not a '\n',
 *  the string is concatenated to the last string in the buffer (the '\0' is
 *  overwritten). This also means an '\n' used in the middle of a string isn't
 *  recognized and could give wrong results with the optimizer.
 *  Even when writing to the output file directly, all strings are buffered
 *  until a whole line is complete.
 *
 *  Global references: stgidx  (altered)
 *                     stgbuf  (altered)
 *                     staging (referred to only)
 */
void stgwrite(char *st)
{
  int len;

  if (staging){
    if (stgidx>=2 && stgbuf[stgidx-1]=='\0' && stgbuf[stgidx-2]!='\n')
      stgidx-=1;                       /* overwrite last '\0' */
    while (*st && stgidx<_stgmax){     /* copy to staging buffer */
      stgbuf[stgidx]=*st;
      st+=1;
      stgidx+=1;
    } /* endwhile */
    if (*st){      /* check buffer overflow (is entire string written?) */
      error(101,"staging buffer");     /* staging buffer overflow */
    } else {                           /* terminate string */
      stgbuf[stgidx]='\0';
      stgidx+=1;
    } /* endif */
  } else {
    strcat(stgbuf,st);
    len=strlen(stgbuf);
    if (len>0 && stgbuf[len-1]=='\n'){
      writef(stgbuf);
      stgbuf[0]='\0';
    } /* endif */
  } /* endif */
}

/*  stgout
 *
 *  Writes the staging buffer to the output file via stgstring() (for
 *  reversing expressions in the buffer) and stgopt() (for optimizing). It
 *  resets "stgidx".
 *
 *  Global references: stgidx  (altered)
 *                     stgbuf  (referred to only)
 *                     staging (referred to only)
 */
void stgout(int index)
{
  if (!staging)
    return;
  stgstring(&stgbuf[index],&stgbuf[stgidx]);
  stgidx=index;
}

/*  stgstring
 *
 *  Analyses wether code strings should be output to the file as they appear
 *  in the staging buffer or wether portions of it should be output reversed.
 *  It calls itself recursively in the latter case and uses the parser stack
 *  to mark expressions that must be written reversed.
 *  In any case, stgstring() sends a block as large as possible to the
 *  optimizer stgopt().
 *
 *  In reversed mode, each set of code strings must start with the token
 *  _exprstart, even the first. If the token _startreverse is represented
 *  by '[', _endreverse by ']' and _exprstart by '|' the following applies:
 *     '[]...'     valid, but useless; nothing is output revesed
 *     '[|...]     valid, but useless; only one string set is output reversed
 *     '[|...|...] valid and usefull
 *     '[...|...]  invalid, first string doesn't start with '|'
 *     '[|...|]    invalid
 */
void stgstring(char *start,char *end)
{
  char *ptr,*lastmark;
  int nest,argc;

  while (start<end){
    if (*start==_startreverse){         /* start_reversed */
      start+=1;                         /* skip token */
      nest=1;      /* 'nest' is nesting counter */
      argc=0;      /* 'argc' is argument counter */
      do {
        switch (*start) {
          case _startreverse:
            nest+=1;
            start+=1;
            break;
          case _endreverse:
            nest-=1;
            start+=1;
            break;
          case _exprstart:
            start+=1;
            if (nest==1){
              pushstk((stkitem)start);
              argc+=1;
            } /* endif */
            break;
          default:
            start+=strlen(start)+1;
        } /* endswitch */
      } while (nest); /* enddo */
      lastmark=start;
      while (argc){
        ptr=start-1; /*'start' points behind '_endreverse'; 'ptr' points at it*/
        start=(char *)popstk();
        stgstring(start,ptr);
        argc-=1;
      } /* endwhile */
      start=lastmark;
    } else {
      ptr=start;
      while (*ptr!=_startreverse && ptr<end)
        ptr+=strlen(ptr)+1;
      stgopt(start,ptr);
      start=ptr;
    } /* endif */
  } /* endwhile */
}

/*  stgdel
 *
 *  Scraps code from the staging buffer by resetting "stgidx" to "index".
 *
 *  Global references: stgidx (altered)
 *                     staging (reffered to only)
 */
void stgdel(int index,cell code_index)
{
  if (staging) {
    stgidx=index;
    code_idx=code_index;
  } /* if */
}

int stgget(int *index,cell *code_index)
{
  if (staging) {
    *index=stgidx;
    *code_index=code_idx;
  } /* if */
  return staging;
}

/*  stgset
 *
 *  Sets staging on or off. If it's turned off, the staging buffer must be
 *  initialized to an empty string. If it's turned on, the routine makes sure
 *  the index ("stgidx") is set to 0 (it should already be 0).
 *
 *  Global references: staging  (altered)
 *                     stgidx   (altered)
 *                     stgbuf   (contents altered)
 */
void stgset(int onoff)
{
  staging=onoff;
  if (staging){
    assert(stgidx==0);
    stgidx=0;
    if (strlen(stgbuf)>0)    /* write any contents that may be put in the */
      writef(stgbuf);        /* buffer by stgwrite() when "staging" was 0 */
  } /* endif */
  stgbuf[0]='\0';
}

/*  stgopt
 *
 *  Optimizes the staging buffer by checking for series of instructions that
 *  can be coded more compact. The routine expects the lines in the staging
 *  buffer to be separated with a newline character.
 *
 *  The longest sequences must be checked first.
 */
typedef struct {
  char *find;
  char *replace;
  int savesize;         /* number of bytes saved (in bytecode) */
} SEQUENCE;
static SEQUENCE sequences[] = {
#if !defined NO_OPTIMIZE
  /* A very common sequence in four varieties
   *    load.s.pri n1           load.s.alt n1
   *    push.pri                load.s.pri n2
   *    load.s.pri n2           -
   *    pop.alt                 -
   *    --------------------------------------
   *    load.pri n1             load.alt n1
   *    push.pri                load.s.pri n2
   *    load.s.pri n2           -
   *    pop.alt                 -
   *    --------------------------------------
   *    load.s.pri n1           load.s.alt n1
   *    push.pri                load.pri n2
   *    load.pri n2             -
   *    pop.alt                 -
   *    --------------------------------------
   *    load.pri n1             load.alt n1
   *    push.pri                load.pri n2
   *    load.pri n2             -
   *    pop.alt                 -
   */
  { "load.s.pri %1!push.pri!load.s.pri %2!pop.alt!",
    "load.s.alt %1!load.s.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  { "load.pri %1!push.pri!load.s.pri %2!pop.alt!",
    "load.alt %1!load.s.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  { "load.s.pri %1!push.pri!load.pri %2!pop.alt!",
    "load.s.alt %1!load.pri %2!",
    2,  /* 8 - 6 */
  },
  { "load.pri %1!push.pri!load.pri %2!pop.alt!",
    "load.alt %1!load.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  /* The above also occurs with "addr.pri" (array indexing) as
   * the first line; so that adds 2 cases
   */
  { "addr.pri %1!push.pri!load.s.pri %2!pop.alt!",
    "addr.alt %1!load.s.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  { "addr.pri %1!push.pri!load.pri %2!pop.alt!",
    "addr.alt %1!load.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  /* And the same sequence with const.pri as either the first
   * or the second load instruction: four more cases.
   */
  { "const.pri %1!push.pri!load.s.pri %2!pop.alt!",
    "const.alt %1!load.s.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  { "const.pri %1!push.pri!load.pri %2!pop.alt!",
    "const.alt %1!load.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  { "load.s.pri %1!push.pri!const.pri %2!pop.alt!",
    "load.s.alt %1!const.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  { "load.pri %1!push.pri!const.pri %2!pop.alt!",
    "load.alt %1!const.pri %2!",
    (2*sizeof(cell)+4) - (2*sizeof(cell)+2)
  },
  /* ??? add references */
  /* Array indexing can merit from special instructions.
   * Simple indexed array lookup can be optimized quite
   * a bit.
   *    addr.pri n1             addr.alt n1
   *    push.pri                load.s.pri n2
   *    load.s.pri n2           bounds n3
   *    bounds n3               lidx
   *    shl.c.pri 2             -
   *    pop.alt                 -
   *    add                     -
   *    load.i                  -
   *
   * And to prepare for storing a value in an array
   *    addr.pri n1             addr.alt n1
   *    push.pri                load.s.pri n2
   *    load.s.pri n2           bounds n3
   *    bounds n3               idxaddr
   *    shl.c.pri 2             -
   *    pop.alt                 -
   *    add                     -
   *
   * Instruction addr.pri can also be const.pri (for global
   * arrays); the bounds instruction can be absent.
   *
   * If the array index is more complex, one can only optimize
   * the last four instructions:
   *    shl.c.pri 2             pop.alt
   *    pop.alt                 lidx
   *    add                     -
   *    loadi                   -
   *    --------------------------------------
   *    shl.c.pri 2             pop.alt
   *    pop.alt                 idxaddr
   *    add                     -
   */
  { "addr.pri %1!push.pri!load.s.pri %2!bounds %3!shl.c.pri 2!pop.alt!add!load.i!",
    "addr.alt %1!load.s.pri %2!bounds %3!lidx!",
    (4*sizeof(cell)+8) - (3*sizeof(cell)+4)
  },
  { "const.pri %1!push.pri!load.s.pri %2!bounds %3!shl.c.pri 2!pop.alt!add!load.i!",
    "const.alt %1!load.s.pri %2!bounds %3!lidx!",
    (4*sizeof(cell)+8) - (3*sizeof(cell)+4)
  },
  { "addr.pri %1!push.pri!load.s.pri %2!shl.c.pri 2!pop.alt!add!load.i!",
    "addr.alt %1!load.s.pri %2!lidx!",
    (3*sizeof(cell)+7) - (2*sizeof(cell)+3)
  },
  { "const.pri %1!push.pri!load.s.pri %2!shl.c.pri 2!pop.alt!add!load.i!",
    "const.alt %1!load.s.pri %2!lidx!",
    (3*sizeof(cell)+7) - (2*sizeof(cell)+3)
  },
  /* array index calculation for storing a value */
  { "addr.pri %1!push.pri!load.s.pri %2!bounds %3!shl.c.pri 2!pop.alt!add!",
    "addr.alt %1!load.s.pri %2!bounds %3!idxaddr!",
    (4*sizeof(cell)+7) - (3*sizeof(cell)+4)
  },
  { "const.pri %1!push.pri!load.s.pri %2!bounds %3!shl.c.pri 2!pop.alt!add!",
    "const.alt %1!load.s.pri %2!bounds %3!idxaddr!",
    (4*sizeof(cell)+7) - (3*sizeof(cell)+4)
  },
  { "addr.pri %1!push.pri!load.s.pri %2!shl.c.pri 2!pop.alt!add!",
    "addr.alt %1!load.s.pri %2!idxaddr!",
    (3*sizeof(cell)+6) - (2*sizeof(cell)+3)
  },
  { "const.pri %1!push.pri!load.s.pri %2!shl.c.pri 2!pop.alt!add!",
    "const.alt %1!load.s.pri %2!idxaddr!",
    (3*sizeof(cell)+6) - (2*sizeof(cell)+3)
  },
  /* the shorter array indexing sequences, see above for comments */
  { "shl.c.pri 2!pop.alt!add!loadi!",
    "pop.alt!lidx!",
    (sizeof(cell)+4) - 2
  },
  { "shl.c.pri 2!pop.alt!add!",
    "pop.alt!idxaddr!",
    (sizeof(cell)+3) - 2
  },
  /* During a calculation, the intermediate result must sometimes
   * be moved from PRI to ALT, like in:
   *    push.pri                move.alt
   *    load.s.pri n1           load.s.pri n1
   *    pop.alt                 -
   *
   * The above also accurs for "load.pri" and for "const.pri",
   * so add another two cases. Also note that "const.pri 0"
   * can be optimized (which adds another case).
   */
  { "push.pri!load.s.pri %1!pop.alt!",
    "move.alt!load.s.pri %1!",
    (sizeof(cell)+3) - (sizeof(cell)+2)
  },
  { "push.pri!load.pri %1!pop.alt!",
    "move.alt!load.pri %1!",
    (sizeof(cell)+3) - (sizeof(cell)+2)
  },
  { "push.pri!const.pri 0!pop.alt!",
    "move.alt!zero.pri!",
    (sizeof(cell)+3) - 2
  },
  { "push.pri!const.pri %1!pop.alt!",
    "move.alt!const.pri %1!",
    (sizeof(cell)+3) - (sizeof(cell)+2)
  },
  /* An even simpler PUSH/POP optimization (occurs in
   * switch statements):
   *    push.pri                move.alt
   *    pop.alt                 -
   */
  { "push.pri!pop.alt!",
    "move.alt!",
    2 - 1
  },
  /* Function calls (parameters are passed on the stack)
   *    load.s.pri n1           push.s n1
   *    push.pri                -
   *    --------------------------------------
   *    load.pri n1             push n1
   *    push.pri                -
   *    --------------------------------------
   *    const.pri n1            push.c n1
   *    push.pri                -
   *    --------------------------------------
   *    zero.pri                push.c 0        (this saves time, at
   *    push.pri                -               the cost of memory)
   */
  { "load.s.pri %1!push.pri!",
    "push.s %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "load.pri %1!push.pri!",
    "push %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "const.pri %1!push.pri!",
    "push.c %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "zero.pri!push.pri!",
    "push.c 0!",
    (2) - (sizeof(cell)+1)
  },
  /* References with a default value generate new cells on the heap
   * dynamically. That code often ends with:
   *    move.pri                push.alt
   *    push.pri                -
   */
  { "move.pri!push.pri!",
    "push.alt!",
    2 - 1
  },
  /* Simple arithmetic operations on constants. Noteworthy is the
   * subtraction of a constant, since it is converted to the addition
   * of the inverse value.
   *    const.alt n1            add.c n1
   *    add                     -
   *    --------------------------------------
   *    const.alt n1            add.c -n1
   *    sub                     -
   *    --------------------------------------
   *    const.alt n1            smul.c n1
   *    smul                    -
   *    --------------------------------------
   *    const.alt n1            eq.c n1
   *    eq                      -
   *    --------------------------------------
   */
  { "const.alt %1!add!",
    "add.c %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "const.alt %1!sub!",
    "add.c -%1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "const.alt %1!smul!",
    "smul.c %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "const.alt %1!eq!",
    "eq.c %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  /* Compare and jump
   *    eq                      jneq n
   *    jzer n                  -
   *    --------------------------------------
   *    eq                      jeq n
   *    jnz n                   -
   *    --------------------------------------
   *    neq                     jeq n
   *    jzer n                  -
   *    --------------------------------------
   *    neq                     jneq n
   *    jnz n                   -
   * Compares followed by jzer occur much more
   * often than compares followed with jnz. So we
   * take the easy route here.
   *    less                    jgeq n
   *    jzer n                  -
   *    --------------------------------------
   *    leq                     jgrtr n
   *    jzer n                  -
   *    --------------------------------------
   *    grtr                    jleq n
   *    jzer n                  -
   *    --------------------------------------
   *    geq                     jless n
   *    jzer n                  -
   *    --------------------------------------
   *    sless                   jsgeq n
   *    jzer n                  -
   *    --------------------------------------
   *    sleq                    jsgrtr n
   *    jzer n                  -
   *    --------------------------------------
   *    sgrtr                   jsleq n
   *    jzer n                  -
   *    --------------------------------------
   *    sgeq                    jsless n
   *    jzer n                  -
   */
  { "eq!jzer %1!",
    "jneq %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "eq!jnz %1!",
    "jeq %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "neq!jzer %1!",
    "jeq %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "neq!jnz %1!",
    "jneq %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "less!jzer %1!",
    "jgeq %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "leq!jzer %1!",
    "jgrtr %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "grtr!jzer %1!",
    "jleq %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "geq!jzer %1!",
    "jless %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "sless!jzer %1!",
    "jsgeq %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "sleq!jzer %1!",
    "jsgrtr %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "sgrtr!jzer %1!",
    "jsleq %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "sgeq!jzer %1!",
    "jsless %1!",
    (sizeof(cell)+2) - (sizeof(cell)+1)
  },
  /* Incrementing and decrementing leaves a value in
   * in PRI which may not be used (for example, as the
   * third expression in a "for" loop).
   *    inc n                   inc n   ; ++n
   *    load.pri n              -
   *    ;^;                     -
   *    --------------------------------------
   *    load.pri n              inc n   ; n++, e.g. "for (n=0; n<10; n++)"
   *    inc n                   -
   *    ;^;                     -
   * Plus the varieties for stack relative increments
   * and decrements.
   */
  { "inc %1!load.pri %1!;^;!",
    "inc %1!",
    2*(sizeof(cell)+1) - 1*(sizeof(cell)+1)
  },
  { "load.pri %1!inc %1!;^;!",
    "inc %1!",
    2*(sizeof(cell)+1) - 1*(sizeof(cell)+1)
  },
  { "inc.s %1!load.s.pri %1!;^;!",
    "inc.s %1!",
    2*(sizeof(cell)+1) - 1*(sizeof(cell)+1)
  },
  { "load.s.pri %1!inc.s %1!;^;!",
    "inc.s %1!",
    2*(sizeof(cell)+1) - 1*(sizeof(cell)+1)
  },
  { "dec %1!load.pri %1!;^;!",
    "dec %1!",
    2*(sizeof(cell)+1) - 1*(sizeof(cell)+1)
  },
  { "load.pri %1!dec %1!;^;!",
    "dec %1!",
    2*(sizeof(cell)+1) - 1*(sizeof(cell)+1)
  },
  { "dec.s %1!load.s.pri %1!;^;!",
    "dec.s %1!",
    2*(sizeof(cell)+1) - 1*(sizeof(cell)+1)
  },
  { "load.s.pri %1!dec.s %1!;^;!",
    "dec.s %1!",
    2*(sizeof(cell)+1) - 1*(sizeof(cell)+1)
  },
  /* ??? the same (increments and decrements) for references */
  /* Loading the constant zero has a special opcode.
   *    const.pri 0             zero n1
   *    stor.pri n1             -
   *    ;^;                     -
   *    --------------------------------------
   *    const.pri 0             zero.s n1
   *    stor.s.pri n1           -
   *    ;^;                     -
   *    --------------------------------------
   *    const.pri 0             zero.pri
   *    --------------------------------------
   *    const.alt 0             zero.alt
   * The last two alternatives save more memory than they save
   * time, but anyway...
   */
  { "const.pri 0!stor.pri %1!;^;!",
    "zero %1!",
    (2*sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "const.pri 0!stor.s.pri %1!;^;!",
    "zero.s %1!",
    (2*sizeof(cell)+2) - (sizeof(cell)+1)
  },
  { "const.pri 0!",
    "zero.pri!",
    (sizeof(cell)+1) - 1
  },
  { "const.alt 0!",
    "zero.alt!",
    (sizeof(cell)+1) - 1
  },
#endif
  /* ----- */
  { NULL, NULL, 0 }
};

#define _maxoptvars     3
#define _aliasmax       10      /* a 32-bit number can be represented in
                                 * 9 decimal digits */

static int matchsequence(char *start,char *end,char *pattern,
                         char symbols[_maxoptvars][_aliasmax+1],
                         int *num)
{
  int var,i;
  char str[_aliasmax+1];

  *num = 0;
  for (var=0; var<_maxoptvars; var++)
    symbols[var][0]='\0';

  while (*start=='\t' || *start==' ')
    start++;
  while (*pattern) {
    if (start>=end)
      return _no;
    switch (*pattern) {
    case '%':
      pattern++;
      assert(isdigit(*pattern));
      var=atoi(pattern) - 1;
      assert(var>=0 && var<_maxoptvars);
      assert(alphanum(*start));
      for (i=0; alphanum(*start); i++,start++) {
        assert(i<=_aliasmax);
        str[i]=*start;
      } /* for */
      str[i]='\0';
      if (symbols[var][0]!='\0') {
        if (strcmp(symbols[var],str)!=0)
          return _no;   /* symbols should be identical */
      } else {
        strcpy(symbols[var],str);
      } /* if */
      break;
    case ' ':
      if (*start!='\t' && *start!=' ')
        return _no;
      while (*start=='\t' || *start==' ')
        start++;
      break;
    case '!':
      while (*start=='\t' || *start==' ')
        start++;                /* skip trailing white space */
      if (*start!='\n')
        return _no;
      assert(*(start+1)=='\0');
      start+=2;                 /* skip '\n' and '\0' */
      while (*start=='\t' || *start==' ')
        start++;                /* skip leading white space */
      *num += 1;                /* one more matched line */
      break;
    default:
      if (tolower(*start) != tolower(*pattern))
        return _no;
      start++;
    } /* switch */
    pattern++;
  } /* while */

  return _yes;
}

static void replacesequence(char *pattern,char symbols[_maxoptvars][_aliasmax+1])
{
  char line[60],*lptr;
  int var;

  lptr=line;
  *lptr++='\t';         /* the "replace" patterns do not have tabs */
  while (*pattern) {
    assert((int)(lptr-line)<60);
    switch (*pattern) {
    case '%':
      /* write out what is accumulated for the line so far */
      *lptr='\0';
      writef(line);
      /* write out the symbol */
      pattern++;
      assert(isdigit(*pattern));
      var=atoi(pattern) - 1;
      assert(var>=0 && var<_maxoptvars);
      assert(symbols[var][0]!='\0');    /* variable should be defined */
      writef(symbols[var]);
      /* restart the line */
      lptr=line;
      break;
    case '!':
      /* write out what is accumulated for the line so far */
      *lptr++='\n';
      *lptr='\0';
      writef(line);
      lptr=line;
      *lptr++='\t';
      break;
    default:
      *lptr++=*pattern;
    } /* switch */
    pattern++;
  } /* while */
}

void stgopt(char *start,char *end)
{
  char symbols[_maxoptvars][_aliasmax+1];
  int num,seq;

  while (start<end){
    for (seq=0; sequences[seq].find!=NULL; seq++) {
      if (matchsequence(start,end,sequences[seq].find,symbols,&num)) {
        replacesequence(sequences[seq].replace,symbols);
        /* Skip the number if matched sequences, minus one. The
         * last sequence is skipped after the "for" loop.
         */
        while (num-- > 1) {
          start += strlen(start) + 1;
          assert(start < end);
        } /* while */
        code_idx-=sequences[seq].savesize;
        break;        /* break out of "for" loop, do not try to match more */
      } /* if */
    } /* for */
    if (sequences[seq].find==NULL)    /* if not matched */
      writef(start);          /* transform line if needed */
    assert(start < end);
    start += strlen(start) + 1; /* to next string */
  } /* while */
}

/*  writef
 *
 *  writes a string to the output file
 *
 *  Global references: outf   (referred to only)
 */
void writef(char *st)
{
  fputs(st,outf);
}

