// copyright (C) 2000 by David Cox All Rights Reserved

#include "parser.h"
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string>


Parser::Parser()
{
	m_leftStartAngle.Type(Token::e_startTag);
	m_rightStartAngle.Type(Token::e_closeTag);

	m_leftEndAngle.Type(Token::e_endTag);
	m_rightEndAngle.Type(Token::e_closeTag);

	m_equal.Type(Token::e_equal);
	m_root = 0;
}


Parser::~Parser()
{
	delete m_root;
}


void Parser::Translate(std::istream & f)
{
	m_f = &f;
	m_lookahead = m_lexan.NextToken(m_lookahead, *m_f);
	GetTag(m_root);
}


Tag * Parser::GetRoot()
{
	return m_root;
}


void Parser::GetTag(Tag * & t)
{
	t = new Tag;
	StartTag(t);			// match start tag
	Content(t);				// match content
	EndTag();				// match end tag
}


void Parser::TagName(std::string & tagName)
{
	if (m_lookahead.Type() == Token::e_name)
	{
		tagName = m_lookahead.String();		// save tag name
		Match(m_lookahead);					// go to next token
	}
	else
		Error();
}


void Parser::AttributeList(Tag * t)
{
	while (m_lookahead != m_rightStartAngle)
	{
		Attribute(t);			// match a bunch of attributes
	}
}


void Parser::Attribute(Tag * t)
{
	std::string name;
	name = m_lookahead.String();
	Match(m_lookahead);			// match attribute name

	Match(m_equal);				// equal sign

	std::string value;
	value = m_lookahead.String();
	Match(m_lookahead);			// match attribute value

	t->AddAttrib(name, value);	// save attribute name and value
}


void Parser::StartTag(Tag * t)
{
	std::string buffer;

	Match(m_leftStartAngle);	// match "<"

	TagName(buffer);			// match tag name
	if (t) t->Name(buffer);		// save tag name

	if (m_lookahead != m_rightStartAngle)
	{
		AttributeList(t);		// match attributes
	}

	Match(m_rightStartAngle);	// match ">"
}


void Parser::EndTag()
{
	std::string tagName;
	Match(m_leftEndAngle);			// match "</"
	TagName(tagName);				// match tag name
	Match(m_rightEndAngle);			// match ">"
}


void Parser::Content(Tag * parent)
{
	// if there is any content it can be either another start tag,
	// character data, or comment
	while (m_lookahead != m_leftEndAngle)
	{
		if (m_lookahead.Type() == Token::e_eof)
			Eof();
		
		else if (m_lookahead.Type() == Token::e_error)
			Error();

		else if (m_lookahead.Type() == Token::e_startTag)
		{
			// get next tag
			Tag * child;
			GetTag(child);
			parent->AddChild(child);
		}

		else if (m_lookahead.Type() == Token::e_comment)
		{
			// get comment
			std::string value;
			value += m_lookahead.String();
			parent->AddComment(value);
			Match(m_lookahead);
		}

		else
		{
			// get char data
			m_lexan.GetCharData(m_lookahead, *m_f);
			std::string value = m_lookahead.String();
			if (value != std::string(""))
				parent->AddContent(value);
			Match(m_lookahead);
		}
	}
}


void Parser::Match(Token & t)
{
	if (t == m_lookahead)
	{
		m_lookahead = m_lexan.NextToken(m_lookahead, *m_f);
	}
	else
		Error();
}


void Parser::Eof()
{
	std::cout << "unexpected end of file" << std::endl;
	exit(1);
}


void Parser::Error()
{
	std::cout << "syntax error line " << m_lexan.LineNumber() << " col: " << m_lexan.ColNumber() << std::endl;
	exit(1);
}


