unit EXPATParser;

//
// TSAXParser wrapper component for James Clark's EXPAT SAX Parser
// written by Danny Heijl, Jan 2001 (danny.heijl@pandora.be)
// for Dr. Dobbs JOURNAL.
//
// Meant to be totally compatible with my TSAXParser component
// for MSXML 3.0.
//
// Some SAX2 features of the MSXML SAX2 parser that I used in the MSXML
// implementation, and that are missing in EXPAT, are implemented here.
//
//
// You can not install both components at the same time, as they share
// the same name
//
// Make sure that "expat.dll" (I used version 1.95.1) is in your Windows System
// directory or in your Delphi\bin when installing the component (it is
// "statically" linked in).
//
// You can get the Expat WIN32 binaries at
// http://sourceforge.net/projects/expat/
//
//

interface

uses
  Windows, Messages, SysUtils, Classes, Expat, Dialogs;

type
  ESAXParser = class(Exception); // our own Exception

const
  NSSEP = '^';

// name space declarations are kept in a list of namespace
// objects so that the prefixes can be reported back to the
// application's element handlers
type
  TSXPPrefixMap = class
  public
    Prefix: string;
    URI: string;
  end;

type
  TSXPPrefixMapList = class(TList)
  public
    function GetItem(const URI: string): TSXPPrefixMap;
    procedure RemovePrefix(const prefix: string);
  end;

// attribute declarations are kept in a list of attribute declaration
// objects so that their type can be passed back to the application's
// element handlers
type
  TSXPAttrDecl = class
  public
   ElName: string;
   AttName: string;
   Att_type: string;
   ValueDefault: string;
   Dflt: string;
  end;

type
  TSXPAttrDeclList = class(TList)
    function GetItem(const ElName, AttName: string): TSXPAttrDecl;
  end;

// attributes are passed back as a list of TSXPAttribute
// objects to the application
type
 TSXPAttribute = class
  public
    URI: string;
    LocalName: string;
    QName: string;
    Value: string;
    AttType: string;
  end;

// add a GetItem function to the TList of TSXPAttributes,
// so that we can get at an attribute by name
type
  TSXPAttributeList = class(TList)
  public
    function GetItem(const AttrName: string): TSXPAttribute;
  end;

type
  TSAXParser = class;

// Expat "UserData" structure, set on parser Create,
// passed back in by all Expat event handler callbacks
// All I need is a backreference to the component that created
// the parser instance

  TUserData = record
    SAXParser:         TSAXParser;
  end;

// error handler event
  TOnError = procedure(line: integer;
                       column: integer;
                       const strMsg: string) of object;

// DTD handler events
  TOnnotationDecl = procedure(const Name: string;
                              const PublicID: string;
                              const SystemID: string) of object;
  TOnunparsedEntityDecl = procedure(const Name: string;
                                    const PublicID: string;
                                    const SystemID: string;
                                    const NotationName: string) of object;
// lexical handler events
  TOnStartDTD      = procedure(const Name: string;
                               const PublicID: string;
                               const SystemID: string) of object;
  TOnEndDTD        = procedure of object;
  TOnStartEntity   = procedure(const Name: string) of object;
  TOnEndEntity     = procedure(const name: string) of object;
  TOnStartCDATA    = procedure of object;
  TOnEndCDATA      = procedure of object;
  TOnComment       = procedure(const Chars: string) of object;
// declaration handler events
  TOnelementDecl = procedure(const Name: string;
                             const Model: string) of object;
  TOnattributeDecl = procedure(const ElementName: string;
                               const AttributeName: string;
                               const AttrType: string;
                               const ValueDefault: string;
                               const Value: string) of object;
  TOninternalEntityDecl = procedure(const Name: string;
                                    const Value: string) of object;
  TOnexternalEntityDecl = procedure(const Name: string;
                                    const PublicID: string;
                                    const SystemID: string) of object;
// contenthandler events
  TOnStartDocument = procedure of object;
  TOnEndDocument   = procedure of object;
  TOnStartElement  = procedure(const NamespaceURI: string;
                               const Localname: string;
                               const QName: string;
                               const Attributes: TSXPAttributeList) of object;
  TOnEndElement    = procedure(const NamespaceURI: string;
                               const Localname: string;
                               const QName: string) of object;
  TOnCharacters    = procedure(const Chars: string) of object;
  TOnSkippedEntity = procedure(const EntityName: string) of object;
  TOnStartPrefixMapping = procedure(const strPrefix: string;
                                    const strURI: string) of object;
  TOnEndPrefixMapping   = procedure(const strPrefix: string) of object;

// TSAXParser

  TSAXParser = class(TComponent)
  private
    // the Delphi memory allocator is used
    FMemorySuite:     TXML_Memory_Handling_Suite;
    // reference to the Expat SAX parser instance
    FParser:          TXML_Parser;
    // the parsed Element Declaration Content Tree
    FModel: PXML_Content;
    // error handler callbacks
    FOnError:         TOnError;
    FOnFatalError:    TOnError;
    FOnWarning:       TOnError;
    //
    FOnnotationDecl:       TOnnotationDecl;
    FOnunparsedEntityDecl: TOnunparsedEntityDecl;
    //
    FOnStartDTD:      TOnStartDTD;
    FOnEndDTD:        TOnEndDTD;
    FOnStartEntity:   TOnStartEntity;
    FOnEndEntity:     TOnEndEntity;
    FOnStartCDATA:    TOnStartCDATA;
    FOnEndCDATA:      TOnEndCDATA;
    FOnComment:       TOnComment;
    //
    FOnelementDecl:         TOnelementDecl;
    FOnattributeDecl:       TOnattributeDecl;
    FOninternalEntityDecl:  TOninternalEntityDecl;
    FOnexternalEntityDecl:  TOnexternalEntityDecl;
    //
    FOnStartDocument:  TOnStartDocument;
    FOnEndDocument:    TOnEndDocument;
    FOnStartElement:   TOnStartElement;
    FOnEndElement:     TOnEndElement;
    FOnCharacters:     TOnCharacters;
    FOnSkippedEntity:  TOnSkippedEntity;
    FOnStartPrefixMapping: TOnStartPrefixMapping;
    FOnEndPrefixMapping:   TOnEndPrefixMapping;
    // do or don't continuously upfate Line/Column info
    FUpdateLocation:   boolean;
    // a list of the attributes in the current element
    FAttributeList:    TSXPAttributeList;
    // a list with all the name space prefixes and URIs
    FNSPrefixList:     TSXPPrefixMapList;
    // a list of all attribute declarations
    FAttrDeclList:     TSXPAttrDeclList;
    // current location in the document
    FLine:             integer;
    FColumn:           integer;
    //
    FXMLSource:      string;
    FUserData:       TUserData;
    // supplied by the Contenthandler's ISAXLocator
    FPublicID:       string;
    FSystemId:       string;
    procedure GetLineColumn;
  public
    property ElementModel:     PXML_Content
                               read FModel
                               write FModel;
    property PublicID:         string  read FPublicID;
    property SystemID:         string  read FSystemID;
    property Line:             integer read FLine
                                       write FLine;
    property Column:           integer read FColumn
                                       write FColumn;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Parse;
    procedure GetLocation;
  published
    property XMLSource:              string
                                     read  FXMLSource
                                     write FXMLSource;
    property UpdateLocation:         boolean
                                     read  FUpdateLocation
                                     write FUpdateLocation;
    property OnnotationDecl:         TOnnotationDecl
                                     read  FOnnotationDecl
                                     write FOnnotationDecl;
    property OnunparsedEntityDecl:   TOnunparsedEntityDecl
                                     read  FOnunparsedEntityDecl
                                     write FOnunparsedEntityDecl;
    property OnelementDecl:          TOnelementDecl
                                     read  FOnelementDecl
                                     write FOnelementDecl;
    property OnattributeDecl:        TOnattributeDecl
                                     read  FOnattributeDecl
                                     write FOnattributeDecl;
    property OninternalEntityDecl:   TOninternalEntityDecl
                                     read  FOninternalEntityDecl
                                     write FOninternalEntityDecl;
    property OnexternalEntityDecl:   TOnexternalEntityDecl
                                     read  FOnexternalEntityDecl
                                     write FOnexternalEntityDecl;
    property OnStartDTD:             TOnStartDTD
                                     read  FOnStartDTD
                                     write FOnStartDTD;
    property OnEndDTD:               TOnEndDTD
                                     read  FOnEndDTD
                                     write FOnEndDTD;
    property OnStartEntity:          TOnStartEntity
                                     read  FOnStartEntity
                                     write FOnStartEntity;
    property OnEndEntity:            TOnEndEntity
                                     read  FOnEndEntity
                                     write FOnEndEntity;
    property OnStartCDATA:           TOnStartCdata
                                     read  FOnStartCDATA
                                     write FOnStartCDATA;
    property OnEndCDATA:             TOnEndCdata
                                     read  FOnEndCDATA
                                     write FOnEndCDATA;
    property OnComment:              TOnComment
                                     read  FOnComment
                                     write FOnComment;
    property OnStartDocument:        TOnStartDocument
                                     read  FOnStartDocument
                                     write FOnStartDocument;
    property OnEndDocument:          TOnEndDocument
                                     read  FOnEndDocument
                                     write FOnEndDocument;
    property OnStartElement:         TOnStartElement
                                     read  FOnStartElement
                                     write FOnStartElement;
    property OnEndElement:           TOnEndElement
                                     read  FOnEndElement
                                     write FOnEndElement;
    property OnCharacters:           TOnCharacters
                                     read  FonCharacters
                                     write FOnCharacters;
    // OnSkippedEntity is not (yet) implemented
    property OnSkippedEntity:        TOnSkippedEntity
                                     read  FOnSkippedEntity
                                     write FOnSkippedEntity;
    property OnStartPrefixMapping: TOnStartPrefixMapping
                                   read FOnStartPrefixMapping
                                   write FOnStartPrefixMapping;
    property OnEndPrefixMapping: TOnEndPrefixmapping
                                 read FOnEndPrefixMapping
                                 write FOnEndPrefixMapping;
    property OnError:                TOnError
                                     read  FOnError
                                     write FOnError;
    property OnFatalError:           TOnError
                                     read  FOnFatalError
                                     write FOnFatalError;
    property OnWarning:              TOnError
                                     read  FOnWarning
                                     write FOnWarning;
  end;

// EXPAT handler implementations
procedure ElementDeclhandler(userData : Pointer;
                             const elname: PXML_Char;
                             model: PXML_Content)
                             cdecl;

procedure AttlistDeclHandler(userData: pointer;
                             const elname: PXML_Char;
                             const attname: PXML_Char;
                             const att_type: PXML_Char;
                             const dflt: PXML_Char;
                             isrequired: integer) cdecl;

procedure XmlDeclHandler(userData: Pointer;
				   const version: TXML_Char;
                         const encoding: TXML_Char;
                         standalone: integer)
                         cdecl;

procedure StartElementHandler(userData: Pointer;
                              const elname: PXML_Char;
                              const atts: TAttrs)
                              cdecl;

procedure EndElementHandler(userData: Pointer;
                            const elname: PXML_Char)
                            cdecl;

{ s is not 0 terminated. }
procedure CharacterDataHandler(userData: Pointer;
					   const s: PXML_Char;
                               len: integer)
                               cdecl;

{ target and data are 0 terminated }
procedure ProcessingInstructionHandler(userData: Pointer;
						     const target: PXML_Char;
                                       const data: PXML_Char)
                                       cdecl;

{ data is 0 terminated }
procedure CommentHandler(userData: Pointer;
                         const data: PXML_Char)
                         cdecl;

procedure StartCdataSectionHandler(userData: Pointer)
                                   cdecl;

procedure EndCdataSectionHandler(userData: Pointer)
                                 cdecl;

procedure DefaultHandler(userData: Pointer;
                         const s: PXML_Char;
                         len: integer)
                         cdecl;

procedure StartDoctypeDeclHandler(userData: Pointer;
                  			const doctypeName: PXML_Char;
                                  const sysid: PXML_Char;
    					const pubid: PXML_Char;
    					has_internal_subset: integer)
                                  cdecl;

procedure EndDoctypeDeclHandler(userData: Pointer)
                                cdecl;

procedure EntityDeclHandler(userData: Pointer;
                            const entityName: PXML_Char;
				      is_parameter_entity: integer;
				      const value: PXML_Char;
				      value_length: integer;
				      const base: PXML_Char;
				      const sysID: PXML_Char;
				      const pubid: PXML_Char;
				      const notationName: PXML_Char)
                            cdecl;

//
// obsolete, handled by the EntityDeclHandler
//
procedure UnparsedEntityDeclHandler(
                                  userData: Pointer;
					      const entityName: PXML_Char;
					      const base: PXML_Char;
					      const sysID: PXML_Char;
					      const pubID: PXML_Char;
					      const notationName: PXML_Char)
                                  cdecl;


procedure NotationDeclHandler(
                            userData: Pointer;
					const notationName: PXML_Char;
                            const base: PXML_Char;
                            const sysID: PXML_Char;
                            const pubid: PXML_Char)
                            cdecl;

procedure StartNamespaceDeclHandler(
                                   userData: pointer;
					       const prefix: PXML_Char;
					       const uri: PXML_Char)
                                   cdecl;

procedure EndNamespaceDeclHandler(
                                 userData: Pointer;
					     const prefix: PXML_Char)
                                 cdecl;

function NotStandaloneHandler(userData: Pointer): integer
                                       cdecl;

function ExternalEntityRefHandler(
                                parser: TXML_Parser;
					    const context: PXML_Char;
					    const base: PXML_Char;
					    const sysID: PXML_Char;
					    const pubid: PXML_Char): integer
                                cdecl;

function UnknownEncodingHandler(
                              encodingHandlerData: Pointer;
					  const name: PXML_Char;
					  info: TXML_Encoding): integer
                              cdecl;

function  XML_Alloc(size: Cardinal): pointer; cdecl;
function  XML_ReAlloc(ptr: pointer; size: Cardinal): pointer; cdecl;
procedure XML_Free(ptr: pointer); cdecl;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('XML', [TSAXParser]);
end;

// map the Delphi memory allocator on the expat memory allocator functions
function XML_Alloc(size: Cardinal): pointer; cdecl;
var
  p: pointer;
begin
  try
    GetMem(p, size);
    Result := p;
  except
    Result := Nil;
  end;
end;

function XML_ReAlloc(ptr: pointer; size: Cardinal): pointer; cdecl;
var
  p: pointer;
begin
  p := ptr;
  try
    ReAllocMem(p, size);
    Result := p;
  except
    Result := Nil;
  end;
end;

procedure XML_Free(ptr: pointer); cdecl;
begin
  FreeMem(ptr);
end;

{ TSXPPrefixMapList }

function TSXPPrefixMapList.GetItem(const URI: string): TSXPPrefixMap;
var
  i: integer;
begin
  Result := nil;
  for i := 0 to Count - 1 do begin
    if CompareText(URI,
                   TSXPPrefixMap(Items[i]).URI) = 0 then begin
      Result := Items[i];
      exit;
    end;
  end;
end;

procedure TSXPPrefixMapList.RemovePrefix(const prefix: string);
var
  i: integer;
  PrefixMap: TSXPPrefixMap;
begin
  for i := 0 to Count - 1 do begin
    if CompareText(prefix,
                   TSXPPRefixMap(Items[i]).Prefix) = 0 then begin
      PrefixMap := items[i];
      PrefixMap.Free;
      Self.Delete(i);
      exit;
    end;
  end;
end;

{ TSXPAttrDeclList }

// get an attribute declaration by elementname/attributename
function TSXPAttrDeclList.GetItem(const ElName: string;
                                  const AttName: string): TSXPAttrDecl;
var
  i: integer;
  AttrDecl: TSXPAttrDecl;
begin
  Result := nil;
  for i := 0 to Count - 1 do begin
    AttrDecl := TSXPAttrDecl(Items[i]);
    if (CompareText(ElName, AttrDecl.Elname) = 0) and
       (CompareText(AttName, AttrDecl.AttName) = 0) then begin
      Result := Items[i];
      exit;
    end;
  end;
end;




{ TSXPAttributeList }

// add case insensitive access by attribute name to the attribute list
function TSXPAttributeList.GetItem(const AttrName: string): TSXPAttribute;
var
  i: integer;
begin
  Result := nil;
  for i := 0 to Count - 1 do begin
    if CompareText(AttrName,
                   TSXPAttribute(Items[i]).LocalName) = 0 then begin
      Result := Items[i];
      exit;
    end;
  end;
end;


{ TSAXPArser }


// get line and column
procedure TSaxParser.GetLineColumn;
begin
  if FUpdateLocation then begin
    FLine := XML_GetCurrentLineNumber(FParser);
    FColumn := XML_GetCurrentColumnNumber(FParser);
  end;
end;

constructor TSAXParser.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  // use the Delphi memory allocator because we have to free the
  // XML_Content tree returned by ElementDeclHandler
  with FMemorySuite do begin
    malloc_fcn  := @XML_Alloc;
    realloc_fcn := @XML_ReAlloc;
    free_fcn    := @XML_Free;
  end;
  FAttributeList := TSXPAttributeList.Create;
  FUpdateLocation := False;
end;

destructor TSAXParser.Destroy;
begin
  FreeAndNil(FAttributeList);
  inherited Destroy;
end;

// fire up the reader
procedure TSAXParser.Parse;
const
  XML_BUFSIZE = 8192;
var
  i: integer;
  PrefixMap: TSXPPrefixMap;
  AttrDEcl: TSXPAttrDecl;
  nread: integer;
  done: integer;
  Buf: PChar;
  F: File;
begin
  AssignFile(F, FXMLSource);
  try
    Reset(F, 1);
  except
    On E:Exception do begin
      CloseFile(F);
      Raise ESAXParser.Create(Format('TSAXParser.Parse(%s): %s',
                                     [FXMLSOURCE, E.Message]));
    end;
  end;
  // create the lists needed to implement MSMXML SAX parser compatibility
  FNSPrefixList  := TSXPPrefixMapList.Create;
  FAttrDeclList  := TSXPAttrDeclList.Create;
  // create the parser
  FParser        := XML_ParserCreate_MM(PChar('iso-8859-1'), @FMemorySuite, NSSEP);
  // and have a go at it
  try
    FLine := 0;
    FColumn := 0;
    FModel := Nil;
    // doesn't work for elements anyway, only for attributes
    // so this is implemented inside TSAXParser
    { XML_SetReturnNSTriplet(FParser, 1);}
    // the userdata for the parser callbacks contains a reference to the
    // TSAXParser component that created the parser
    FUserData.SAXParser := Self;
    XML_SetUserData(FParser,
                    @FUserData);
    // set the handler callbacks
    XML_SetElementHandler(FParser,
                          @StartElementHandler,
                          @EndElementHandler);
    XML_SetCharacterDataHandler(FParser,
                                @CharacterDataHandler);
    XML_SetCommentHandler(FParser,
                          @CommentHandler);
    XML_SetNameSpaceDeclHandler(FParser,
                                @StartNamespaceDeclhandler,
                                @EndNamespaceDeclhandler);
    XML_SetElementDeclHandler(FParser,
                              @ElementDeclhandler);
    XML_SetCdataSectionHandler(FParser,
                               @StartCdataSectionHandler,
                               @EndCdataSectionHandler);
    XML_SetDocTypeDeclHandler(FParser,
                               @StartDocTypeDeclHandler,
                               @EndDocTypeDeclHandler);
    XML_SetAttListDeclHandler(FParser,
                              @AttListDeclhandler);
    XML_SetEntityDeclHandler(FParser, @EntityDeclHandler);
    XML_SetNotationDeclHandler(FParser, @NotationDeclHandler);

    if Assigned(FOnStartDocument) then FOnStartDocument;
    // read the file and pass it on to the parser for eh... parsing
    done := 0;
    while done = 0 do begin
       Buf := XML_GetBuffer(FParser, XML_BUFSIZE);
       if (Buf = Nil) then begin
         Raise ESAXParser.Create('TSAXParser.Parse: XML_GetBuffer() fails');
       end;
       BlockRead(F, Buf^, XML_BUFSIZE, nread);
       if nread < XML_BUFSIZE then done := 1 else done := 0;
       if (XML_ParseBuffer(FParser, nread, done) = 0) then begin
         GetLocation;
         CloseFile(F);
         Raise ESAXParser.Create(
               Format('XML Parse error %d (%s) at line %d in %s',
                      [Ord(XML_GetErrorCode(FParser)),
                      string(XML_ErrorString(Ord(XML_GetErrorCode(FParser)))),
                      FLine,
                      FXMLSource]) );
         exit;
       end;
    end;
    if Assigned(FOnEndDocument) then FOnEndDocument;
    CloseFile(F);
  finally
    XML_ParserFree(FParser);
    FParser := Nil;
    for i := FNSPrefixList.Count -1 downto 0 do begin
      PrefixMap := FNSPrefixList.Items[i];
      Prefixmap.Free;
      FNSPrefixList.Delete(i);
    end;
    FreeAndNil(FNSPrefixList);
    for i := FAttrDeclList.Count -1 downto 0  do begin
      AttrDecl := FAttrDEclList.Items[i];
      AttrDecl.Free;
      FAttrDeclList.Delete(i);
    end;
    FreeAndNil(FAttrDeclList);
  end;
end;

// if UpdateLocation is off, you can still ask for
// a one-shot update of the current Line and Column properties
procedure TSAXParser.GetLocation;
begin
  if FUpdateLocation = False then begin
    FUpdateLocation := True;
    GetLineColumn;
    FUpdateLocation := False;
  end;
end;


// EXPAT handler implementations

procedure ElementDeclhandler(userData : Pointer;
                             const elname: PXML_Char;
                             model: PXML_Content); cdecl;
  //
  // the expat parser returns element declarations as a tree
  // of content nodes, the MSXML parser gives a "normalized" string
  // Here I "rebuild" a MSXML Model string from the EXPAT content tree
  // The TSAXParser.Model property gets a pointer to the Expat Content
  // tree in case the application prefers to use the tree rather than parse
  // the string again
  //
  function expand_node(const ContentNode: PXML_Content): string;
  var
    i: Cardinal;
    Child: PXML_Content;
  begin
    Result := '';
    case ContentNode._type of
      XML_CTYPE_EMPTY:
        Result := 'EMPTY';
      XML_CTYPE_ANY:
        Result := 'ANY';
      XML_CTYPE_NAME:
        Result := string(ContentNode.name);
      XML_CTYPE_CHOICE, XML_CTYPE_SEQ: begin
          Result := Result + '(';
          if ContentNode.numchildren > 0 then begin
            Child := ContentNode.children;
            for i := 0 to ContentNode.numchildren - 1 do begin
              Result := Result + expand_node(Child);
              if i < ContentNode.numchildren - 1 then begin
                if ContentNode._type = XML_CTYPE_SEQ then
                  Result := Result+ ','
                else
                  Result := Result + '|';
              end;
              Inc(Child);
            end;
          end;
          Result := Result + ')';
        end;
      XML_CTYPE_MIXED: begin
          Result := Result + '(#PCDATA';
          if ContentNode.numchildren > 0 then begin
            Child := ContentNode.children;
            for i := 0 to ContentNode.numchildren - 1 do begin
              Result := Result + '|' + expand_node(Child);
             Inc(Child);
            end;
          end;
          Result := Result + ')';
        end;
    end;
    case ContentNode.quant of
      XML_CQUANT_OPT  : Result := Result + '?';
      XML_CQUANT_REP  : Result := Result + '*';
      XML_CQUANT_PLUS : Result := Result + '+';
    end;
  end;

var
  XP: TSAXParser;
  strName, strModel: string;
begin
  XP := TUserdata(UserData^).SAXParser;
  try
    with XP do begin
      if Assigned(FOnelementDecl) then begin
        GetLineColumn;
        strName := elname;
        // build an MSXML-like model string
        strModel := expand_node(model);
        // but give the application access to the parsed model tree anyway
        FModel := model;
        // call the element declaration handler event
        FOnelementDecl(strname, strModel);
        // and free the model tree allocated by the parser
        XML_Free(model);
        FModel := Nil;
      end;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnelementDecl:' +  E.Message);
  end;
end;


procedure AttlistDeclHandler(userData: pointer;
                             const elname: PXML_Char;
                             const attname: PXML_Char;
                             const att_type: PXML_Char;
                             const dflt: PXML_Char;
                             isrequired: integer); cdecl;
var
  XP: TSAXParser;
  strValueDefault: string;
  AttrDecl: TSXPAttrDecl;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      GetLineColumn;
      AttrDecl := TSXPAttrDecl.Create;
      // compute ValueDefault value based on combination
      // of isrequired and dflt
      if isrequired <> 0 then begin
        if dflt = Nil then
          strValueDefault := '#REQUIRED'
        else
          strValueDefault := '#FIXED';
      end else begin
         if dflt = Nil then
           strValueDefault := '#IMPLIED'
         else
           strValueDefault := '';
      end;
      AttrDecl.ElName      := string(elname);
      AttrDecl.AttName     := string(attname);
      AttrDecl.Att_Type     := string(att_type);
      AttrDecl.ValueDefault:= strValueDefault;
      AttrDecl.Dflt        := string(dflt);
      FAttrDeclList.Add(AttrDecl);
      if Assigned(FonAttributeDecl) then begin
        FOnAttributeDecl(AttrDecl.ElName,
                         AttrDecl.AttName,
                         AttrDecl.Att_Type,
                         strValueDefault,
                         AttrDecl.Dflt);
      end;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnAttributeDecl:' +  E.Message);
  end;
end;

// not implemented because not in MSXML
procedure XmlDeclHandler(userData: Pointer;
				   const version: TXML_Char;
                         const encoding: TXML_Char;
                         standalone: integer); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      GetLineColumn;
      // not present in MSXML
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnXMLDecl:' +  E.Message);
  end;
end;

procedure StartElementHandler(userData: Pointer;
                              const elname: PXML_Char;
                              const atts: TAttrs); cdecl;
var
  XP:          TSAXParser;
  Attribute:   TSXPAttribute;
  PrefixMap:   TSXPPrefixMap;
  AttrDecl:    TSXPAttrDecl;
  nAttributes: integer;
  i:           integer;
  iPosSep:     integer;
  strURI, strLocalName: string;
  strName, strAttname:  string;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      if Assigned(FOnstartElement) then begin
        GetLineColumn;
        try
          if atts <> Nil then begin
            nAttributes := XML_GetSpecifiedAttributeCount(XP.FParser) shr 1;
            FAttributeList.Capacity := nAttributes;
            for i := 0 to nAttributes - 1 do begin
              Attribute  := TSXPAttribute.Create;
              strAttName := atts[i * 2];
              iPosSep    := Pos(NSSEP, strAttName);
              with Attribute do begin
                if iPosSep <> 0 then begin
                  // if namespace: split into URI and name parts
                  URI := Copy(strAttName, 1, iPosSep - 1);
                  Localname := Copy(strAttname, iPosSep + 1, Length(strAttName) -iPosSep);
                  // translate URI to prefix
                  PrefixMap := FNSPrefixList.GetItem(URI);
                  if Assigned(Prefixmap) then begin
                    strAttName := Format('%s:%s', [PrefixMap.Prefix, LocalName]);
                  end;
                end else begin
                  URI := '';
                  LocalName := strAttName;
                end;
                QName     := strAttName;
                Value     := atts[(i * 2)+1];
                AttrDecl  := FAttrDeclList.GetItem(string(elname), LocalName);
                if Assigned(AttrDecl) then begin
                  AttType := AttrDecl.Att_type;
                end else begin
                  AttType   := 'CDATA'
                end;
              end;
              FAttributeList.Add(Attribute);
            end; // for
          end; // if atts <> Nil
          // now call the event handler
          strName    := elname;
          iPosSep    := Pos(NSSEP, strName);
          if iPosSep <> 0 then begin
            // if namespace: split into URI and name part
            strURI := Copy(strName, 1, iPosSep - 1);
            strLocalName := Copy(strName, IPosSep + 1, Length(strName) - iPosSep);
            // translate URI to prefix
            PrefixMap := FNSPrefixList.GetItem(strURI);
            if Assigned(Prefixmap) then begin
              strname := Format('%s:%s', [PrefixMap.Prefix, strLocalName]);
            end;
          end else begin
            strURI := '';
            strLocalName := strName;
          end;
          FOnStartElement(strURI,
                          strLocalName,
                          strName,
                          FAttributeList);
        finally
          // clean up the attributes list
          for i := 0 to FAttributeList.Count - 1 do begin
            TSXPAttribute(FAttributeList.Items[i]).Free;
          end;
          FAttributeList.Clear;
        end;
      end // if Assigned(FOnstartElement)
    end; //with
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnStartElement:' +  E.Message);
  end; // try
end;

procedure EndElementHandler(userData: Pointer;
                            const elname: PXML_Char); cdecl;
var
  XP: TSAXParser;
  i: integer;
  strName, strURI, strLocalName: string;
  PrefixMap: TSXPPrefixMap;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      if Assigned(FOnendElement) then begin
        GetLineColumn;
        strName := elname;
        i := Pos(NSSEP, strName);
        if i > 0 then begin
          strURI := Copy(strName, 1, i-1);
          strLocalName := Copy(strName, i + 1, Length(strName) - i);
          PrefixMap := FNSPrefixList.GetItem(strURI);
          if Assigned(Prefixmap) then begin
            strname := Format('%s:%s', [PrefixMap.Prefix, strLocalName]);
          end;
        end else begin
          strURI := '';
          strLocalName := strName;
        end;
        FOnEndElement(strURI, strLocalName, strName);
      end; // if Assigned
    end; // with XP
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnEndElement:' +  E.Message);
  end; // try
end;

{ s is not 0 terminated. }
procedure CharacterDataHandler(userData: Pointer;
					   const s: PXML_Char;
                               len: integer); cdecl;
var
  XP: TSAXParser;
  str: string;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      if Assigned(FOnCharacters) then begin
          GetLineColumn;
          SetString(str, s, len);
          FOnCharacters(str);
      end; // if Assigned
    end; // with XP
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnCharacters:' +  E.Message);
  end; // try
end;

{ target and data are 0 terminated }
// not yet implemented
procedure ProcessingInstructionHandler(userData: Pointer;
						     const target: PXML_Char;
                                       const data: PXML_Char); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  //if Assigned(XP.FOnprocessingInstruction) then begin
  //
  //end;
end;

{ data is 0 terminated }
procedure CommentHandler(userData: Pointer;
                         const data: PXML_Char); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    if Assigned(XP.FOnComment) then begin
      XP.GetLineColumn;
      XP.OnComment(string(data));
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnComment:' +  E.Message);
  end;
end;

procedure StartCdataSectionHandler(userData: Pointer); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    if Assigned(XP.FOnStartCdata) then begin
      XP.GetLineColumn;
      XP.FOnStartCdata;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnStartCData:' +  E.Message);
  end;
end;

procedure EndCdataSectionHandler(userData: Pointer); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    if Assigned(XP.FOnEndCdata) then begin
      XP.GetLineColumn;
      XP.FOnEndCdata;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnEndCData:' +  E.Message);
  end;
end;

// not implemented
procedure DefaultHandler(userData: Pointer;
                         const s: PXML_Char;
                         len: integer); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  //if Assigned(XP.FOndefault) then begin
  //
  //end;
end;

procedure StartDoctypeDeclHandler(userData: Pointer;
                  			const doctypeName: PXML_Char;
                                  const sysid: PXML_Char;
    					const pubid: PXML_Char;
    					has_internal_subset: integer); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      if Assigned(FOnStartDTD) then begin
        GetLineColumn;
        FPublicId := string(pubid);
        FSystemID := string(sysid);
        FOnStartDTD(string(doctypeName), FPublicID, FSystemID);
      end;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnStartDTD:' +  E.Message);
  end;
end;

procedure EndDoctypeDeclHandler(userData: Pointer); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    if Assigned(XP.FOnEndDTD) then begin
      XP.GetLineColumn;
      XP.FOnEndDTD;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnEndDTD:' +  E.Message);
  end;
end;

// combines msxml externalentitydeclhandler,
// internalentitydeclhandler and unparsedentitydeclhandler
procedure EntityDeclHandler(userData: Pointer;
                            const entityName: PXML_Char;
				      is_parameter_entity: integer;
				      const value: PXML_Char;
				      value_length: integer;
				      const base: PXML_Char;
				      const sysID: PXML_Char;
				      const pubid: PXML_Char;
				      const notationName: PXML_Char); cdecl;
var
  XP: TSAXParser;
  strValue: string;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      GetLineColumn;
      // for unparsed entities notationName is not NULL
      if notationName <> Nil then begin
        if Assigned(FOnUnparsedEntityDecl) then begin
          GetLineColumn;
          FOnUnparsedEntityDecl(string(entityname),
                                  string(pubid),
                                  string(sysID),
                                  string(notationName));
        end;
      end else begin
        // for external entities value is NULL
        if value = Nil then begin
          if Assigned(FOnExternalEntityDecl) then begin
            GetLineColumn;
            FOnExternalEntityDecl(string(entityname),
                                  string(pubid),
                                  string(sysID));
          end;
        end else begin
          // for internal entities value is not NULL
          if Assigned(FOnInternalEntityDecl) then begin
            GetLineColumn;
            SetString(strValue, value, value_length);
            FOnInternalEntityDecl(string(entityname), strValue);
          end;
        end; // if value = Nil
      end; // if notationName <> Nil
    end; // with XP
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnEntityDecl:' +  E.Message);
  end;
end;

//
// this is OBSOLETE and handled by the EntityDeclhandler
//
procedure UnparsedEntityDeclHandler(
                                  userData: Pointer;
					      const entityName: PXML_Char;
					      const base: PXML_Char;
					      const sysID: PXML_Char;
					      const pubID: PXML_Char;
					      const notationName: PXML_Char)
                                  cdecl;
var
  XP: TSAXParser;
begin
    XP := TUserData(UserData^).SAXParser;
//  try
//    with XP do begin
//      if Assigned(FOnUnparsedEntityDecl) then begin
//        GetLineColumn;
//        FOnUnparsedEntityDecl(string(entityname),
//                              string(pubid),
//                              string(sysID),
//                              string(notationName));
//      end;
//    end;
//  except
//    On E:Exception do
//      Raise ESAXParser.Create('TSAXParser.UnparsedEntityDecl:' +  E.Message);
//  end;
end;


procedure NotationDeclHandler(
                            userData: Pointer;
					const notationName: PXML_Char;
                            const base: PXML_Char;
                            const sysID: PXML_Char;
                            const pubid: PXML_Char); cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      if Assigned(XP.FOnNotationDecl) then begin
        GetLineColumn;
        FOnNotationDecl(string(notationName),
                              string(pubid),
                              string(sysID));
      end;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnNotationDecl:' +  E.Message);
  end;
end;

// expat namespacedecl = msxml prefixmapping
procedure StartNamespaceDeclHandler(
                                   userData: pointer;
					       const prefix: PXML_Char;
					       const uri: PXML_Char); cdecl;
var
  XP: TSAXParser;
  PrefixMap: TSXPPrefixMap;
begin
  XP := TUserData(UserData^).SAXParser;
  try
    with XP do begin
      // we have to keep the prefix for later use
      PrefixMap := TSXPPRefixMap.Create;
      PrefixMap.URI := uri;
      PrefixMap.Prefix := prefix;
      FNSPrefixList.Add(PrefixMap);
      if Assigned(FOnstartPrefixMapping) then begin
        GetLineColumn;
        FOnStartPrefixMapping(prefix, uri);
      end;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnStartPrefixMapping:' +  E.Message);
  end;
end;

// expat namespacedecl = msxml prefixmapping
procedure EndNamespaceDeclHandler(
                                 userData: Pointer;
					     const prefix: PXML_Char); cdecl;
var
  XP: TSAXParser;
begin
  try
    XP := TUserData(UserData^).SAXParser;
    with XP do begin
      FNSPrefixList.RemovePrefix(prefix);
      if Assigned(XP.FOnEndPrefixMapping) then begin
        GetLineColumn;
        XP.FOnEndPrefixMapping(prefix);
      end;
    end;
  except
    On E:Exception do
      Raise ESAXParser.Create('TSAXParser.OnEndPrefixMapping:' +  E.Message);
  end;
end;

// not implemented
function NotStandaloneHandler(userData: Pointer): integer; cdecl;
var
  XP: TSAXParser;
begin
  XP := TUserData(UserData^).SAXParser;
  //if Assigned(XP.FOnnotStandalone) then begin
  //
  //end;
  Result := 0;
end;

// this is a special case that is not used here
// external entities are handled by EntityDeclhandler
function ExternalEntityRefHandler(
                                parser: TXML_Parser;
					    const context: PXML_Char;
					    const base: PXML_Char;
					    const sysID: PXML_Char;
					    const pubid: PXML_Char): integer; cdecl;
begin
  //if Assigned(XP.FOnexternalEntityRef) then begin
  //
  //end;
  Result := 0;
end;

// not implemented
function UnknownEncodingHandler(
                              encodingHandlerData: Pointer;
					  const name: PXML_Char;
					  info: TXML_Encoding): integer; cdecl;
begin
  Result := 0;
end;

end.
