unit MSSAXParser;

// wrapper component for the SAX2 parser in the MSXML 3.0
// written by Danny Heijl, Jan 2001 (danny.heijl@pandora.be)
// for Dr. Dobbs JOURNAL

interface

uses
  Windows, Messages, SysUtils, Classes,
  ComObj, Oleserver, ActiveX, MSXML2_TLB;

type
  ESAXParser = class(Exception); // our own Exception

  TSAXParser = class;

// define he error handler callback procedure
  TOnError = procedure(line: integer;
                       column: integer;
                       const strMsg: string) of object;

// TSAXErrorHandler implements the ISAXErrorHandler interface
 TSAXErrorHandler = class(TInterfacedObject, ISAXErrorHandler)
  private
    // TSAXParser event handler pointers
    FOnError:         TOnError;
    FOnFatalError:    TOnError;
    FOnWarning:       TOnError;
  public
    function  error(const pLocator: ISAXLocator;
                    const pwchErrorMessage: pWideChar;
                    hrErrorCode: HResult): HResult; stdcall;
    function  fatalError(const pLocator: ISAXLocator;
                         const pwchErrorMessage: pWideChar;
                         hrErrorCode: HResult): HResult; stdcall;
    function  ignorableWarning(const pLocator: ISAXLocator;
                               const pwchErrorMessage: pWideChar;
                               hrErrorCode: HResult): HResult; stdcall;
    property  OnError:       TOnError
                             read FOnError
                             write FOnError;
    property  OnFatalError:  TOnError
                             read FOnFatalError
                             write FOnFatalError;
    property  OnWarning:     TOnError
                             read FOnWarning
                             write FOnWarning;
  end;

// define the DTD handler event procedures
  TOnnotationDecl = procedure(const Name: string;
                              const PublicID: string;
                              const SystemID: string) of object;
  TOnunparsedEntityDecl = procedure(const Name: string;
                                    const PublicID: string;
                                    const SystemID: string;
                                    const NotationName: string) of object;

// TSAXDTDHandler implements the ISAXDTDHandler interface
  TSAXDTDHandler = class(TInterfacedObject, ISAXDTDHandler)
  private
    FSAXParser:            TSAXParser;
    // TSAXParser event handler pointers
    FOnnotationDecl:       TOnnotationDecl;
    FOnunparsedEntityDecl: TOnunparsedEntityDecl;
  public
    function  notationDecl(const pwchName: pWideChar;
                           cchName: SYSINT;
                           const pwchPublicId: pWideChar;
                           cchPublicId: SYSINT;
                           const pwchSystemId: pWideChar;
                           cchSystemId: SYSINT): HResult; stdcall;
    function  unparsedEntityDecl(const pwchName: pWideChar;
                                 cchName: SYSINT;
                                 const pwchPublicId: pWideChar;
                                 cchPublicId: SYSINT;
                                 const pwchSystemId: pWideChar;
                                 cchSystemId: SYSINT;
                                 const pwchNotationName: pWideChar;
                                 cchNotationName: SYSINT): HResult; stdcall;
    property  OnnotationDecl : TOnnotationDecl
                               read FOnnotationDecl
                               write FOnnotationDecl;
    property  OnunparsedEntityDecl : TOnunparsedEntityDecl
                                     read FOnunparsedEntityDecl
                                     write FOnunparsedEntityDecl;
    property  SAXParser: TSAXParser read FSAXParser write FSAXParser;
  end;


// define the lexical handler callback procedures
  TOnStartDTD      = procedure(const Name: string;
                               const PublicID: string;
                               const SystemID: string) of object;
  TOnEndDTD        = procedure of object;
  TOnStartEntity   = procedure(const Name: string) of object;
  TOnEndEntity     = procedure(const name: string) of object;
  TOnStartCDATA    = procedure of object;
  TOnEndCDATA      = procedure of object;
  TOnComment       = procedure(const Chars: string) of object;

// TSAXLexicalHandler implements the ISAXLexicalHandler interface
 TSAXLexicalHandler = class(TInterfacedObject, ISAXLexicalHandler)
  private
    FSAXParser:       TSAXParser;
    // TSAXParser event handler pointers
    FOnStartDTD:      TOnStartDTD;
    FOnEndDTD:        TOnEndDTD;
    FOnStartEntity:   TOnStartEntity;
    FOnEndEntity:     TOnEndEntity;
    FOnStartCDATA:    TOnStartCDATA;
    FOnEndCDATA:      TOnEndCDATA;
    FOnComment:       TOnComment;
  public
    function  startDTD(const pwchName: pWideChar;
                       cchName: SYSINT;
                       const pwchPublicId: pWideChar;
                       cchPublicId: SYSINT;
                       const pwchSystemId: pWideChar;
                       cchSystemId: SYSINT): HResult; stdcall;
    function  endDTD: HResult; stdcall;
    function  startEntity(const pwchName: pWideChar;
                          cchName: SYSINT): HResult; stdcall;
    function  endEntity(const pwchName: pWideChar;
                        cchName: SYSINT): HResult; stdcall;
    function  startCDATA: HResult; stdcall;
    function  endCDATA: HResult; stdcall;
    function  comment(const pwchChars: pWideChar;
                      cchChars: SYSINT): HResult; stdcall;
    property  OnStartDTD:    TOnStartDTD
                             read FOnStartDTD
                             write FOnStartDTD;
    property  OnEndDTD:      TOnEndDTD
                             read FOnEndDTD
                             write FOnEndDTD;
    property  OnStartEntity: TOnStartEntity
                             read FOnStartEntity
                             write FOnStartEntity;
    property  OnEndEntity:   TOnEndEntity
                             read FOnEndEntity
                             write FOnEndEntity;
    property  OnStartCDATA:  TOnStartCdata
                             read FOnStartCDATA
                             write FOnStartCDATA;
    property  OnEndCDATA:    TOnEndCdata
                             read FOnEndCDATA
                             write FOnEndCDATA;
    property  OnComment:     TOnComment
                             read FOnComment
                             write FOnComment;
    property  SAXParser: TSAXParser read FSAXParser write FSAXParser;
  end;

// define the declaration handler callbacks
  TOnelementDecl = procedure(const Name: string;
                             const Model: string) of object;
  TOnattributeDecl = procedure(const ElementName: string;
                               const AttributeName: string;
                               const AttrType: string;
                               const ValueDefault: string;
                               const Value: string) of object;
  TOninternalEntityDecl = procedure(const Name: string;
                                    const Value: string) of object;
  TOnexternalEntityDecl = procedure(const Name: string;
                                    const PublicID: string;
                                    const SystemID: string) of object;

// TSAXDeclhandler implements the ISAXDeclHandler Interface
 TSAXDeclHandler = class(TInterfacedObject, ISAXDeclHandler)
  private
    FSAXParser:             TSAXParser;
    // TSAXParser event handler pointers
    FOnelementDecl:         TOnelementDecl;
    FOnattributeDecl:       TOnattributeDecl;
    FOninternalEntityDecl:  TOninternalEntityDecl;
    FOnexternalEntityDecl:  TOnexternalEntityDecl;
  public
    function  elementDecl(const pwchName: pWideChar;
                          cchName: SYSINT;
                          const pwchModel: pWideChar;
                          cchModel: SYSINT): HResult; stdcall;
    function  attributeDecl(const pwchElementName: pWideChar;
                            cchElementName: SYSINT;
                            const pwchAttributeName: pWideChar;
                            cchAttributeName: SYSINT;
                            const pwchType: pWideChar;
                            cchType: SYSINT;
                            const pwchValueDefault: pWideChar;
                            cchValueDefault: SYSINT;
                            const pwchValue: pWideChar;
                            cchValue: SYSINT): HResult; stdcall;
    function  internalEntityDecl(const pwchName: pWideChar;
                                 cchName: SYSINT;
                                 const pwchValue: pWideChar;
                                 cchValue: SYSINT): HResult; stdcall;
    function  externalEntityDecl(const pwchName: pWideChar;
                                 cchName: SYSINT;
                                 const pwchPublicId: pWideChar;
                                 cchPublicId: SYSINT;
                                 const pwchSystemId: pWideChar;
                                 cchSystemId: SYSINT): HResult; stdcall;
    property OnelementDecl:         TOnelementDecl
                                    read  FOnelementDecl
                                    write FOnelementDecl;
    property OnattributeDecl:       TOnattributeDecl
                                    read  FOnattributeDecl
                                    write FOnattributeDecl;
    property OninternalEntityDecl:  TOninternalEntityDecl
                                    read  FOninternalEntityDecl
                                    write FOninternalEntityDecl;
    property OnexternalEntityDecl:  TOnexternalEntityDecl
                                    read  FOnexternalEntityDecl
                                    write FOnexternalEntityDecl;
    property  SAXParser: TSAXParser read FSAXParser write FSAXParser;
  end;


// attributes are passed back as a list of TSXPAttribute
// objects to the application
 TSXPAttribute = class
  public
    URI: string;
    LocalName: string;
    QName: string;
    Value: string;
    AttType: string;
  end;

// add a GetItem function to the TList of TSXPAttributes,
// so that we can get at an attribute by name
  TSXPAttributeList = class(TList)
  public
    function GetItem(const AttrName: string): TSXPAttribute;
  end;

// define the TSAXParser contenthandler callback procedures
  TOnStartDocument = procedure of object;
  TOnEndDocument   = procedure of object;
  TOnStartElement  = procedure(const NamespaceURI: string;
                               const Localname: string;
                               const QName: string;
                               const Attributes: TSXPAttributeList) of object;
  TOnEndElement    = procedure(const NamespaceURI: string;
                               const Localname: string;
                               const QName: string) of object;
  TOnCharacters    = procedure(const Chars: string) of object;
  TOnSkippedEntity = procedure(const EntityName: string) of object;

// TSAXContentHandler implements the ISAXContentHandler interface
  TSAXContentHandler = class(TInterfacedObject, ISAXContentHandler)
  private
    FSAXParser:        TSAXParser;      // the TSAXParser that owns us
    FDocumentLocator:  ISAXLocator;     // set by putDocumentlocator
    FUpdateLocation:   boolean;
    FAttributeList:    TSXPAttributeList;
    FLine:             integer;
    FColumn:           integer;
    // TSAXParser event handler pointers
    FOnStartDocument:  TOnStartDocument;
    FOnEndDocument:    TOnEndDocument;
    FOnStartElement:   TOnStartElement;
    FOnEndElement:     TOnEndElement;
    FOnCharacters:     TOnCharacters;
    FOnSkippedEntity:  TOnSkippedEntity;
  public
    function  putDocumentLocator(const pLocator: ISAXLocator):
                                 HResult; stdcall;
    function  startDocument: HResult; stdcall;
    function  endDocument: HResult; stdcall;
    function  startPrefixMapping(const pwchPrefix: pWideChar;
                                 cchPrefix: SYSINT;
                                 const pwchUri: pWideChar;
                                 cchUri: SYSINT): HResult; stdcall;
    function  endPrefixMapping(const pwchPrefix: pWideChar;
                               cchPrefix: SYSINT): HResult; stdcall;
    function  startElement(const pwchNamespaceUri: pWideChar;
                           cchNamespaceUri: SYSINT;
                           const pwchLocalName: pWideChar;
                           cchLocalName: SYSINT;
                           const pwchQName: pWideChar;
                           cchQName: SYSINT;
                           const pAttributes: ISAXAttributes):
                           HResult; stdcall;
    function  endElement(const pwchNamespaceUri: pWideChar;
                         cchNamespaceUri: SYSINT;
                         const pwchLocalName: pWideChar;
                         cchLocalName: SYSINT;
                         const pwchQName: pWideChar;
                         cchQName: SYSINT): HResult; stdcall;
    function  characters(const pwchChars: pWideChar;
                         cchChars: SYSINT): HResult; stdcall;
    function  ignorableWhitespace(const pwchChars: pWideChar;
                                  cchChars: SYSINT): HResult; stdcall;
    function  processingInstruction(const pwchTarget: pWideChar;
                                    cchTarget: SYSINT;
                                    const pwchData: pWideChar;
                                    cchData: SYSINT):
                                    HResult; stdcall;
    function  skippedEntity(const pwchName: pWideChar;
                            cchName: SYSINT): HResult; stdcall;
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure GetLineColumn;
    property  OnStartDocument:  TOnStartDocument
                                read FOnStartDocument
                                write FOnStartDocument;
    property  OnEndDocument:    TOnEndDocument
                                read FOnEndDocument
                                write FOnEndDocument;
    property  OnStartElement:   TOnStartElement
                                read FOnStartElement
                                write FOnStartElement;
    property  OnEndElement:     TOnEndElement
                                read FOnEndElement
                                write FOnEndElement;
    property  OnCharacters:     TOnCharacters
                                read FOnCharacters
                                write FOnCharacters;
    property  OnSkippedEntity:  TOnSkippedEntity
                                read FOnSkippedEntity
                                write FOnSkippedEntity;
    property  SAXParser:        TSAXParser
                                read FSaxParser
                                write FSAXparser;
    property UpdateLocation:    boolean
                                read FUpdateLocation
                                write FUpdateLocation default False;
    property Line:              integer
                                read FLine
                                write FLine;
    property Column:            integer
                                read FColumn
                                write FColumn;
  end;

// TSAXParser encapsulates the ISAX... COM interfaces
// and exposes them through a single regular Delphi VCL component
  TSAXParser = class(TComponent)
  private
    // reference to the SAX Reader COM object instance
    FReader:         ISAXXMLReader;
    // reference to the handler implementation classes
    FContenthandler: TSAXContentHandler;
    FErrorHandler:   TSAXErrorHandler;
    FLexicalHandler: TSAXLexicalHandler;
    FDTDHandler:     TSAXDTDHandler;
    FDeclHandler:    TSAXDeclHandler;
    //
    FXMLSource:      string;
    // supplied by the Contenthandler's ISAXLocator
    FPublicID:       string;
    FSystemId:       string;
    // property get/set access methods
    // 1. for TSAXLexicalHandler
    procedure SetOnStartDTD(const Value: TOnStartDTD);
    function  GetOnStartDTD: TOnStartDTD;
    procedure SetOnEndDTD(const Value: TOnEndDTD);
    function  GetOnEndDTD: TOnEndDTD;
    procedure SetOnStartEntity(const Value: TOnStartEntity);
    function  GetOnStartEntity: TOnStartEntity;
    procedure SetOnEndEntity(const Value: TOnEndEntity);
    function  GetOnEndEntity: TOnEndEntity;
    procedure SetOnStartCDATA(const Value: TOnStartCDATA);
    function  GetOnStartCDATA: TOnStartCDATA;
    procedure SetOnEndCDATA(const Value: TOnEndCDATA);
    function  GetOnEndCDATA: TOnEndCDATA;
    procedure SetOnComment(const Value: TOnComment);
    function  GetOnComment: TOnComment;
    // 2. For TSAXContentHandler
    procedure SetOnStartDocument(const Value: TOnStartDocument);
    function  GetOnStartDocument: TOnStartDocument;
    procedure SetOnEndDocument(const Value: TOnEndDocument);
    function  GetOnEndDocument: TOnEndDocument;
    procedure SetOnStartElement(const Value: TOnStartElement);
    function  GetOnStartElement: TOnStartElement;
    procedure SetOnEndElement(const Value: TOnEndElement);
    function  GetOnEndElement: TOnEndElement;
    procedure SetOnCharacters(const Value: TOnCharacters);
    function  GetOnCharacters: TOnCharacters;
    procedure SetOnSkippedEntity(const Value: TOnSkippedEntity);
    function  GetOnSkippedEntity: TOnSkippedEntity;
    // 3. For TSAXErroHandler
    procedure SetOnError(const Value: TOnError);
    function  GetOnError: TOnError;
    procedure SetOnFatalError(const Value: TOnError);
    function  GetOnFatalError: TOnError;
    procedure SetOnWarning(const Value: TOnError);
    function  GetOnWarning: TOnError;
    // 4. For TSAXDTDThandler
    function GetOnnotationDecl: TOnnotationDecl;
    function GetOnunparsedEntityDecl: TOnunparsedEntityDecl;
    procedure SetOnnotationDecl(const Value: TOnnotationDecl);
    procedure SetOnunparsedEntityDecl(const Value: TOnunparsedEntityDecl);
    // 5. For TSAXDeclHandler
    function GetOnattributeDecl: TOnattributeDecl;
    function GetOnelementDecl: TOnelementDecl;
    function GetOnexternalEntityDecl: TOnexternalEntityDecl;
    function GetOninternalEntityDecl: TOninternalEntityDecl;
    function GetColumn: integer;
    function GetLine: integer;
    function GetUpdateLocation: boolean;
    procedure SetOnattributeDecl(const Value: TOnattributeDecl);
    procedure SetOnelementDecl(const Value: TOnelementDecl);
    procedure SetOnexternalEntityDecl(const Value: TOnexternalEntityDecl);
    procedure SetOninternalEntityDecl(const Value: TOninternalEntityDecl);
    procedure SetColumn(const Value: integer);
    procedure SetLine(const Value: integer);
    procedure SetUpdateLocation(const Value: boolean);
  public
    property ContentHandler:   TSAXContentHandler
                               read  FContentHandler;
    property PublicID:         string  read FPublicID;
    property SystemID:         string  read FSystemID;
    property Line:             integer read GetLine
                                       write SetLine;
    property Column:           integer read GetColumn
                                       write SetColumn;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Parse;
    procedure GetLocation;
  published
    // TSAXParser
    property XMLSource:        string
                               read  FXMLSource
                               write FXMLSource;
    property UpdateLocation:   boolean
                               read GetUpdateLocation
                               write SetUpdateLocation;
    // TSAXDTDHandler
    property  OnnotationDecl : TOnnotationDecl
                               read GetOnnotationDecl
                               write SetOnnotationDecl;
    property  OnunparsedEntityDecl : TOnunparsedEntityDecl
                                     read GetOnunparsedEntityDecl
                                     write SetOnunparsedEntityDecl;
    // TSAXDeclHandler
    property OnelementDecl:         TOnelementDecl
                                    read  GetOnelementDecl
                                    write SetOnelementDecl;
    property OnattributeDecl:       TOnattributeDecl
                                    read  GetOnattributeDecl
                                    write SetOnattributeDecl;
    property OninternalEntityDecl:  TOninternalEntityDecl
                                    read  GetOninternalEntityDecl
                                    write SetOninternalEntityDecl;
    property OnexternalEntityDecl:  TOnexternalEntityDecl
                                    read  GetOnexternalEntityDecl
                                    write SetOnexternalEntityDecl;
    // TSAXLexicalHandler
    property OnStartDTD:       TOnStartDTD
                               read GetOnStartDTD
                               write SetOnStartDTD;
    property OnEndTDT:         TOnEndDTD
                               read GetOnEndDTD
                               write SetOnEndDTD;
    property OnStartEntity:    TOnStartEntity
                               read GetOnStartEntity
                               write SetOnStartEntity;
    property OnEndEntity:      TOnEndEntity
                               read GetOnEndEntity
                               write SetOnEndEntity;
    property OnStartCDATA:     TOnStartCdata
                               read GetOnStartCDATA
                               write SetOnStartCDATA;
    property OnEndCDATA:       TOnEndCdata
                               read GetOnEndCDATA
                               write SetOnEndCDATA;
    property OnComment:        TOnComment
                               read GetOnComment
                               write SetOnComment;
    // TSAXContentHandler
    property OnStartDocument:  TOnStartDocument
                               read GetOnStartDocument
                               write SetOnStartDocument;
    property OnEndDocument:    TOnEndDocument
                               read GetOnEndDocument
                               write SetOnEndDocument;
    property OnStartElement:   TOnStartElement
                               read GetOnStartElement
                               write SetOnStartElement;
    property OnEndElement:     TOnEndElement
                               read GetOnEndElement
                               write SetOnEndElement;
    property OnCharacters:     TOnCharacters
                               read GetonCharacters
                               write SetOnCharacters;
    property OnSkippedEntity:  TOnSkippedEntity
                               read GetOnSkippedEntity
                               write SetOnSkippedEntity;
    // TSAXErrorHandler
    property OnError:          TOnError
                               read GetOnError
                               write SetOnError;
    property OnFatalError:     TOnError
                               read GetOnFatalError
                               write SetOnFatalError;
    property OnWarning:        TOnError
                               read GetOnWarning
                               write SetOnWarning;
  end;


procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('XML', [TSAXParser]);
end;

{ TSXPAttributeList }

// add case insensitive access by attribute name to the attribute list
function TSXPAttributeList.GetItem(const AttrName: string): TSXPAttribute;
var
  i: integer;
begin
  Result := nil;
  for i := 0 to Count - 1 do begin
    if CompareText(AttrName,
                   TSXPAttribute(Items[i]).LocalName) = 0 then begin
      Result := Items[i];
      exit;
    end;
  end;
end;

{ TSAXErrorhandler }

//
//  TSAXErrorHandler, implements the ISAXErrorHandler interface
//
// The error location and message are extracted, and these are
// handed on to the eventhandlers set by TSAXParser
//
function  TSAXErrorhandler.error(const pLocator: ISAXLocator;
                                 const pwchErrorMessage: pWideChar;
                                 hrErrorCode: HResult): HResult; stdcall;
var
  strMsg: string;
  line: integer;
  column: integer;
begin
  if pLocator <> Nil then begin
    pLocator.getColumnNumber(column);
    pLocator.getLineNumber(line);
  end else begin
    line := -1;
    column := -1;
  end;
  WideCharToStrVar(pwchErrorMessage, strMsg);
  if Assigned(FOnError) then begin
    try
      FOnError(line, column, strMsg);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXErrorhandler.fatalError(const pLocator: ISAXLocator;
                                      const pwchErrorMessage: pWideChar;
                                      hrErrorCode: HResult): HResult; stdcall;
var
  strMsg: string;
  line: integer;
  column: integer;
begin
  if pLocator <> Nil then begin
    pLocator.getColumnNumber(column);
    pLocator.getLineNumber(line);
  end else begin
    line := -1;
    column := -1;
  end;
  WideCharToStrVar(pwchErrorMessage, strMsg);
  if Assigned(FOnFatalError) then begin
    try
      FOnFatalError(line, column, strMsg);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXErrorhandler.ignorableWarning(const pLocator: ISAXLocator;
                                            const pwchErrorMessage: pWideChar;
                                            hrErrorCode: HResult): HResult; stdcall;
var
  strMsg: string;
  line: integer;
  column: integer;
begin
  if pLocator <> Nil then begin
    pLocator.getColumnNumber(column);
    pLocator.getLineNumber(line);
  end else begin
    line := -1;
    column := -1;
  end;
  WideCharToStrVar(pwchErrorMessage, strMsg);
  if Assigned(FOnWarning) then begin
    try
      FOnWarning(line, column, strMsg);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

{ TSAXDTDHandler }

//
// TSAXDTDHandler implements the ISAXDTDHandler interface
//
function  TSAXDTDHandler.notationDecl(const pwchName: pWideChar;
                                      cchName: SYSINT;
                                      const pwchPublicId: pWideChar;
                                      cchPublicId: SYSINT;
                                      const pwchSystemId: pWideChar;
                                      cchSystemId: SYSINT): HResult; stdcall;
var
  strName, strPublicID, strSystemID: string;
begin
  if Assigned(FOnnotationDecl) then begin
    FSAXParser.ContentHandler.GetLineColumn;
    try
      if pwchName <> Nil then
        strName := WideCharLenToString(pwchName, cchName)
      else
        strName := '';
      if pwchPublicID <> Nil then
        strPublicID := WideCharLenToString(pwchPublicID, cchPublicID)
      else
        strPublicID := '';
      if pwchSystemID <> Nil then
        strSystemID := WideCharLenToString(pwchSystemID, cchSystemID)
      else
        strSystemID := '';
      FOnnotationDecl(strName, strPublicID, strSystemID);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXDTDHandler.unparsedEntityDecl(const pwchName: pWideChar;
                                            cchName: SYSINT;
                                            const pwchPublicId: pWideChar;
                                            cchPublicId: SYSINT;
                                            const pwchSystemId: pWideChar;
                                            cchSystemId: SYSINT;
                                            const pwchNotationName: pWideChar;
                                            cchNotationName: SYSINT): HResult; stdcall;
var
  strName, strPublicID, strSystemID, strNotationName: string;
begin
  if Assigned(FOnunparsedEntityDecl) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchName <> Nil then
        strName := WideCharLenToString(pwchName, cchName)
      else
        strName := '';
      if pwchPublicID <> Nil then
        strPublicID := WideCharLenToString(pwchPublicID, cchPublicID)
      else
        strPublicID := '';
      if pwchSystemID <> Nil then
        strSystemID := WideCharLenToString(pwchSystemID, cchSystemID)
      else
        strSystemID := '';
      if pwchNotationName <> Nil then
        strNotationName := WideCharLenToString(pwchNotationName,
                                                   cchNotationName)
      else
        strNotationName := '';
      FOnunparsedEntityDecl(strName, strPublicID, strSystemID, strNotationName);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

{ TSAXDeclhandler }

//
// TSAXDeclHandler implements ISAXDeclHandler
//
function TSAXDeclHandler.attributeDecl(const pwchElementName: pWideChar;
                                       cchElementName: SYSINT;
                                       const pwchAttributeName: pWideChar;
                                       cchAttributeName: SYSINT;
                                       const pwchType: pWideChar;
                                       cchType: SYSINT;
                                       const pwchValueDefault: pWideChar;
                                       cchValueDefault: SYSINT;
                                       const pwchValue: pWideChar;
                                       cchValue: SYSINT): HResult;
var
  strElementName, strAttributeName, strType, strValueDefault, strValue: string;
begin
  if Assigned(FOnattributeDecl) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchElementName <> Nil then
        strElementName := WideCharLenToString(pwchElementName, cchElementName)
      else
        strElementName := '';
      if pwchAttributeName <> Nil then
        strAttributeName := WideCharLenToString(pwchAttributeName,
                                                cchAttributeName)
      else
        strAttributeName := '';
      if pwchType <> Nil then
        strType := WideCharLenToString(pwchType, cchType)
      else
        strType := '';
      if pwchValueDefault <> Nil then
        strValueDefault := WideCharLenToString(pwchValueDefault,
                                               cchValueDefault)
      else
        strValueDefault := '';
      if pwchValue <> Nil then
        strValue := WideCharLenToString(pwchValue, cchValue)
      else
        strValue := '';

      FOnattributeDecl(strElementname, strAttributeName, strType, strValueDefault, strValue);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function TSAXDeclHandler.elementDecl(const pwchName: pWideChar;
                                     cchName: SYSINT;
                                     const pwchModel: pWideChar;
                                     cchModel: SYSINT): HResult;
var
  strName, strModel: string;
begin
  if Assigned(FOnelementDecl) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchName <> Nil then
        strName := WideCharLenToString(pwchName, cchName)
      else
        strName := '';
      if pwchModel <> Nil then
        strModel := WideCharLenToString(pwchModel, cchModel)
      else
        strModel := '';

      FOnElementDecl(strName, strModel);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function TSAXDeclHandler.externalEntityDecl(const pwchName: pWideChar;
                                            cchName: SYSINT;
                                            const pwchPublicId: pWideChar;
                                            cchPublicId: SYSINT;
                                            const pwchSystemId: pWideChar;
                                            cchSystemId: SYSINT): HResult;
var
  strName, strPublicID, strSystemID: string;
begin
  if Assigned(FOnexternalEntityDecl) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchName <> Nil then
        strName := WideCharLenToString(pwchName, cchName)
      else
        strName := '';
      if pwchPublicID <> Nil then
        strPublicID := WideCharLenToString(pwchPublicID, cchPublicID)
      else
        strPublicID := '';
      if pwchSystemID <> Nil then
        strSystemID := WideCharLenToString(pwchSystemID, cchSystemID)
      else
        strSystemID := '';
      FOnexternalEntityDecl(strName, strPublicID, strSystemID);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;


function TSAXDeclHandler.internalEntityDecl(const pwchName: pWideChar;
                                            cchName: SYSINT;
                                            const pwchValue: pWideChar;
                                            cchValue: SYSINT): HResult;
var
  strName, strValue: string;
begin
  if Assigned(FOninternalEntityDecl) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchName <> Nil then
        strName := WideCharLenToString(pwchName, cchName)
      else
        strName := '';
      if pwchValue <> Nil then
        strValue := WideCharLenToString(pwchValue, cchValue)
      else
        strValue := '';

      FOninternalEntityDecl(strName, strValue);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

{ TSAXLexicalhandler }

//
//  TSAXLexicalHandler, implements the ISAXLexicalHandler interface
//

function  TSAXLexicalHandler.startDTD(const pwchName: pWideChar;
                                      cchName: SYSINT;
                                      const pwchPublicId: pWideChar;
                                      cchPublicId: SYSINT;
                                      const pwchSystemId: pWideChar;
                                      cchSystemId: SYSINT): HResult; stdcall;
var
  strName, strPublicID, strSystemID: string;
begin
  if Assigned(FOnStartDTD) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchName <> Nil then
        strName := WideCharLenToString(pwchName, cchName)
      else
        strName := '';
      if pwchPublicID <> Nil then
        strPublicID := WideCharLenToString(pwchPublicID, cchPublicID)
      else
        strPublicID := '';
      if pwchSystemID <> Nil then
        strSystemID := WideCharLenToString(pwchSystemID, cchSystemID)
      else
        strSystemID := '';
      FOnStartDTD(strName, strPublicID, strSystemID);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXLexicalHandler.endDTD: HResult; stdcall;
begin
  if Assigned(FOnEndDTD) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      FOnEndDTD;
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXLexicalHandler.startEntity(const pwchName: pWideChar;
                                         cchName: SYSINT): HResult; stdcall;
var
  strname: string;
begin
  if Assigned(FOnStartEntity) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchName <> Nil then
        strName := WideCharLenToString(pwchName, cchName)
      else
        strName := '';
      FOnStartEntity(strName);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXLexicalHandler.endEntity(const pwchName: pWideChar;
                                       cchName: SYSINT): HResult; stdcall;
var
  strName: string;
begin
  if Assigned(FOnEndEntity) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchName <> Nil then
        strName := WideCharLenToString(pwchName, cchName)
      else
        strName := '';
      FOnEndEntity(strName);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXLexicalHandler.startCDATA: HResult; stdcall;
begin
  if Assigned(FOnStartCDATA) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      FOnStartCDATA;
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXLexicalHandler.endCDATA: HResult; stdcall;
begin
  if Assigned(FOnendCDATA) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      FOnEndCDATA;
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXLexicalHandler.comment(const pwchChars: pWideChar;
                                     cchChars: SYSINT): HResult; stdcall;
var
  strChars: string;
begin
  if Assigned(FOnComment) then begin
    FSAXparser.ContentHandler.GetLineColumn;
    try
      if pwchChars <> Nil then
        strChars := WideCharLenToString(pwchChars, cchChars)
      else
        strChars := '';
      FOnComment(strChars);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

{ TSAXContentHandler }

//
//  TSAXContentHandler, implements the ISAXContentHandler interface
//
//
procedure TSAXContentHandler.AfterConstruction;
begin
  inherited;
  FAttributeList := TSXPAttributeList.Create;
  FDocumentLocator := Nil;
  FUpdateLocation := False;
  FLine := 0;
  FColumn := 0;
end;

procedure TSAXContentHandler.BeforeDestruction;
begin
  FreeAndNil(FAttributeList);
  FDocumentLocator := Nil;
  inherited;
end;

// these functions implement the content handler callbacks,
// the information supplied by the reader is extracted, formatted
// and passed on to the event handlers set by TSAXParser
//

// putDocumentLocator saves a reference to the ISAXLocator instance
// provided by the reader for later use in the other ContentHandler events
// it will provide access to line and column information
function TSAXContenthandler.putDocumentLocator(const pLocator: ISAXLocator):
                                               HResult; stdcall;
begin
  FDocumentLocator := pLocator;
  FLine := 0;
  FColumn := 0;
  Result := S_OK;
end;

// startDocument uses the Locator saved by putDocumentLocator to
// retrieve the SystemID,and PublicID (if any)
// It then calls the eventhandler (if any) set by the TSAXParser
// component that created this TSAXContenthandler instance
function TSAXContenthandler.startDocument: HResult; stdcall;
var
  pPublicID, pSystemID: pWideChar;
begin
  if Assigned(FDocumentLocator) then begin
    FDocumentLocator.getPublicId(pPublicId);
    if pPublicID <> Nil then begin
      FSaxParser.FPublicID := WideCharToString(pPublicID);
    end;
    FDocumentLocator.getSystemId(pSystemId);
    if pSystemID <> Nil then begin
      FSaxParser.FSystemID := WideCharToString(pSystemID);
    end;
  end;
  if Assigned(FOnStartDocument) then begin
    GetLineColumn;
    try
      FOnStartDocument;
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function TSAXContenthandler.endDocument: HResult; stdcall;
begin
  if Assigned(FOnEndDocument) then begin
    GetLineColumn;
    try
      FOnEndDocument;
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function TSAXContenthandler.startPrefixMapping(
                        const pwchPrefix: pWideChar;
                        cchPrefix: SYSINT;
                        const pwchUri: pWideChar;
                        cchUri: SYSINT): HResult; stdcall;
begin
  GetLineColumn;
  // todo
  Result := S_OK;
end;

function  TSAXContenthandler.endPrefixMapping(
                         const pwchPrefix: pWideChar;
                         cchPrefix: SYSINT): HResult; stdcall;
begin
  GetLineColumn;
  // todo
  Result := S_OK;
end;

function  TSAXContenthandler.startElement(
                         const pwchNamespaceUri: pWideChar;
                         cchNamespaceUri: SYSINT;
                         const pwchLocalName: pWideChar;
                         cchLocalName: SYSINT;
                         const pwchQName: pWideChar;
                         cchQName: SYSINT;
                         const pAttributes: ISAXAttributes): HResult; stdcall;
var
  strNamespaceURI: string;
  strLocalName: string;
  strQName: string;
  Attribute: TSXPAttribute;
  i: integer;
  nAttributes : integer;
  pURI, pLocalName, pQname, pValue, pType: pWideChar;
  URIsize, Localsize, Qsize, Typesize, size: integer;
begin
  if Assigned(FOnStartElement) then begin
    GetLineColumn;
    // convert element name and URI to string
    try
      if pwchNamespaceUri <> Nil then
        strNamespaceURI := WideCharLenToString(pwchNamespaceUri,
                                               cchNamespaceUri)
      else
        strNamespaceURI := '';
      if pwchLocalName <> Nil then
        strLocalName := WideCharLenToString(pwchLocalName,
                                            cchLocalName)
      else
        strLocalName := '';
      if pwchQName <> Nil then
        strQName := WideCharLenToString(pwchQName,
                                        cchQName)
      else
        strQName := '';
      // build the attribute list
      try
        if pAttributes <> Nil then begin
          with pAttributes do begin
            getLength(nAttributes);
            FAttributeList.Capacity := nAttributes;
            for i := 0 to nAttributes - 1 do begin
              getName(i, pURI, URIsize,
                         pLocalName, Localsize,
                         pQName, Qsize);
              getValue(i, pValue, size);
              getType(i, pType, Typesize);
              Attribute := TSXPAttribute.Create;
              with Attribute do begin
                URI       := WideCharLenToString(pURI, URIsize);
                LocalName := WideCharLenToString(pLocalName, Localsize);
                QName     := WideCharLenToString(pQName, Qsize);
                Value     := WideCharLenToString(pValue, size);
                AttType   := WideCharLenToString(pType, Typesize);
              end;
              FAttributeList.Add(Attribute);
            end; // for
          end; // with
        end; // if pAttributes <> Nil
        // now call the event handler
        FOnStartElement(strNamespaceUri,
                        strLocalName,
                        strQName,
                        FAttributeList);
      finally
        // clean up the attributes list
        for i := 0 to FAttributeList.Count - 1 do begin
          TSXPAttribute(FAttributeList.Items[i]).Free;
        end;
        FAttributeList.Clear;
      end;

      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXContenthandler.endElement(
                         const pwchNamespaceUri: pWideChar;
                         cchNamespaceUri: SYSINT;
                         const pwchLocalName: pWideChar;
                         cchLocalName: SYSINT;
                         const pwchQName: pWideChar;
                         cchQName: SYSINT): HResult; stdcall;
var
  strNamespaceURI: string;
  strLocalName: string;
  strQName: string;
begin
  if Assigned(FOnEndElement) then begin
    GetLineColumn;
    try
      if pwchNamespaceUri <> Nil then
        strNamespaceURI := WideCharLenToString(pwchNamespaceUri,
                                               cchNamespaceUri)
      else
        strNamespaceURI := '';
      if pwchLocalName <> Nil then
        strLocalName := WideCharLenToString(pwchLocalName,
                                            cchLocalName)
      else
        strLocalName := '';
      if pwchQName <> Nil then
        strQName := WideCharLenToString(pwchQName, cchQName)
      else
        strLocalName := '';
      FOnEndElement(strNamespaceUri, strLocalName, strQName);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

function  TSAXContenthandler.characters(
                         const pwchChars: pWideChar;
                         cchChars: SYSINT): HResult; stdcall;
var
  strChars: string;
begin
  if Assigned(FOnCharacters) then begin
    GetLineColumn;
    try
      if pwchChars <> Nil then
        strChars := WideCharLenToString(pwchChars, cchChars)
      else
        strChars := '';
      FOnCharacters(strChars);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

// currently not implemented by the reader in MSXML3.0
function  TSAXContenthandler.ignorableWhitespace(
                         const pwchChars: pWideChar;
                         cchChars: SYSINT): HResult; stdcall;
begin
  GetLineColumn;
  // todo
  Result := S_OK;
end;

function  TSAXContenthandler.processingInstruction(
                         const pwchTarget: pWideChar;
                         cchTarget: SYSINT;
                         const pwchData: pWideChar;
                         cchData: SYSINT): HResult; stdcall;
begin
  GetLineColumn;
  // todo
  Result := S_OK;
end;

function  TSAXContenthandler.skippedEntity(
                         const pwchName: pWideChar;
                         cchName: SYSINT): HResult; stdcall;
var
  strEntity: string;
begin
  if Assigned(FOnSkippedEntity) then begin
    GetLineColumn;
    try
      if pwchName <> Nil then
        strEntity := WideCharLenToString(pwchName, cchName)
      else
        strEntity := '';
      FOnSkippedEntity(strEntity);
      Result := S_OK;
    except
      Result := E_Fail;
    end;
  end else begin
    Result := S_OK;
  end;
end;

// get line and column from the ISAXLocator
procedure TSaxContentHandler.GetLineColumn;
begin
  if FUpdateLocation then begin
    if Assigned(FDocumentLocator) then begin
      FDocumentLocator.getLineNumber(FLine);
      FDocumentLocator.getColumnNumber(FColumn);
    end;
  end;
end;

{ TSAXPArser }

//
// TSAXParser Delphi wrapper
//
constructor TSAXParser.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  // create an instance of the Reader COM object
  FReader := CreateComObject(CLASS_SAXXMLReader) as ISAXXMLReader;
  // instantiate the handler classes
  FErrorHandler   := TSAXErrorHandler.Create;
  FLexicalHandler := TSAXLexicalHandler.Create;
  FContenthandler := TSAXContentHandler.Create;
  FDTDHandler     := TSAXDTDHandler.Create;
  FDeclHandler    := TSAXDeclhandler.Create;
  // Handlers need a reference pointing back to us
  // because they will continually update
  // Line and Column properties on each event
  FLexicalhandler.SAXParser := Self;
  FContenthandler.SAXParser := Self;
  FDTDHandler.SAXParser := Self;
  FDeclHandler.SAXParser := Self;
  // pass the handler implementations to the reader
  FReader.putErrorHandler(FErrorhandler);
  FReader.putContentHandler(FContentHandler);
  FReader.putDTDHandler(FDTDHandler);
  Freader.putProperty('http://xml.org/sax/properties/lexical-handler',
                      FLexicalhandler as ISAXLexicalHandler);
  Freader.putProperty('http://xml.org/sax/properties/declaration-handler',
                      FDeclhandler as ISAXdeclHandler);
end;

destructor TSAXParser.Destroy;
begin
  FReader := Nil;
  inherited Destroy;
end;

// fire up the reader
procedure TSAXParser.Parse;
var
  hr: HRESULT;
  wstrURL: Widestring;
begin
  wstrURL := FXMLSource;
  hr := FReader.ParseURL(pWideChar(wstrURL));
  if hr <> S_OK then begin
    Raise ESAXParser.Create(Format('Error <%x> parsing <%s>',
                                   [hr, FXMLSource]));
  end;
end;

// if UpdateLocation is off, you can still ask for
// a one-shot update of the current Line and Column properties
procedure TSAXParser.GetLocation;
begin
  if UpdateLocation = False then begin
    UpdateLocation := True;
    FContenthandler.GetLineColumn;
    UpdateLocation := False;
  end;
end;


// the event handler method pointers are not stored in TSAXParser, but in the
// TASAX....Handler objects instantiated in the constructor

// 1. TSAXLexicalhandler
procedure TSAXParser.SetOnStartDTD(const Value: TOnStartDTD);
begin
  FLexicalHandler.OnStartDTD := Value;
end;
function  TSAXParser.GetOnStartDTD: TOnStartDTD;
begin
  Result := FLexicalHandler.OnStartDTD;
end;
procedure TSAXParser.SetOnEndDTD(const Value: TOnEndDTD);
begin
  FLexicalHandler.OnEndDTD := Value;
end;
function  TSAXParser.GetOnEndDTD: TOnEndDTD;
begin
  Result := FLexicalHandler.OnEndDTD;
end;
procedure TSAXParser.SetOnStartEntity(const Value: TOnStartEntity);
begin
  FLexicalHandler.OnStartEntity := Value;
end;
function  TSAXParser.GetOnStartEntity: TOnStartEntity;
begin
  Result := FLexicalHandler.OnStartEntity;
end;
procedure TSAXParser.SetOnEndEntity(const Value: TOnEndEntity);
begin
  FLexicalHandler.OnEndEntity := Value;
end;
function  TSAXParser.GetOnEndEntity: TOnEndEntity;
begin
  Result := FLexicalHandler.OnEndEntity;
end;
procedure TSAXParser.SetOnStartCDATA(const Value: TOnStartCDATA);
begin
  FLexicalHandler.OnStartCDATA := Value;
end;
function  TSAXParser.GetOnStartCDATA: TOnStartCDATA;
begin
  Result := FLexicalHandler.OnStartCDATA;
end;
procedure TSAXParser.SetOnEndCDATA(const Value: TOnEndCDATA);
begin
  FLexicalHandler.OnEndCDATA := Value;
end;
function  TSAXParser.GetOnEndCDATA: TOnEndCDATA;
begin
  Result := FLexicalHandler.OnEndCDATA;
end;
procedure TSAXParser.SetOnComment(const Value: TOnComment);
begin
  FLexicalHandler.OnComment := Value;
end;
function  TSAXParser.GetOnComment: TOnComment;
begin
  Result := FLexicalHandler.OnComment;
end;

// 2. TSAXContentHandler

function TSAXParser.GetUpdateLocation: boolean;
begin
  Result := FContentHandler.UpdateLocation;
end;
procedure TSAXParser.SetUpdateLocation(const Value: boolean);
begin
  FContentHandler.UpdateLocation := Value;
end;
function TSAXParser.GetColumn: integer;
begin
  Result := FContentHandler.Column;
end;
function TSAXParser.GetLine: integer;
begin
  Result := FContentHandler.Line;
end;
procedure TSAXParser.SetColumn(const Value: integer);
begin
  FContentHandler.Column := Value;
end;
procedure TSAXParser.SetLine(const Value: integer);
begin
  FContentHandler.Line := value;
end;

procedure TSAXParser.SetOnStartDocument(const Value: TOnStartDocument);
begin
  FContentHandler.OnStartDocument := Value;
end;
function TSAXParser.GetOnStartDocument: TOnStartDocument;
begin
  Result := FContentHandler.OnStartDocument;
end;
procedure TSAXParser.SetOnEndDocument(const Value: TOnEndDocument);
begin
  FContentHandler.OnEndDocument := Value;
end;
function TSAXParser.GetOnEndDocument: TOnEndDocument;
begin
  Result := FContentHandler.OnEndDocument;
end;
procedure TSAXParser.SetOnStartElement(const Value: TOnStartElement);
begin
  FContentHandler.OnStartElement := Value;
end;
function TSAXParser.GetOnStartElement: TOnStartElement;
begin
  Result := FContentHandler.OnStartElement;
end;
procedure TSAXParser.SetOnEndElement(const Value: TOnEndElement);
begin
  FContentHandler.OnEndElement := Value;
end;
function TSAXParser.GetOnEndElement: TOnEndElement;
begin
  Result := FContentHandler.OnEndElement;
end;
procedure TSAXParser.SetonCharacters(const Value: TOnCharacters);
begin
  FContentHandler.OnCharacters := Value;
end;
function TSAXParser.GetonCharacters: TOnCharacters;
begin
  Result := FContentHandler.OnCharacters;
end;
procedure TSAXParser.SetonSkippedEntity(const Value: TOnSkippedEntity);
begin
  FContentHandler.OnSkippedEntity := Value;
end;
function TSAXParser.GetonSkippedEntity: TOnSkippedEntity;
begin
  Result := FContentHandler.OnSkippedEntity;
end;

// 3. TSAXErrorHandler
procedure TSAXParser.SetonError(const Value: TOnError);
begin
  FErrorHandler.OnError := Value;
end;
function TSAXParser.GetonError: TOnError;
begin
  Result := FErrorHandler.OnError;
end;
procedure TSAXParser.SetonFatalError(const Value: TOnError);
begin
  FErrorHandler.OnFatalError := Value;
end;
function TSAXParser.GetonFatalError: TOnError;
begin
  Result := FErrorHandler.OnFatalError;
end;
procedure TSAXParser.SetonWarning(const Value: TOnError);
begin
  FErrorHandler.OnWarning := Value;
end;
function TSAXParser.GetonWarning: TOnError;
begin
  Result := FErrorHandler.OnWarning;
end;
// 4. TSAXDTDHandler
function TSAXParser.GetOnnotationDecl: TOnnotationDecl;
begin
  Result := FDTDHandler.OnnotationDecl;
end;
function TSAXParser.GetOnunparsedEntityDecl: TOnunparsedEntityDecl;
begin
  Result := FDTDHandler.OnunparsedEntityDecl;
end;
procedure TSAXParser.SetOnnotationDecl(const Value: TOnnotationDecl);
begin
  FDTDHandler.OnnotationDecl := Value;
end;
procedure TSAXParser.SetOnunparsedEntityDecl(
                     const Value: TOnunparsedEntityDecl);
begin
  FDTDHandler.OnunparsedEntityDecl := Value;
end;

// 5. TSAXDeclHandler
function TSAXParser.GetOnattributeDecl: TOnattributeDecl;
begin
  Result := FDeclHandler.FOnattributeDecl;
end;
function TSAXParser.GetOnelementDecl: TOnelementDecl;
begin
  Result := FDeclHandler.FOnelementDecl;
end;
function TSAXParser.GetOnexternalEntityDecl: TOnexternalEntityDecl;
begin
  Result := FDeclHandler.FOnexternalEntityDecl;
end;
function TSAXParser.GetOninternalEntityDecl: TOninternalEntityDecl;
begin
  Result := FDeclHandler.FOninternalEntityDecl;
end;
procedure TSAXParser.SetOnattributeDecl(const Value: TOnattributeDecl);
begin
  FDeclhandler.FOnattributeDecl := Value;
end;
procedure TSAXParser.SetOnelementDecl(const Value: TOnelementDecl);
begin
  FDeclhandler.FOnelementDecl := Value;
end;
procedure TSAXParser.SetOnexternalEntityDecl(const Value: TOnexternalEntityDecl);
begin
  FDeclhandler.FOnexternalEntityDecl := Value;
end;
procedure TSAXParser.SetOninternalEntityDecl(const Value: TOninternalEntityDecl);
begin
  FDeclhandler.FOninternalEntityDecl := Value;
end;


end.
