unit demoform;

// Small demo application to demonstrate the TSAXParser component
// written by Danny Heijl, Jan 2001 (danny.heijl@pandora.be)
// for Dr. Dobbs JOURNAL

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, MSSAXParser, ExtCtrls;

{.$DEFINE TEST_MEM_LEAK}  // remove the '.' to enable

{$IFDEF TEST_MEM_LEAK}
const LOOPCOUNT=1000;
{$ENDIF}

type
  TfrmDemo = class(TForm)
    LB1: TListBox;
    ChooseFile: TOpenDialog;
    Panel1: TPanel;
    Button1: TButton;
    SXP: TSAXParser;
    lblTime: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure Button1Click(Sender: TObject);
    procedure SXPStartDocument;
    procedure SXPEndDocument;
    procedure SXPStartElement(const NamespaceURI: string;
                              const Localname: string;
                              const QName: string;
                              const Attributes: TSXPAttributeList);
    procedure SXPEndElement(const NamespaceURI: string;
                            const Localname: string;
                            const QName: string);
    procedure SXPCharacters(const Chars: string);
    procedure SXPError(line: integer;
                       column: integer;
                       const strMsg: string);
    procedure SXPFatalError(line: integer;
                            column: integer;
                            const strMsg: string);
    procedure SXPWarning(line: integer;
                         column: integer;
                         const strMsg: string);
    procedure SXPSkippedEntity(const SkippedEntity: string);
    procedure SXPComment(const Chars: String);
    procedure SXPStartDTD(const Name, PublicID, SystemID: String);
    procedure SXPEndDTD;
    procedure SXPEndCDATA;
    procedure SXPStartCDATA;
    procedure SXPEndEntity(const name: String);
    procedure SXPStartEntity(const Name: String);
    procedure SXPnotationDecl(const Name, PublicID, SystemID: String);
    procedure SXPunparsedEntityDecl(const Name, PublicID, SystemID,
      NotationName: String);
    procedure SXPattributeDecl(const ElementName, AttributeName, AttrType,
      ValueDefault, Value: String);
    procedure SXPelementDecl(const Name, Model: String);
    procedure SXPexternalEntityDecl(const Name, PublicID,
      SystemID: String);
    procedure SXPinternalEntityDecl(const Name, Value: String);
    procedure SXPEndPrefixMapping(strPrefix: String);
    procedure SXPStartPrefixMapping( strPrefix, strURI: String);
  private
    { Private declarations }
    iLevel: integer;
    procedure DumpLine(strText: string);
  public
    { Public declarations }
  end;

var
  frmDemo: TfrmDemo;

implementation

{$R *.DFM}



// add a string to the listbox, indented by iLevel
procedure TfrmDemo.DumpLine(strText: string);
begin
{$IFNDEF TEST_MEM_LEAK}
  LB1.Items.Add(Format('%5.5d %s',
                       [SXP.Line,
                        StringOfChar('.', iLevel) + strText]));
{$ENDIF}
end;

procedure TfrmDemo.FormCreate(Sender: TObject);
begin

end;


// choose a file to parse, and parse it
procedure TfrmDemo.Button1Click(Sender: TObject);
var
  ts, te: TDateTime;
{$IFDEF TEST_MEM_LEAK}
  i: integer;
{$ENDIF}
begin
  lblTime.Caption := '';
  with ChooseFile do begin
    InitialDir := GetCurrentDir;
    if not Execute then exit;
    SXP.XMLSource := 'file://' + FileName;
  end;
  Button1.Enabled := False;
  try
    LB1.Clear;
    LB1.Items.Add('Parsing: ' + ChooseFile.FileName);
    LB1.Update;
    with SXP do begin
      try
        ts := Now;
        {$IFDEF TEST_MEM_LEAK}
        for i := 1 to LOOPCOUNT do begin
        {$ENDIF}
          iLevel := 0;
          LB1.Items.BeginUpdate;
          Parse;
          LB1.Items.EndUpdate;
        {$IFDEF TEST_MEM_LEAK}
          Application.ProcessMessages;
        end;
        {$ENDIF}
        te := Now;
        lblTime.Caption := Format('%4.6f sec', [(te -ts)*24*3600]);
      except
        on E:Exception do begin
          ShowMessage(E.Message);
          Close;
        end;
      end;
    end;
  finally
    Button1.Enabled := True;
  end;
end;

// the TSAXParser events

// Content handler events
procedure TfrmDemo.SXPStartDocument;
begin
  DumpLine('[CH] StartDocument');
  Inc(iLevel);
end;

procedure TfrmDemo.SXPEndDocument;
begin
  Dec(iLevel);
  DumpLine('[CH] EndDocument');
end;

procedure TfrmDemo.SXPStartElement(const NamespaceURI: string;
                              const Localname: string;
                              const QName: string;
                              const Attributes: TSXPAttributeList);
var
  i: integer;
  attr: TSXPAttribute;
begin
//  DumpLine(Format('[CH] <%s>', [LocalName]));
  DumpLine(Format('[CH] Local:<%s> Q:<%s> URI: %s', [LocalName, QName, NamespaceURI]));
  For i := 0 to Attributes.Count - 1 do begin
    attr := Attributes.Items[i];
    with attr do begin
      DumpLine(Format('[CH]  Attr: name=%s, Type=%s, value=''%s''',
                      [QName, AttType, Value]));
//                      [LocalName, AttType, Value]));
    end;
  end;
  Inc(iLevel);

  // Here's how you can get at attributes by name (case insensitive):
  // if Attributes.GetItem('Clerk') <> Nil then
  //   ShowMessage('clerk = ' + Attributes.GetItem('clerk').Value);

end;

procedure TfrmDemo.SXPEndElement(const NamespaceURI: string;
                                 const Localname: string;
                                 const QName: string);
begin
  Dec(iLevel);
//  DumpLine(Format('[CH] </%s>', [Localname]));
  DumpLine(Format('[CH] </%s>', [Qname]));
end;

procedure TfrmDemo.SXPCharacters(const Chars: string);

begin
  if Length(Trim(Chars)) > 0 then // skip whitespace lines
    DumpLine(Format('[CH] ''%s''', [Chars]));
end;

procedure TfrmDemo.SXPSkippedEntity(const SkippedEntity: string);
begin
  DumpLine('[CH]  *Skipped : ''' + SkippedEntity + '''');
end;

// Error handler events
procedure TfrmDemo.SXPError(line: integer;
                            column: integer;
                            const strMsg: string);
begin
  DumpLine(Format('[ErrH] Error %s at line %d, column %d',
                   [strMsg, line, column]));
end;

procedure TfrmDemo.SXPFatalError(line: integer;
                                 column: integer;
                                 const strMsg: string);
begin
  DumpLine(Format('[ErrH] Fatal Error %s at line %d, column %d',
                  [strMsg, line, column]));
end;

procedure TfrmDemo.SXPWarning(line: integer;
                              column: integer;
                              const strMsg: string);
begin
  DumpLine(Format('[ErrH] Warning %s at line %d, column %d',
                  [strMsg, line, column]));
end;

// Lexcical handler events
procedure TfrmDemo.SXPComment(const Chars: String);
begin
  DumpLine('[LexH] Comment: ''' + Chars + '''');
end;

procedure TfrmDemo.SXPStartDTD(const Name, PublicID, SystemID: String);
begin
  DumpLine('[LexH] Start DTD: ''' + Name + '''');
end;

procedure TfrmDemo.SXPEndDTD;
begin
  DumpLine('[LexH] End DTD');
end;

procedure TfrmDemo.SXPEndCDATA;
begin
  DumpLine('[LexH] End CDATA');
end;

procedure TfrmDemo.SXPStartCDATA;
begin
  DumpLine('[LexH] Start CDATA');
end;

procedure TfrmDemo.SXPEndEntity(const name: String);
begin
  DumpLine('[LexH] End Entity: ''' + Name + '''');
end;

procedure TfrmDemo.SXPStartEntity(const Name: String);
begin
  DumpLine('[LexH] Start Entitity: ''' + Name + '''');
end;

// DTD handler events
procedure TfrmDemo.SXPnotationDecl(const Name, PublicID, SystemID: String);
begin
  Dumpline(Format('[DTDH] NotationDecl: Name=%s, PublicID=%s, SystemID=%s',
                  [Name, PublicID, SystemID]));
end;

procedure TfrmDemo.SXPunparsedEntityDecl(const Name, PublicID, SystemID,
  NotationName: String);
begin
  Dumpline(Format('[DTDH] UnparsedEntityDecl: Name=%s, PublicID=%s, SystemID=%s, NotationName=%s',
                  [Name, PublicID, SystemID, NotationName]));
end;

// Declaration handler events
procedure TfrmDemo.SXPattributeDecl(const ElementName, AttributeName,
  AttrType, ValueDefault, Value: String);
begin
  DumpLine(Format(
  '[DeclH]AttributeDecl: ElementName=%s, AttributeName=%s, Type=%s, ValueDefault=%s, Value=%s',
  [ElementName, AttributeName, AttrType, ValueDefault, Value]));
end;

procedure TfrmDemo.SXPelementDecl(const Name, Model: String);
begin
  DumpLine(Format('[DeclH] ElementDecl: name=%s, Model=%s',
                  [Name, Model]));
end;

procedure TfrmDemo.SXPexternalEntityDecl(const Name, PublicID,
  SystemID: String);
begin
  DumpLine(Format('[DeclH]ExternalEntityDecl: Name=%s, PublicID=%s, SystemID=%s',
                   [Name, PublicID, SysteMID]));
end;

procedure TfrmDemo.SXPinternalEntityDecl(const Name, Value: String);
begin
  DumpLine(Format('[DeclH]InternalEntityDecl: Name=%s, Value=%s',
                   [Name, Value]));
end;

procedure TfrmDemo.SXPEndPrefixMapping(strPrefix: String);
begin
  DumpLine(Format('[CH]EndPrefixMapping: Prefix=%s',
                   [strPrefix]));
end;

procedure TfrmDemo.SXPStartPrefixMapping(strPrefix, strURI: String);
begin
  DumpLine(Format('[CH]StartPrefixMapping: URI=%s, Prefix=%s',
                   [strURI, strPrefix]));
end;

end.
