`timescale 1ns/1ns
module adr_arb (CLK, N_RESET, N_TS, N_BRA, N_BRB, ADR_BRA, ADR_BRB, N_BGA, N_BGB) ;

input CLK ;		// Buffered system clock.
input N_RESET ;		// Power on reset to initialize logic.
input N_TS ;		// Transfer start output from processors.
input N_BRA ;		// Bus request processor A.
input N_BRB ;		// Bus request processor B.

output ADR_BRA ;	// Address qualifier source: processor A.
output ADR_BRB ;	// Address qualifier source: processor B.
output N_BGA ;		// Bus grant to processor A.
output N_BGB ;		// Bus grant to processor B.

// Mealy state machine with synchronous outputs. This design provides a two
// processor address bus arbiter to support pipelined address transactions
// for a PowerPC 60X bus system.

// Module declarations:
  reg ADR_BRA ;
  reg ADR_BRB ;
  reg N_BGA ;
  reg N_BGB ;
  
// current and next state registers
// Note: increase size if the binary state machine grows beyond 7 states!! 
   reg [6:0] current, next;  
  
  // One Hot state machine encoding for address bus arbiter.
  parameter s0 = 7'b000_0001,  // IDLE
  	    s1 = 7'b000_0010,  // Grant A
  	    s2 = 7'b000_0100,  // Transfer start processor A
  	    s3 = 7'b000_1000,  // B request while A master
  	    s4 = 7'b001_0000,  // Grant B
  	    s5 = 7'b010_0000,  // Transfer start processor B
  	    s6 = 7'b100_0000;  // A request while B master

// State machine code:
always @(posedge CLK)
begin
  #2  // add output switching delay to emulate DFF switching delays.
  current = next;
  // start out in IDLE state after first rising edge of clk.
  next = s0;            
  ADR_BRA = 1'b0; ADR_BRB = 1'b0; N_BGA = 1'b1; N_BGB = 1'b1;  //deassert outputs 
  
  case(current)
  s0:
   begin
     if( ~N_BRA )
       begin  // uP_A has requested an ADDRESS BUS tenure. Grant it now.
         next = s1;    
         ADR_BRA = 1'b1; ADR_BRB = 1'b0; N_BGA = 1'b0; N_BGB = 1'b1;     
       end
     else
     if( (~N_BRB) & N_BRA ) // uP_B has requested an ADDRESS BUS tenure. Grant it now.
       begin       
         next = s2;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b1; N_BGA = 1'b1; N_BGB = 1'b0;    
       end
     else
       begin // Neither uP has requested an ADDRESS BUS tenure.        
         next = s0;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b0; N_BGA = 1'b1; N_BGB = 1'b1;          
       end
   end

  s1:
   begin
     if( ~N_TS )  // uP_A has started an ADDRESS BUS tenure.
       begin
         next = s3;    
         ADR_BRA = 1'b1; ADR_BRB = 1'b0; N_BGA = 1'b0; N_BGB = 1'b1;        
       end   
     else
     if( N_TS & (~N_BRB) )  // No transfer start from uP_A, uP_B request asserted.
       begin        
         next = s5;    
         ADR_BRA = 1'b1; ADR_BRB = 1'b0; N_BGA = 1'b0; N_BGB = 1'b1;          
       end
     else     
       begin  // Wait for uP_A to start the ADDRESS BUS tenure.        
         next = s1;    
         ADR_BRA = 1'b1; ADR_BRB = 1'b0; N_BGA = 1'b0; N_BGB = 1'b1;          
       end
   end

  s2:
   begin
     if( ~N_TS )  // uP_B has started an ADDRESS BUS tenure.
       begin        
         next = s4;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b1; N_BGA = 1'b1; N_BGB = 1'b0;          
       end   
     else
     if( N_TS & (~N_BRA) )  // No transfer start from uP_B, uP_A request asserted.
       begin        
         next = s6;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b1; N_BGA = 1'b1; N_BGB = 1'b0;          
       end
     else     
       begin  // Wait for uP_B to start the ADDRESS BUS tenure.        
         next = s2;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b1; N_BGA = 1'b1; N_BGB = 1'b0;          
       end
   end

  s3:
   begin  // this state stores address qualifiers associated with uP_A address tenure.
     if( ~N_BRB )
       begin
         // uP_B has requested an ADDRESS BUS tenure.
         // Store address & bus qualifiers in address pipeline and deassert N_BGA.
         next = s2;    
         ADR_BRA = 1'b1; ADR_BRB = 1'b0; N_BGA = 1'b1; N_BGB = 1'b0;      
       end   
     else     
       begin  // Park address grant on uP_A.        
         next = s1;    
         ADR_BRA = 1'b1; ADR_BRB = 1'b0; N_BGA = 1'b0; N_BGB = 1'b1;           
       end
   end

  s4:
   begin
     if( ~N_BRA )
       begin
         // uP_A has requested an ADDRESS BUS tenure.
         // Store address & bus qualifiers in address pipeline and deassert N_BGB.
         next = s1;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b1; N_BGA = 1'b0; N_BGB = 1'b1;     
       end   
     else     
       begin  // Park address grant on uP_B.
         next = s2;    
       ADR_BRA = 1'b0; ADR_BRB = 1'b1; N_BGA = 1'b1; N_BGB = 1'b0;         
       end
   end

  s5:
   begin
     if( ~N_TS )
       begin
         // uP_A has started the ADDRESS BUS tenure while grant is transitioning.
         // Go to s3 to store uP_A address bus qualifiers.
         next = s3;    
         ADR_BRA = 1'b1; ADR_BRB = 1'b0; N_BGA = 1'b1; N_BGB = 1'b0;      
       end   
     else     
       begin  // Transition grant to uP_B.        
         next = s2;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b0; N_BGA = 1'b1; N_BGB = 1'b0;           
       end
   end

  s6:
   begin
     if( ~N_TS )
       begin
         // uP_B has started the ADDRESS BUS tenure while grant is transitioning.
         // Go to S4 to store uP_B address bus qualifiers.
         next = s4;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b1; N_BGA = 1'b0; N_BGB = 1'b1;      
       end   
     else     
       begin  // Transition grant to uP_A.
         next = s2;    
         ADR_BRA = 1'b0; ADR_BRB = 1'b0; N_BGA = 1'b0; N_BGB = 1'b1;           
       end
   end

  endcase
end
endmodule 
