`timescale 1ns/1ns
module adrpipe (A, N_TS, TT, TSIZ, N_TBST,  
                MEMA, BURST, R_nW, BWE, BUS_REQ, BUSREQ_FLASH, 
                BANKSEL, DONE, DONE_FLASH, DATAXFR, 
                CLK, N_RESET, ENABLE_XFR ) ;

input [0:31] A;  	// 60x Address bus
input N_TS;             // 60x uP starting address bus tenure.
input [0:4] TT ;  	// 60x transfer type
input [0:2] TSIZ ;	// 60x transfer size
input N_TBST ;		// 60x transfer burst
input ENABLE_XFR;       // Move address from input to output stage.
input DONE ;        	// SRAM controller completed data bus tenure.
input DONE_FLASH ; 	// Flash controller completed data bus tenure.

input CLK;  		// 60x system clock
input N_RESET;      	// Active high Power on reset

output [0:17] MEMA ;    // Synchronized PPC address bus.
output BURST ;	        // Synchronized burst.
output R_nW ;           // Syncrhonized transfer size.
output [0: 7] BWE ;     // Synchronized transfer type bits.
output BUS_REQ ;	
output BUSREQ_FLASH ;
output DATAXFR ;        // Data bus tenure required based upon TT[0:4] inputs.
output [0:1] BANKSEL;   // SRAM memory bank ( A[9:10] )
// Module declarations:    
   
   reg [ 0:31 ] ADR ;
   reg [ 0: 4 ] TX_TYPE ;
   reg [ 0: 2 ] TX_SIZE ;
   reg [ 0:17 ] MEMA ;
   reg [ 0: 7 ] BWE; 
   reg [ 0: 1 ] BANKSEL ;

   reg TBST, BURST, BUS_REQ, BUSREQ_FLASH, R_nW; 

   wire  BUS_BURST, READ, SRAM, FLASH ;
   wire [0:7] BYTE_WE;
//******************************************************************       
// If transfer start is not asserted, hold prior address information.
// If transfer start is asserted, clock in new address information.
// If ENABLE_XFR move from input to output stage DFF.
// Multiplexer permits data to be held without gated clock design.
//******************************************************************
   wire [ 0: 31 ] MUX_ADDR ;
   wire [ 0:  4 ] MUX_TX_TYPE ;
   wire [ 0:  2 ] MUX_TX_SIZE ;      
   wire [ 0: 17 ] MUX_MEMA ;
   wire [ 0:  7 ] MUX_BWE ; 
   wire [ 0:  1 ] MUX_BANKSEL ;  

   wire MUX_BRST, MUX_BURST, MUX_RnW, MUX_SRAM, MUX_FLASH ;
  
   // input stage address multiplexor.
   assign MUX_ADDR    = N_TS ? ADR      : A ;
   assign MUX_TX_TYPE = N_TS ? TX_TYPE  : TT ;
   assign MUX_TX_SIZE = N_TS ? TX_SIZE  : TSIZ ; 
   assign MUX_BRST    = N_TS ? TBST     : ~N_TBST ;

   // output stage address multiplexor.
   assign MUX_MEMA    = ENABLE_XFR ?  ADR[ 11:28 ] : MEMA ;
   assign MUX_BURST   = ENABLE_XFR ?  BUS_BURST    : BURST;
   assign MUX_BWE     = ENABLE_XFR ?  BYTE_WE      : BWE ;
   assign MUX_RnW     = ENABLE_XFR ?  READ         : R_nW ;
   assign MUX_BANKSEL = ENABLE_XFR ?  ADR[ 9:10 ]  : BANKSEL ;

   // Use feedback from SRAMCTL and FLASH_IF to disable bus request
   // when the state machine has signaled bus access completed. Conversely
   // enable bus request to flow through if state machines are IDLE.
   assign MUX_SRAM    = ENABLE_XFR ?  SRAM         :  BUS_REQ & ~DONE;   
   assign MUX_FLASH   = ENABLE_XFR ?  FLASH        :  BUSREQ_FLASH & ~DONE_FLASH;     

// D Flip Flop (asynchronous Reset) to hold incoming address & qualifiers.
// Always store address transaction when N_TS is asserted. Data bus arbiter
// asserts address acknowledge N_AACK, so this data won't change until the
// decode logic has had a chance to determine if a data bus tenure is needed.

always @(posedge CLK or negedge N_RESET)
begin
   if (~N_RESET)  //asynchronous RESET
     begin
       // Point to 1st instruction address but don't enable transfers.
       // input stage.     
       ADR      = 32'hFFF0_0100;  // 1st instruction fetched after HRESET.
       TX_TYPE  =  5'b0_0000;     // Flush block address only.
       TX_SIZE  =  3'b010;        // Eight words. 
       TBST     =  1'b0;          // Burst disabled.
       // output stage. 
       MEMA     = 18'b00_0000_0000_0000_0000; // SRAM memory address.
       BURST    =  1'b0 ;         // disable 4 beat request.
       BWE      =  8'h00;         // disable byte write enables.
       R_nW     =  1'b1;          // Read not write.
       BANKSEL  =  2'b00;         // SRAM Bank 0
       BUS_REQ  =  1'b0; 
       BUSREQ_FLASH = 1'b0;
     end
   else          
     begin
     #2 // simulate DFF with output switching delays.
       // at CLK rising edge save old qualifiers or load next.
       // Enable transfer from input to output stage if ENABLE_XFR asserted.
       // input stage.
       ADR 	=  MUX_ADDR;
       TX_TYPE  =  MUX_TX_TYPE;
       TX_SIZE  =  MUX_TX_SIZE;
       TBST 	=  MUX_BRST;
       // output stage.
       MEMA     =  MUX_MEMA ;     // SRAM memory address.
       BURST    =  MUX_BURST ;    // disable 4 beat request.
       BWE      =  MUX_BWE ;       // disable byte write enables.
       R_nW     =  MUX_RnW ;       // Read not write.
       BANKSEL  =  MUX_BANKSEL;    // SRAM Bank 0:3 selected by address inputs.
       BUS_REQ  =  MUX_SRAM;       // Request SRAM memory access.
       BUSREQ_FLASH = MUX_FLASH;   // Request Flash memory access.
     end
end

wire [0:2] BYTE_ADDR  = ADR[29:31]; 

// Decode address bus qualifiers and return values to other modules.
// Must determine if ADDRESS ONLY bus tenure and ignore.
// Must determine Byte Write Enables based upon TX_SIZE bits.
// Decode address bus A[0:7] to determine if N_AACK should be asserted.

adr_dcd my_decd ( ADR[0:7], DATAXFR, SRAM, FLASH );
tx_type my_addr ( TX_TYPE, READ, DATAXFR );
tx_size my_size ( ~TBST, TX_SIZE, BYTE_ADDR, BUS_BURST, BYTE_WE );

endmodule

module tx_type ( TT, READ, DATAXFR );
 input [ 0:4 ] TT;
 output READ, DATAXFR;
 reg READ, DATAXFR;

// This module decodes the Transfer Type Qualifers to determine if a data bus
// cycle will be required following the address bus tenure. Address only bus
// cycles are not stored downstream and will not cause accesses to SRAM or flash.
always @( TT )
  begin
    casex (TT)
      5'b1_x100 : DATAXFR = 1'b1;
      5'b1_0010 : DATAXFR = 1'b1;
      5'b1_1x10 : DATAXFR = 1'b1;
      5'b0_xx10 : DATAXFR = 1'b1;
      5'b0_1011 : DATAXFR = 1'b1;

      default   : DATAXFR = 1'b0;
    endcase
    
    READ = TT[1];
  end
endmodule

module tx_size ( N_TBST, TSIZE, ADR, BUS_BURST, BYTE_WE );
 input N_TBST;
 input  [0:2] TSIZE;
 input  [0:2] ADR;

 output BUS_BURST;
 output [0:7] BYTE_WE;

 reg    BUS_BURST;
 reg    [0:7] BYTE_WE;

// This module decodes the Transfer Size Qualifers to determine 
// if a burst cycle is requested and for write cycles what the 
// byte enables should be. Note that BYTE_WE's get inverted in the
// sramctl.v module.

always @( N_TBST or TSIZE or ADR)
  begin
    casex ( {N_TBST, TSIZE, ADR} )
    
      7'b0_010_000 : BYTE_WE = 8'hFF; // 4 beat 8 byte burst   
      
      7'b1_000_000 : BYTE_WE = 8'hFF; // 1 beat 8 byte
      
      7'b1_001_000 : BYTE_WE = 8'h80; // 1 beat byte 0
      7'b1_001_001 : BYTE_WE = 8'h40; // 1 beat byte 1
      7'b1_001_010 : BYTE_WE = 8'h20; // 1 beat byte 2
      7'b1_001_011 : BYTE_WE = 8'h10; // 1 beat byte 3
      7'b1_001_100 : BYTE_WE = 8'h08; // 1 beat byte 4
      7'b1_001_101 : BYTE_WE = 8'h04; // 1 beat byte 5
      7'b1_001_110 : BYTE_WE = 8'h02; // 1 beat byte 6
      7'b1_001_111 : BYTE_WE = 8'h01; // 1 beat byte 7
      
      7'b1_010_000 : BYTE_WE = 8'hC0; // 1 beat bytes 0&1 
      7'b1_010_001 : BYTE_WE = 8'h60; // 1 beat bytes 1&2
      7'b1_010_010 : BYTE_WE = 8'h30; // 1 beat bytes 2&3
      7'b1_010_100 : BYTE_WE = 8'h0C; // 1 beat bytes 4&5 
      7'b1_010_101 : BYTE_WE = 8'h06; // 1 beat bytes 5&6
      7'b1_010_110 : BYTE_WE = 8'h03; // 1 beat bytes 6&7 
      
      // if 3B transfer crosses word boundry it gets
      // broken up into two smaller transfers. Only
      // odd bytes in word boundries work in 1 beat.     
      7'b1_011_000 : BYTE_WE = 8'hE0; // 3 beat bytes 0&1&2
      7'b1_011_001 : BYTE_WE = 8'h70; // 3 beat bytes 0&1&2
      7'b1_011_100 : BYTE_WE = 8'h0E; // 3 beat bytes 0&1&2
      7'b1_011_101 : BYTE_WE = 8'h07; // 3 beat bytes 0&1&2
      
      7'b1_100_000 : BYTE_WE = 8'hF0;
      7'b1_100_100 : BYTE_WE = 8'h0F;
      
      // don't believe BIU will do 5,6,7 bytes.
      7'b1_101 : BYTE_WE = 8'hFF; 
      7'b1_110 : BYTE_WE = 8'hFF; 
      7'b1_111 : BYTE_WE = 8'hFF;
      
      default: BYTE_WE = 8'hFF;

    endcase
    
    BUS_BURST = ~N_TBST ;
  end
endmodule


module adr_dcd ( A, DATAXFR, SRAM, FLASH );
 input [0:7] A;
 input DATAXFR;
 output SRAM, FLASH ;
 reg    SRAM, FLASH ;

always @( A or DATAXFR )
  begin
    // Just decode raw address for now. SRAM @0x40 and FLASH @0xFF.
    // Add transfer qualifiers later to ignore address only transactions.
    SRAM     = DATAXFR & ~A[0] &  A[1] & ~A[2] & ~A[3] & ~A[4] & ~A[5] &  ~A[6] & ~A[7] ;
    FLASH    = DATAXFR &  A[0] &  A[1] &  A[2] &  A[3] &  A[4] &  A[5] &   A[6] &  A[7] ;
  end

endmodule


