`timescale 1ns/1ns
module dataflow (N_TS, N_AACK, N_ARTRY, ADR_BRA, ADR_BRB, 
                 N_DBGA, N_DBGB, DATAXFR, CLK, N_RESET, 
                 N_DBB, ENABLE_XFR, BUS_REQ, BUSREQ_FLASH, DONE ) ;

input N_TS ;      // Transfer start.
output N_AACK ;   // Address acknowledge.
input N_ARTRY ;   // Address retry asserted by bus snooper.
input ADR_BRA ;   // Address bus grant A and N_TS asserted.
input ADR_BRB ;   // Address bus grant B and N_TS asserted.
output N_DBGA ;   // Data bus grant uP_A.
output N_DBGB ;   // Data bus grant uP_B. 
input DATAXFR ;   // Address decode indicates data bus required for transfer.
input CLK ;       // System clock (globally buffered).
input N_RESET ;   // System reset input.
input N_DBB ;     // Data bus busy (optional input).
output ENABLE_XFR; // Move address from input stage to output stage.

// If a bus cycle is in progress, don't assert N_AACK until bus cycle done.
input BUS_REQ, BUSREQ_FLASH, DONE;

// Module declarations:

reg N_AACK, N_DBGA, N_DBGB, ENABLE_XFR ;

// ONE HOT state machine state assignments
   parameter s0 = 6'b00_0001,  // idle
             s1 = 6'b00_0010,
             s2 = 6'b00_0100, 
             s3 = 6'b00_1000, 
             s4 = 6'b01_0000, 
             s5 = 6'b10_0000;
            /*  s6 = 11'b000_0100_0000, 
             s7 = 11'b000_1000_0000, 
             s8 = 11'b001_0000_0000, 
             s9 = 11'b010_0000_0000, 
             s10= 11'b100_0000_0000; 
             */
    
// current and next state registers
// increase size as the one hot state machine grows!! 
   reg [5:0] current, next;
// add your code here

always @(posedge CLK)
begin
#2 // delay 1 ns to emulate output switching delays.
  current = next;
  // start out in IDLE state after first rising edge of clk.
  next = s0; ENABLE_XFR = 1'b0;           
  N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;
  case(current)
  s0:
   begin
     if( ~N_TS & ~(BUS_REQ | BUSREQ_FLASH) ) 
       begin
         // Start of address bus tenure. Proceed to state 1.
         next = s1; ENABLE_XFR = 1'b0; 
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;        
       end     
     else
     if( ~N_TS & (BUS_REQ | BUSREQ_FLASH) & ADR_BRA )  
       begin
         // Start of address bus tenure. Data bus is busy.
         next = s4; ENABLE_XFR = 1'b0; 
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;        
       end     
     else
     if( ~N_TS & (BUS_REQ | BUSREQ_FLASH) & ADR_BRB )  
       begin
         // Start of address bus tenure. Data bus is busy.
         next = s5; ENABLE_XFR = 1'b0; 
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;        
       end 
     else
       begin
         // wait for a next address bus tenure start.
         next = s0; ENABLE_XFR = 1'b0; 
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;
       end       
   end

  s1:
   begin
   // Need information from address arbiter to determine which
   // data bus grant to issue. 
     if( DATAXFR & ~(BUS_REQ | BUSREQ_FLASH) )
       begin
         // Address decoder determined data bus transaction needed.
         // SRAM and Flash state machines are not busy processing a bus request.
         // Assert data bus grant to address bus master now.
         next = s2; ENABLE_XFR = 1'b0;  
         N_DBGA = ADR_BRB; N_DBGB = ADR_BRA; N_AACK = 1'b0;        
       end      
     else        
       begin
         // Address only transaction.
         // Wait for the next address bus tenure to start.
         next = s0; ENABLE_XFR = 1'b0; 
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b0;         
       end       
   end

  s2:
   begin
   // Deassert data bus grant and address acknowledge.
   // This is the clock that N_ARTRY gets asserted by bus snooper.
   //****************************************************************
   // Note: if address retry support is NOT required, the ENABLE_XFR
   // signal could be asserted in state:s1 and the memory latency
   // would be reduced by one clock period. Since ARTRY is supported
   // for this design the memory request queue will not be filled
   // until after the ARTRY clock phase. 
   // Note: NEED to make ARTRY_SUPPORT a parameter passed into the 
   // design which can modify the design behavior as part of the
   // logic synthesis process.  
   //****************************************************************
     if( ~N_ARTRY )
       begin
         // Bus Snooper has this location in cache.
         // Abort data transfer and return to IDLE.
         // Cancel pending data transfer in queue.
         next = s0; ENABLE_XFR = 1'b0; 
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;        
       end     
     else
       begin
         // move address qualifiers to output stage.
         // make room for next address qualifer at input stage.
         next = s3; ENABLE_XFR = 1'b1;
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;
       end
   end

  s3:
   begin
     if( ~N_TS & ADR_BRA)
       begin
         // Start of pipelined address bus tenure.
         // Can't assert N_AACK until N_TA asserted for data tenure.
         next = s4;  ENABLE_XFR = 1'b0; 
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;        
       end 
     else   
     if( ~N_TS & ADR_BRB)
       begin
         // Start of pipelined address bus tenure.
         // Can't assert N_AACK until N_TA asserted for data tenure.
         next = s5; ENABLE_XFR = 1'b0; 
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;        
       end            
     else
       begin
         // wait for address bus tenure to start.
         next = s3;  ENABLE_XFR = 1'b0;
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;
       end       
   end

  s4:
   begin
   // Waiting for data transfer to complete before asserting N_AACK and N_DBGA
   // data bus grant to issue. If the data bus is busy, wait until the cycle has
   // finished before asserting address acknowledge and data bus grant. If the
   // cycle is an address only transaction, return to IDLE state, without waiting
   // for data tenure to complete. Need to check busy at exit of s0 to inhibit
   // further transactions if busy.

     if( ~DATAXFR )
       begin
         // address only transaction. No data bus tenure required. Return to IDLE.
         next = s0; ENABLE_XFR = 1'b0;  
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b0;               
       end
     else
     if( DATAXFR & ~(BUS_REQ | BUSREQ_FLASH) )
       begin
         // Address decoder determined data bus transaction needed.
         // SRAM and Flash state machines are not busy processing a bus request.
         // Assert data bus grant to address bus master now.
         next = s2; ENABLE_XFR = 1'b0;  
         N_DBGA = 1'b0; N_DBGB = 1'b1; N_AACK = 1'b0;        
       end 
     else  
       begin
         // wait for data bus tenure to finish.
         next = s4;  
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;
       end       
   end

  s5:
   begin
   // Waiting for data transfer to complete before asserting N_AACK and N_DBGB
   // data bus grant. Data bus grant can only be asserted one clock prior to
   // start of processor driving bus. Must wait for bus request to deassert. If the
   // cycle is an address only transaction, return to IDLE state, without waiting
   // for data tenure to complete. Need to check busy at exit of s0 to inhibit
   // further transactions if busy.
   
     if( ~DATAXFR )
       begin
         // address only transaction. No data bus tenure required. Return to IDLE.
         next = s0; ENABLE_XFR = 1'b0;  
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b0;               
       end
     else
     if( DATAXFR &  ~(BUS_REQ | BUSREQ_FLASH) )
       begin
         // Address decoder determined data bus transaction needed.
         // SRAM and Flash state machines are not busy processing a bus request.
         // Assert data bus grant to address bus master now.
         next = s2; ENABLE_XFR = 1'b0;  
         N_DBGA = 1'b1; N_DBGB = 1'b0; N_AACK = 1'b0;        
       end 
     else  
       begin
         // wait for data bus tenure to finish.
         next = s5;  
         N_DBGA = 1'b1; N_DBGB = 1'b1; N_AACK = 1'b1;
       end       
   end

  endcase
end


endmodule 
