`timescale 1ns/1ns
module flash_if (CLK, N_RESET, R_nW, BUSREQ_FLASH,  DONE_FLASH,
		 N_FLASHCE, N_FLASHWE, N_FLASHOE, FLASH_ADR,
		 MEMD, FLASH_DB, FLASH_DBP, FLASH_TA ) ;

input CLK ;		 // Buffered system clock.
input N_RESET ;		 // Power on reset to initialize chip.
input R_nW ;		 // Read not write inferred from TT[1]. 
input BUSREQ_FLASH ;  	 // Bus Request for Flash memory access.
output DONE_FLASH ;      // Flash bus cycle completed.
output N_FLASHCE ;   	 // Flash chip enable.
output N_FLASHWE ;   	 // Flash write enable.
output N_FLASHOE ;   	 // Flash output enable.
output [0:1] FLASH_ADR;  // Flash address bits 0:1 (FLASH_ADR[2]=ADR[29])

input  [0:15] MEMD ;      // Flash data bus read input
output [0:63] FLASH_DB;   // Flash double word data output to uP
output [0:7]  FLASH_DBP;  // Odd byte lane parity generated for flash reads.
output FLASH_TA;          // Flash transfer acknowledge signal.

// PowerPC interface to flash memory. This design assumes a flash
// memory with 70ns access times. The specific device used in the
// design is an AMD29LV400B, organized 512Kx8 or 256Kx16 (Width of
// flash is selected by strapping BYTE# flash input low or high).

// Module declarations:
 wire  CNT_THREE;
 reg   DONE_FLASH, N_FLASHCE, N_FLASHOE, N_FLASHWE;
 reg   LOAD, CNT_UP;
 wire  [0:1] FLASH_ADR;

 parameter s0 = 12'b0000_0000_0001, s1  = 12'b0000_0000_0010,  s2 = 12'b0000_0000_0100,
           s3 = 12'b0000_0000_1000, s4  = 12'b0000_0001_0000,  s5 = 12'b0000_0010_0000,
           s6 = 12'b0000_0100_0000, s7  = 12'b0000_1000_0000,  s8 = 12'b0001_0000_0000,
           s9 = 12'b0010_0000_0000, s10 = 12'b0100_0000_0000, s11 = 12'b1000_0000_0000;   
                  
// Registers for one hot state machine encoding.
 reg [11:0] current, next;
 
 wire   RD_EN;
 
// Mealy state machine for control of boot flash.

always @(posedge CLK)
begin
  #2    // add delay to emulate DFF switching delay.
  current = next;
  next = s0;  DONE_FLASH = 1'b0;  LOAD= 1'b1; CNT_UP = 1'b0;         
  N_FLASHCE = 1'b1; N_FLASHOE = 1'b1; N_FLASHWE = 1'b1;  
  case(current)
  s0:
   begin
     if( R_nW & BUSREQ_FLASH )
       begin
         // Start access. Addresses are stable before asserting CE & OE!
         next = s1;  DONE_FLASH = 1'b0;  LOAD = 1'b1; CNT_UP= 1'b0; 
         N_FLASHCE = 1'b0; N_FLASHOE = 1'b0; N_FLASHWE = 1'b1;  
       end
     else
     if( ~R_nW & BUSREQ_FLASH )
       begin
         // Addresses must be stable before asserting CE & OE.
         next = s1;  DONE_FLASH = 1'b0;  LOAD = 1'b1; CNT_UP= 1'b0; 
         N_FLASHCE = 1'b0; N_FLASHOE = 1'b1; N_FLASHWE = 1'b1;  
       end
     else    
       begin  // wait for a valid bus request.
         next = s0;  DONE_FLASH = 1'b0; LOAD = 1'b1; CNT_UP= 1'b0;   
         N_FLASHCE = 1'b1; N_FLASHOE = 1'b1; N_FLASHWE = 1'b1;  
       end       
   end
   
  s1:    
   begin  // Delay 1 clk from CE to OE to meet Toeh after a write cycle.
     next = s2;  DONE_FLASH = 1'b0;  LOAD = 1'b0; CNT_UP= 1'b0;   
     N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE = R_nW; 
   end
      
  s2:
   begin
     next = s3;  DONE_FLASH = 1'b0;  LOAD = 1'b0; CNT_UP= 1'b0;       
     N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE = R_nW;
   end
   
  s3:
   begin
     next = s4;  DONE_FLASH = 1'b0;  LOAD = 1'b0; CNT_UP= 1'b0;      
     N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE =  R_nW;         
   end  
   
  s4:
   begin
     next = s5;  DONE_FLASH = 1'b0; LOAD = 1'b0; CNT_UP= 1'b0;          
     N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE = R_nW; 
   end          
   
  s5:
   begin
     next = s6;  DONE_FLASH = 1'b0;  LOAD = 1'b0; CNT_UP= 1'b0;      
     N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE = R_nW;  
   end   

  s6:
   begin
     next = s7;  DONE_FLASH = 1'b0;  LOAD = 1'b0; CNT_UP= 1'b0;      
     N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE = R_nW;  
   end 

  s7:
   begin
     if ( CNT_THREE )
       begin
         // Assert DONE for completion of flash access cycle.
         // Deasert chip enable, output enable, write enable.
         next = s8; DONE_FLASH = 1'b1; LOAD = 1'b0; CNT_UP= 1'b0;      
         N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE = R_nW;
       end
   else
     begin
       next = s8;  DONE_FLASH = 1'b0; LOAD = 1'b0; CNT_UP= 1'b0;      
       N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE = R_nW; 
     end
   end       
   
  s8:
   begin
   if ( CNT_THREE )
     begin
       // Assert DONE for completion of flash access cycle.
       // Deasert chip enable, output enable, write enable.
       next = s0; DONE_FLASH = 1'b0; LOAD = 1'b0; CNT_UP= 1'b0;      
       N_FLASHCE = 1'b1; N_FLASHOE = 1'b1; N_FLASHWE = 1'b1;
     end 
   else 
     begin
       // After first read cycle enable counter to count up now.
       // Need next address to assemble 64-bit word for read from flash.
       // Give the counter one clock cycle to increment before accessing flash again.
       next = s9; DONE_FLASH= 1'b0; LOAD = 1'b0; CNT_UP = 1'b1;
       N_FLASHCE = 1'b1; N_FLASHOE = 1'b1; N_FLASHWE = 1'b1;
     end
   end
   
  s9:
   begin
     // Flash output goes high-Z 25ns after N_FLASHOE deasserted.
     // Don't enable flash until output driver has turned off.
     // Wait 30ns (at 100MHz SYS_CLK) rate.
     // Delay needs adjusted as SYS_CLK frequency changes.
     next = s10; DONE_FLASH = 1'b0; LOAD = 1'b0; CNT_UP = 1'b0;
     N_FLASHCE = 1'b1; N_FLASHOE = 1'b1; N_FLASHWE = 1'b1;
   end    

  s10:
   begin
     next = s11; DONE_FLASH = 1'b0; LOAD = 1'b0; CNT_UP = 1'b0;
     N_FLASHCE = 1'b1; N_FLASHOE = 1'b1; N_FLASHWE = 1'b1;
   end

  s11:
   begin
     next = s1; DONE_FLASH = 1'b0; LOAD = 1'b0; CNT_UP = 1'b0;
     N_FLASHCE = 1'b0; N_FLASHOE = ~R_nW; N_FLASHWE = R_nW;
   end

  endcase
end

  assign RD_EN = current[7];

 flash_cntr my_cntr (.CLK        ( CLK ),
                     .N_RESET    ( N_RESET ),
                     .LOAD       ( LOAD ),
                     .CNT_UP     ( CNT_UP ),
                     .CNT_THREE  ( CNT_THREE ),
                     .FLASH_ADR  ( FLASH_ADR )
                    );

 data_assy my_data(.CLK        ( CLK ),
                   .N_RESET    ( N_RESET ), 
                   .D_IN       ( MEMD[0:15] ), 
                   .RD_EN      ( RD_EN ), 
                   .CNT        ( FLASH_ADR ), 
                   .FLASH_DB   ( FLASH_DB ), 
                   .FLASH_DBP  ( FLASH_DBP ),
                   .DONE_FLASH ( DONE_FLASH ),
                   .FLASH_TA   ( FLASH_TA )
                  );                    

endmodule


module flash_cntr (CLK, N_RESET, LOAD, CNT_UP, CNT_THREE, FLASH_ADR ) ;

  parameter width = 2;

  input  CLK, N_RESET, LOAD, CNT_UP ;
  output CNT_THREE ;
  output [0:1] FLASH_ADR ;
  
//*********************************************************************
// This module interfaces with the flash_if state machine and keeps 
// track of the number of times the state machine has accessed flash 
// during a bus request. The flash interface must form a 32 or 64-bit 
// word to present to the processor during boot time. The counter
// provides the least significant address bits to the byte/halfword 
// organized flash (depending upon external system implementation).
//
// The counter provides a mechanism to steer the byte or halfword
// flash data to the correct byte lanes for the PowerPC750 read:
//
// B0..B1..B2..B3..B4..B5..B6..B7  PowerPC byte format
// ..00......01......02......03..  CNT incrementing: 16-bit flash
// 00..01..02..03..04..05..06..07  CNT incrementing:  8-bit flash 
//
// Note: The MEMA outputs are aligned to a 8 byte boundary.
// For 16-bit flash: use A[29] as 3rd least significant address bit.
//*********************************************************************
// Module declarations:
  wire [width - 1 : 0] FLASH_ADR;
  reg  [width - 1 : 0] CNT;
  reg  CNT_THREE ;
  
 always@(posedge CLK or negedge N_RESET)
   begin
     if ( ~N_RESET )
       CNT = 2'h0;
     else
       if ( LOAD )
         CNT = 2'h0;
     else
       if ( CNT_UP )
         CNT = CNT + 1;
     else
       CNT = CNT;
   end

 assign FLASH_ADR = CNT ;

 always @(CNT)
   begin  
     CNT_THREE = CNT[1] & CNT[0];
   end
endmodule


module data_assy ( CLK, N_RESET, D_IN, RD_EN, CNT, FLASH_DB, FLASH_DBP, 
                   DONE_FLASH, FLASH_TA );

 input  CLK, N_RESET, RD_EN, DONE_FLASH;
 input  [0:15] D_IN;
 input  [0: 1] CNT;
 output [0:63] FLASH_DB;
 output [0: 7] FLASH_DBP;
 output FLASH_TA ;
  
//********************************************************************
// This module stores each sequential 16-bit read cycle in a holding
// register until the entire double word has been assembled. Once
// assembled Transfer Acknowledge (N_TA) can be asserted to the uP.
// Note: ODD PARITY must also be generated on the byte lanes!
//*********************************************************************
 wire [0:63] DIN;
 reg  [0:63] FLASH_DB;
 reg  [0: 7] FLASH_DBP;
 reg  FLASH_TA;
 
 // Note: counter bits are big endian format. Bit 0 = most significant bit.
 assign DIN[ 0:15] = ( (RD_EN & ~CNT[0] & ~CNT[1] )) ? D_IN[ 0:15] : FLASH_DB[ 0:15];
 assign DIN[16:31] = ( (RD_EN & ~CNT[0] &  CNT[1] )) ? D_IN[ 0:15] : FLASH_DB[16:31];
 assign DIN[32:47] = ( (RD_EN &  CNT[0] & ~CNT[1] )) ? D_IN[ 0:15] : FLASH_DB[32:47];
 assign DIN[48:63] = ( (RD_EN &  CNT[0] &  CNT[1] )) ? D_IN[ 0:15] : FLASH_DB[48:63];
 
 
 always @(posedge CLK or negedge N_RESET)
 begin
   if ( ~N_RESET )
     begin
       FLASH_DB = 64'h0000_0000_0000_0000; 
       FLASH_TA = 1'b0;
     end 
   else
     begin 
       # 2 // emulate output switching delays.
       FLASH_DB = DIN;
       FLASH_TA = DONE_FLASH;
     end  
 end
 //******************************************************************** 
 // Generate odd parity on flash byte lanes.
 // SRAM stores odd parity with data. May need a datapath multiplexor
 // to select parity generation(flash) or flowthrough (SRAM).
 //********************************************************************
 always @ ( FLASH_DB )
   begin // odd parity generation.
     FLASH_DBP[0:7] = { ~( ^FLASH_DB[ 0: 7] ),  ~( ^FLASH_DB[ 8:15] ),   
                         ~( ^FLASH_DB[16:23] ),  ~( ^FLASH_DB[24:31] ),
                         ~( ^FLASH_DB[32:39] ),  ~( ^FLASH_DB[40:47] ),
                         ~( ^FLASH_DB[48:55] ),  ~( ^FLASH_DB[56:63] )
                       } ;
   end     
endmodule
