/****************************************************************************************
*
*    File Name:  MT58L256L36F.V
*      Version:  1.3
*         Date:  March 8th, 1999
*        Model:  BUS Functional
*    Simulator:  Model Technology (PC version 4.7i PE)
*
* Dependencies:  None
*
*       Author:  Son P. Huynh
*        Email:  sphuynh@micron.com
*        Phone:  (208) 368-3825
*      Company:  Micron Technology, Inc.
*       Part #:  MT58L256L36F (256K x 36)
*
*  Description:  This is Micron's Synburst SRAM (Flow-through)
*
*   Limitation:
*
*   Disclaimer:  THESE DESIGNS ARE PROVIDED "AS IS" WITH NO WARRANTY 
*                WHATSOEVER AND MICRON SPECIFICALLY DISCLAIMS ANY 
*                IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR
*                A PARTICULAR PURPOSE, OR AGAINST INFRINGEMENT.
*
*                Copyright (c) 1997 Micron Semiconductor Products, Inc.
*                All rights researved
*
* Rev  Author          Phone         Date        Changes
* ---  ----------------------------  ----------  ---------------------------------------
* 1.3  Son P. Huynh    208-368-3825  03/08/1999  Improve model functionality
*      Micron Technology, Inc.
*
****************************************************************************************/

// DO NOT CHANGE THE TIMESCALE
// MAKE SURE YOUR SIMULATOR USE "PS" RESOLUTION
`timescale 1ns / 100ps

module mt58l256l36f (Dq, Addr, Mode, Adv_n, Clk, Adsc_n, Adsp_n, Bwa_n, Bwb_n, Bwc_n, Bwd_n, Bwe_n, Gw_n, Ce_n, Ce2, Ce2_n, Oe_n, Zz);

    parameter  addr_bits =     18;         //  18 bits
    parameter  data_bits =     36;         //  36 bits
    parameter  mem_sizes = 262143;         // 256 K
    parameter  reg_delay =    0.1;         // 100 ps
    parameter  out_delay =    0.1;         // 100 ps
    parameter  tKQHZ     =    5.0;         // 5.0 ns

    inout     [(data_bits - 1) : 0] Dq;   // Data IO
    input     [(addr_bits - 1) : 0] Addr; // Address
    input     Mode;      // Burst Mode
    input     Adv_n;     // Synchronous Address Advance
    input     Clk;       // Clock
    input     Adsc_n;    // Synchronous Address Status Controller
    input     Adsp_n;    // Synchronous Address Status Processor
    input     Bwa_n;     // Synchronous Byte Write Enables
    input     Bwb_n;     // Synchronous Byte Write Enables
    input     Bwc_n;     // Synchronous Byte Write Enables
    input     Bwd_n;     // Synchronous Byte Write Enables
    input     Bwe_n;     // Byte Write Enable
    input     Gw_n;      // Global Write
    input     Ce_n;      // Synchronous Chip Enable
    input     Ce2;       // Synchronous Chip Enable
    input     Ce2_n;     // Synchronous Chip Enable
    input     Oe_n;      // Output Enable
    input     Zz;        // Snooze Mode

    reg [((data_bits / 4) - 1) : 0] bank0 [0 : mem_sizes];  // Memory Bank 0
    reg [((data_bits / 4) - 1) : 0] bank1 [0 : mem_sizes];  // Memory Bank 1
    reg [((data_bits / 4) - 1) : 0] bank2 [0 : mem_sizes];  // Memory Bank 2
    reg [((data_bits / 4) - 1) : 0] bank3 [0 : mem_sizes];  // Memory Bank 3

    reg [(data_bits - 1) : 0] din;          // Data In
    reg [(data_bits - 1) : 0] dout;         // Data Out
    reg [(addr_bits - 1) : 0] addr_reg_in;  // Address Register In
    reg [1 : 0] bcount;                     // 2-bit Burst Counter

    reg  ce_reg;
    reg  bwa_reg;
    reg  bwb_reg;
    reg  bwc_reg;
    reg  bwd_reg;
    reg  Sys_clk;
    
    wire  ce      = (~Ce_n & Ce2 & ~Ce2_n);
    wire  bwa_n   = ((Bwa_n | Bwe_n) & Gw_n | (~Ce_n & ~Adsp_n));
    wire  bwb_n   = ((Bwb_n | Bwe_n) & Gw_n | (~Ce_n & ~Adsp_n));
    wire  bwc_n   = ((Bwc_n | Bwe_n) & Gw_n | (~Ce_n & ~Adsp_n));
    wire  bwd_n   = ((Bwd_n | Bwe_n) & Gw_n | (~Ce_n & ~Adsp_n));
    wire  clr     = (~Adsc_n | (~Adsp_n & ~Ce_n));

    wire  [(addr_bits - 1) : 0] addr_reg_out;  // Address Register Out
    wire  baddr1;  // Burst Address 1
    wire  baddr0;  // Burst Address 0

    // Initialize
    initial begin
        ce_reg = 1'b0;
        Sys_clk = 1'b0;
        $timeformat (-9, 1, " ns", 10); // Format time unit
    end

    // System Clock Decode
    always begin
        @ (posedge Clk) begin
            Sys_clk = ~Zz;
        end
        @ (negedge Clk) begin
            Sys_clk = 1'b0;
        end
    end

    always @ (posedge Sys_clk) begin
        // Address Register
        if (clr) addr_reg_in <= Addr;

        // Binary Counter and Logic
        if      ( Mode  &  clr) bcount <= 0;               // Interleaved Burst
        else if (~Mode  &  clr) bcount <= Addr [1 : 0];    // Linear Burst
        else if (~Adv_n & ~clr) bcount <= (bcount + 1);    // Advance Counter

        // Byte Write Register    
        bwa_reg <= ~bwa_n;
        bwb_reg <= ~bwb_n;
        bwc_reg <= ~bwc_n;
        bwd_reg <= ~bwd_n;

        // Enable Register
        if (clr) ce_reg <= ce;
    end

    // Burst Address Decode
    assign addr_reg_out = {addr_reg_in [(addr_bits - 1) : 2], baddr1, baddr0};
    assign baddr1 = Mode ? (bcount [1] ^ addr_reg_in [1]) : bcount [1];
    assign baddr0 = Mode ? (bcount [0] ^ addr_reg_in [0]) : bcount [0];

    // Write Driver
    always @ (posedge Clk) begin
        #reg_delay;
        if (ce_reg & bwa_reg) begin
            din [8 : 0] <= Dq [8 : 0];
            bank0 [addr_reg_out] <= Dq [8 : 0];
        end
        if (ce_reg & bwb_reg) begin
            din [17 : 9] <= Dq [15 : 9];
            bank1 [addr_reg_out] <= Dq [17 : 9];
        end       
        if (ce_reg & bwc_reg) begin
            din [26 : 18] <= Dq [26 : 18];
            bank2 [addr_reg_out] <= Dq [26 : 18];
        end
        if (ce_reg & bwd_reg) begin
            din [35 : 27] <= Dq [35 : 27];
            bank3 [addr_reg_out] <= Dq [35 : 27];
        end
    
    end

    // Output Registers
    always @ (posedge Clk) begin
        #out_delay;
        if (ce_reg & ~(bwa_reg | bwb_reg | bwc_reg | bwd_reg)) begin
            dout [ 8 :  0] <= bank0 [addr_reg_out];
            dout [17 :  9] <= bank1 [addr_reg_out];
            dout [26 : 18] <= bank2 [addr_reg_out];
            dout [35 : 27] <= bank3 [addr_reg_out];
        end else begin
            dout [35 :  0] <= 36'bz;
        end
    end
    
    // Output Buffers
    assign #tKQHZ Dq =  ~Zz & ~Oe_n ? dout : 36'bz;

    // Timing Check (10ns access / 15ns cycle)
    // Please obtain the latest datasheet from our Web site
    //      http://www.micron.com/mti
    specify
        specparam   tKC     = 10,       // Clock  (was 15 ST)      - Clock cycle time
                    tKH     =  4,       //                Clock HIGH time
                    tKL     =  4,       //                Clock LOW time
                    tAS     =  2.5,     // Setup Times  - Address
                    tADSS   =  2.5,     //                Address Status (ADSC#, ADSP#)
                    tAAS    =  2.5,     //                Address Advance (ADV#)
                    tWS     =  2.5,     //                Byte Write Enables (BWa#-BWd#, GW#, BWE#)
                    tDS     =  2.5,     //                Data-in
                    tCES    =  2.5,     //                Chip Enable (CE#)
                    tAH     =  0.5,     // Hold Times   - Address                                  
                    tADSH   =  0.5,     //                Address Status (ADSC#, ADSP#)            
                    tAAH    =  0.5,     //                Address Advance (ADV#)                   
                    tWH     =  0.5,     //                Byte Write Enables (BWa#-BWd#, GW#, BWE#)
                    tDH     =  0.5,     //                Data-in                                  
                    tCEH    =  0.5;     //                Chip Enable (CE#)

        $width      (negedge Clk, tKL);
        $width      (posedge Clk, tKH);
        $period     (negedge Clk, tKC);
        $period     (posedge Clk, tKC);
        $setuphold  (posedge Clk, Adsp_n, tADSS, tADSH);
        $setuphold  (posedge Clk, Adsc_n, tADSS, tADSH);
        $setuphold  (posedge Clk, Addr,   tAS,   tAH);
        $setuphold  (posedge Clk, Bwa_n,  tWS,   tWH);
        $setuphold  (posedge Clk, Bwb_n,  tWS,   tWH);
        $setuphold  (posedge Clk, Bwc_n,  tWS,   tWH);
        $setuphold  (posedge Clk, Bwd_n,  tWS,   tWH);
        $setuphold  (posedge Clk, Bwe_n,  tWS,   tWH);
        $setuphold  (posedge Clk, Gw_n,   tWS,   tWH);
        $setuphold  (posedge Clk, Ce_n,   tCES,  tCEH);
        $setuphold  (posedge Clk, Ce2,    tCES,  tCEH);
        $setuphold  (posedge Clk, Ce2_n,  tCES,  tCEH);
        $setuphold  (posedge Clk, Adv_n,  tAAS,  tAAH);
    endspecify                        

endmodule

