`timescale 1ns/1ns
module sramctl (N_RESET, RnW, BUSREQ, BURST, BWE, CLK, N_CE, N_OE, N_WE,
		N_ADV, N_ADSC, DONE, SRAM_TA ) ;

input N_RESET ;     // Power on reset to initialize chip logic.
input RnW ;         // Read not write derived from TT[4:0] bus qualifiers.
input BUSREQ ;      // Active bus request from pipeline controller.
input BURST ;       // 60X burst (fixed: @4/8 transfers of 64/32 bits).
input [0:7] BWE ;   // Byte writes derived from TBST & TSIZ[0:2].
input CLK ;         // buffered SYS_CLK.

output N_CE ;       // SSRAM chip enable.
output N_OE ;       // SSRAM output enable.
output [0:7] N_WE ; // SSRAM byte write enables.
output N_ADV ;      // SSRAM advance for read bursting.
output N_ADSC ;     // SSRAM address strobe.
output DONE;        // Signal that SRAM bus cycle has completed. 
output SRAM_TA;     // Signal that SRAM data transfer beat has completed. 

//*********************************************************************
// Mealy state machine with synchronous outputs.
// Controls 1 bank of synchronous SRAM memory (256Kx72 bits).
// Not implemented: PPC603e configured as a 32-bit bus device. 
//
// To support a cache line read burst that doesn't start on a cache line
// boundary, the MODE input of the SSRAM must a logic 0 to select linear 
// burst addressing. Linear address mode works as follows:
//
// If the External Address ADR[1:0] is:
// ADR[1:0] = 0x00, burst internal address = 0x01..0x10..0x11.
// ADR[1:0] = 0x01, burst internal address = 0x10..0x11..0x00.
// ADR[1:0] = 0x10, burst internal address = 0x11..0x00..0x01.
// ADR[1:0] = 0x11, burst internal address = 0x00..0x01..0x10.
//
//*********************************************************************

// Module declarations:
// ONE HOT state machine state assignments
   parameter s0 = 11'b000_0000_0001,  // idle
             s1 = 11'b000_0000_0010,  // read states
             s2 = 11'b000_0000_0100, 
             s3 = 11'b000_0000_1000, 
             s4 = 11'b000_0001_0000, 
             s5 = 11'b000_0010_0000,
             s6 = 11'b000_0100_0000, 
             s7 = 11'b000_1000_0000, 
             s8 = 11'b001_0000_0000, 
             s9 = 11'b010_0000_0000, 
             s10= 11'b100_0000_0000;	//write states
    
// current and next state registers
// increase size as the binary state machine grows!! 
   reg [10:0] current, next;
  
// SRAM interface registers                        
   reg N_CE, N_OE, N_ADV, N_ADSC, DONE, SRAM_TA;
   reg [0:7] N_WE;

// Start state machine at positive edge of system clock (CLK).

always @(posedge CLK)
begin
#2 // delay 2 ns to emulate DFF output switching delays.
  current = next;
  // start out in IDLE state after first rising edge of clk.
  next = s0;  DONE = 1'b0;  SRAM_TA = 1'b0;        
  N_CE = 1'b1; N_OE = 1'b1; N_ADV = 1'b1; N_ADSC = 1'b1; N_WE = 8'hFF;
  
  case(current)
  s0:
   begin
     if(BUSREQ & RnW)
       begin
         // Start a read cycle. Pipeline controller issues valid BUSREQ.
         // Assert N_OE at same time to get valid data on next clock.
         next = s1; DONE = 1'b0;  SRAM_TA = 1'b0; 
         N_CE = 1'b0; N_OE = 1'b0; N_ADV = 1'b1; N_ADSC = 1'b0; N_WE = 8'hFF;        
       end
     else
     if(BUSREQ & ~RnW & ~BURST)
       begin
         // Start a single beat write cycle.
         next = s6; DONE = 1'b1; SRAM_TA = 1'b1;   
         N_CE = 1'b0; N_OE = 1'b1; N_ADV = 1'b1; N_ADSC = 1'b0; N_WE = ~BWE;        
       end
     else     
     if(BUSREQ & ~RnW & BURST)
       begin
         // Start a four beat write cycle. All byte lanes asserted.
         next = s7; DONE = 1'b0;  SRAM_TA = 1'b0; 
         N_CE = 1'b1; N_OE = 1'b1; N_ADV = 1'b1; N_ADSC = 1'b1; N_WE = 8'hFF;        
       end      
     else
       begin
         // wait for a valid bus request.
         next = s0; DONE = 1'b0;  SRAM_TA = 1'b0;
         N_CE = 1'b1; N_OE = 1'b1; N_ADV = 1'b1; N_ADSC = 1'b1; N_WE = 8'hFF;         
       end       
   end
   
  s1:
   begin
     if(BURST)          
       begin
         // Start a four beat burst read ( 8 bytes per beat).
         next = s2; DONE = 1'b0; SRAM_TA = 1'b1;
         // External address selects starting address for read burst.
         // After initial address is latched, use SSRAM internal address counter.    
         N_CE = 1'b1; N_OE = 1'b0; N_ADV = 1'b0; N_ADSC = 1'b1; N_WE = 8'hFF;        
       end
     else
       begin
         // Start a single beat read ( 8 bytes ).
         // External address selects which location to read.
         next = s5; DONE = 1'b1; SRAM_TA = 1'b1;     
         N_CE = 1'b1; N_OE = 1'b0; N_ADV = 1'b1; N_ADSC = 1'b1; N_WE = 8'hFF;               
       end
   end
   
  s2:
   begin
     // continue 4 beat read cycle.
     next = s3; DONE = 1'b0; SRAM_TA = 1'b1;
     // deassert chip enable, assert output enable.
     // assert advance signal to increment internal SSRAM address counter.           
     N_CE = 1'b1; N_OE = 1'b0; N_ADV = 1'b0; N_ADSC = 1'b1; N_WE = 8'hFF; 
   end
   
  s3:
   begin
     // continue 4 beat read cycle.
     next = s4; DONE = 1'b0; SRAM_TA = 1'b1;
     // increment internal SSRAM address counter.           
     N_CE = 1'b1; N_OE = 1'b0; N_ADV = 1'b0; N_ADSC = 1'b1; N_WE = 8'hFF;
   end  
   
  s4:
   begin
     // continue 4 beat read cycle.
     next = s5; DONE = 1'b1; SRAM_TA = 1'b1;         
     // assert advance signal to increment internal SSRAM address counter.
     N_CE = 1'b1; N_OE = 1'b0; N_ADV = 1'b1; N_ADSC = 1'b1; N_WE = 8'hFF;
     
   end    
   
  s5:
   begin
     // continue 4 beat read cycle or deassert OE for single beat read.
     next = s0; DONE = 1'b0;  SRAM_TA = 1'b0;       
     // deassert advance signal to complete 4 beat burst.
     N_CE = 1'b1; N_OE = 1'b1; N_ADV= 1'b1; N_ADSC = 1'b1; N_WE = 8'hFF;
     // SRAM read cycle has completed. Ready for next bus access.     
   end     


// WRITE STATE MACHINE STATEMENTS FOLLOW:
  
  s6:
   begin
     // Single beat write cycle complete. Deassert CE, OE & return to IDLE.
     next = s0; DONE = 1'b0; SRAM_TA = 1'b0;        
     N_CE = 1'b1; N_OE = 1'b1; N_ADV= 1'b1; N_ADSC = 1'b1; N_WE = 8'hFF;
     // SRAM write cycle has completed. Return to IDLE and wait for next request.     
   end     

  s7:
   begin
     // Second beat of four beat write cycle, deassert OE for write cycles.
     next = s8; DONE = 1'b0; SRAM_TA = 1'b0;       
     N_CE = 1'b0; N_OE = 1'b1; N_ADV= 1'b1; N_ADSC = 1'b0; N_WE = 8'h00;     
   end

  s8:
   begin
     // Third beat of four beat read cycle.
     next = s9; DONE = 1'b0; SRAM_TA = 1'b0;     
     N_CE = 1'b1; N_OE = 1'b1; N_ADV= 1'b0; N_ADSC = 1'b1; N_WE = 8'h00;    
   end

  s9:
   begin
     // Fourth beat of four beat read cycle.
     // SRAM_TA is used to deassert N_TA on next clock edge.
     next = s10; DONE = 1'b0; SRAM_TA = 1'b1;       
     N_CE = 1'b1; N_OE = 1'b1; N_ADV= 1'b0; N_ADSC = 1'b1; N_WE = 8'h00;      
   end

  s10:
   begin
     // Assert DONE for completion of four beat write cycle.
     next = s6; DONE = 1'b1;  SRAM_TA = 1'b0;     
     N_CE = 1'b1; N_OE = 1'b1; N_ADV= 1'b0; N_ADSC = 1'b1; N_WE = 8'h00;
     // SRAM write cycle has completed. Ready for next bus access.      
   end

     
  endcase
end

endmodule 
