/*---------------------------------------------------------------------------
 --                                                                        --
 --     This confidential and proprietary software may be used only        --
 --     as authorized by a licensing agreement from Synopsys Inc.          --
 --     In the event of publication, the following notice is applicable:   --
 --                                                                        --
 --                    (C) COPYRIGHT 1998   SYNOPSYS INC.                  --
 --                          ALL RIGHTS RESERVED                           --
 --                                                                        --
 --        The entire notice above must be reproduced on all authorized    --
 --        copies.                                                         --
 --                                                                        --
 -- FILE: mpc750_l2_tst.v                                                 --                                                                        --
 -- ABSTRACT: Example verilog testbench for mpc750_l2_fx model.          --
 --                                                                        --
 -- RUNTIME: 25300 ns                                                     --
 --                                                                        --
 --------------------------------------------------------------------------*/

/***************************************************************/  
/* Top level BFM for Steve Trynosky's PPC750 Applications Note */
/***************************************************************/   
 
   `timescale 1ns / 100ps

   `define time_scale_multiplier 10
   `define MPC740ModelId_1 "mpcInst_1"
   `define MPC750_L2ModelId_1 "mpc750_l2Inst_1"
   `define clk_period 1 * `time_scale_multiplier

module top (
    );

   `include "mpc740_pkg.inc"
   `include "mpc750_l2_pkg.inc"
         
   `define true 1
   `define false 0
   `define TRUE 1
   `define FALSE 0      

   reg Sysclk_i;
   reg Hreset_i;
   
/*---------------------------------
 * The top level signals for mpc740
 *---------------------------------*/
   wire   Br_o;
   reg    Abb_ior;
   tri1   Abb_io;
   reg    Ts_ior;
   wire   Ts_io;
   reg [(`MPC740_A_WIDTH - 1):0] A_ior;
   wire [(`MPC740_A_WIDTH - 1):0] A_io;
   reg [(`MPC740_AP_WIDTH - 1):0] Ap_ior;
   wire [(`MPC740_AP_WIDTH - 1):0] Ap_io;
   reg [(`MPC740_TT_WIDTH - 1):0]  Tt_ior;
   wire [(`MPC740_TT_WIDTH - 1):0] Tt_io;
   reg                             Tbst_ior;
   wire                            Tbst_io;
   reg [(`MPC740_TSIZ_WIDTH - 1):0] Tsiz_ior;
   wire [(`MPC740_TSIZ_WIDTH - 1):0] Tsiz_io;
   reg                               Gbl_ior;
   wire                              Gbl_io;
   reg                               Wt_ior;
   wire                              Wt_io;
   reg                               Ci_ior;
   wire                              Ci_io;
   reg                               Artry_ior;
   tri1                              Artry_io;
   reg                               Dbwo_i;
   reg                               Dbb_ior;
   tri1                              Dbb_io;
   reg [(`MPC740_DL_WIDTH - 1):0]     Dl_ior;
   wire [(`MPC740_DL_WIDTH - 1):0]    Dl_io;
   reg [(`MPC740_DH_WIDTH - 1):0]     Dh_ior;
   wire [(`MPC740_DH_WIDTH - 1):0]    Dh_io;
   reg [(`MPC740_DP_WIDTH - 1):0]    Dp_ior;
   wire [(`MPC740_DP_WIDTH - 1):0]   Dp_io;
   reg                               Dbdis_ior;
   wire                              Dbdis_io;

   reg   Drtry_i;

   reg  Tea_i, Int_i, Smi_i, Mcp_i, Sreset_i, Ckstp_in_i;
   wire Ckstp_out_o, Rsrv_o, Qreq_o, Clk_out_o;
   reg  Tben_i, Tlbisync_i, Qack_i;
   reg [(`MPC740_PLL_CFG_WIDTH - 1):0] Pll_cfg_i;

   reg [(`MPC740_JTAG_COP_WIDTH - 1):0] Jtag_cop_ior;
   wire [(`MPC740_JTAG_COP_WIDTH - 1):0] Jtag_cop_io;
   reg [(`MPC740_TEST_WIDTH - 1):0]      Test_ior;
   wire [(`MPC740_TEST_WIDTH - 1):0]     Test_io;

/*----------------------------------------------
 * The variable/parameter declaration for mpc740
 *-----------------------------------------------*/
   integer status, mpcInst_1, cmd_tag, valid, priority;
   
   reg [63:0] read_rslt;

   initial begin
      priority = 0;
   end

   reg intr_signal;
   reg [14:0] transfer_attr;
   initial begin
      transfer_attr = 15'b000000000000000;
   end
   
   parameter read_ext_ctrl    = 7'b0000001;
   parameter read_normal      = 7'b0000010;
   parameter read_itnt_mod    = 7'b0000100;
   parameter read_atomic      = 7'b0001000;
   parameter read_itnt_mod_at = 7'b0010000;
   parameter read_no_cache    = 7'b0100000;
   
   parameter write_ext_ctrl = 7'b0000001;
   parameter write_flush    = 7'b0000010;
   parameter write_kill     = 7'b0000100;
   parameter write_flush_at = 7'b0001000;

   parameter clean_blk   = 9'b000000001;
   parameter flush_blk   = 9'b000000010;
   parameter sync        = 9'b000000100;
   parameter kill_blk    = 9'b000001000;
   parameter eieio_addr  = 9'b000010000;
   parameter tlb_invalib = 9'b000100000;
   parameter lwarx       = 9'b001000000;
   parameter tlbsync     = 9'b010000000;
   parameter icbi        = 9'b100000000;

   parameter trans_1_byte   = 5'b00001;
   parameter trans_2_bytes  = 5'b00010;
   parameter trans_3_bytes  = 5'b00011; 
   parameter trans_4_bytes  = 5'b00100;
   parameter trans_8_bytes  = 5'b01000;
   parameter trans_32_bytes = 5'b10000;

   parameter cache_0_address = 32'hffffffff;
   parameter cache_0_status  = 2'b00;
   parameter cache_0_data0   = 64'h2323124554213232;
   parameter cache_0_data1   = 64'h2342110031002100;
   parameter cache_0_data2   = 64'h43329129abcdbcef;
   parameter cache_0_data3   = 64'h2343912121192921;

   parameter cache_1_address = 32'h00000000;
   parameter cache_1_status  = 2'b00;

   time tclk;
   initial begin
      tclk = 100 *`time_scale_multiplier;
   end
   
   reg [18:0] c_xfer ;
   initial begin
      c_xfer = 19'h0;
   end



/*------------------------------------
 * The top level signals for mpc750_l2
 *------------------------------------*/
   wire [(`MPC750_L2_L2ADDR_WIDTH - 1):0] L2addr_o;
   reg [(`MPC750_L2_L2DATA_WIDTH - 1):0]  L2data_ior;
   wire [(`MPC750_L2_L2DATA_WIDTH - 1):0] L2data_io;
   reg [(`MPC750_L2_L2DP_WIDTH - 1):0]    L2dp_ior;
   wire [(`MPC750_L2_L2DP_WIDTH - 1):0]   L2dp_io;

   wire L2ce_o, L2we_o, L2clk_outa_o, L2clk_outb_o;
   wire L2sync_out_o, L2zz_o;
   reg  L2sync_in_i, Mode_o_tr2ts;

/*--------------------------------------------------
 * The variable/parameter declaration for mpc750_l2
 *--------------------------------------------------*/
   integer     mpc750_l2Inst_1, status_mpc750_l2;
   integer     cmdtag [20:0] ;
   
   parameter   l2pe = 1'b1;
   parameter   l2pe_n = 1'b0;  
   parameter   one_mb = 2'b11;
   parameter   one_512kb = 2'b01;
   parameter   one_256kb = 2'b10;  
   parameter   ssram_flow = 2'b00;
   parameter   ssram_pipe = 2'b01;
   parameter   ssram_late = 2'b11;
   
   parameter   l2df = 1'b1;
   parameter   l2df_n = 1'b1;
   
   reg [31:0]  l2cr_reg ;
   initial begin
      l2cr_reg = 32'b0 ;
     end
   
   reg [31:0] reg_val;
   reg pin_val, start_f, drive_f, end_of_cycle ;
   reg enable_post_data, enable_post_data_z; 
   reg enable_post_data_rmw, enable_post_data_rmw_z;      
   integer    read_burst_counter [0:2];
   integer    write_burst_counter [0:2];
   integer    rmw_counter [0:2];
   reg [63:0] Data0 [0:2];
   reg [63:0] Data1 [0:2];

   
   parameter  READ   = 3'b000;
   parameter  WRITE  = 3'b001;
   parameter  IDLE   = 3'b010;
   parameter  NULL_N = 3'b011;
   parameter  ABORT  = 3'b100;
   parameter  READ_MODIFY_WRITE = 3'b101;
   reg [2:0]  Cmd  [0:2];
     

   //************************************   
   // SRAM Memory I/O connected to u3.
   //************************************
   wire [0:63] MEMD_io;
   wire [0:17] MEMA_io;
   wire [0: 7] MEMDP_io, N_MEMBEW_io;
   
   
   /*-----------------------------------------------------------------
    * Instantiate the mpc740 model and connect to the testbench.
    *-----------------------------------------------------------------*/
   mpc740   u1  (
                 .BR  		( Br_o ),
                 .BG  		( N_BGA ), 
                 .ABB  		( Abb_io ),  // NOT IMPLEMENTED in 750CX package!
                 .TS  		( Ts_io ),
                 .A  		( A_io ),
                 .AP  		( Ap_io ),
                 .TT  		( Tt_io ),
                 .TBST  	( Tbst_io ),
                 .TSIZ  	( Tsiz_io ),
                 .GBL  		( Gbl_io ),
                 .WT  		( Wt_io ),
                 .CI  		( Ci_io ),
                 .AACK  	( N_AACK ),
                 .ARTRY 	( Artry_io ),
                 .DBG  		( N_DBGA ), 
                 .DBWO  	( Dbwo_i ),
                 .DBB 		( Dbb_io ),  // NOT IMPLEMENTED in 750CX package!
                 .DL  		( Dl_io ),
                 .DH  		( Dh_io ),
                 .DP  		( Dp_io ),
                 .DBDIS 	( Dbdis_io ), // NOT IMPLEMENTED in 750CX package!
                 .TA  		( N_TA ), 
                 .DRTRY 	( Drtry_i ),  // NOT IMPLEMENTED in 750CX package!
                 .TEA  		( Tea_i ),
                 .INT  		( Int_i ),
                 .SMI  		( Smi_i ),
                 .MCP  		( Mcp_i ),
                 .SRESET	  	( Sreset_i ),
                 .HRESET	  	( Hreset_i ),
                 .CKSTP_IN   	( Ckstp_in_i ),
                 .CKSTP_OUT  	( Ckstp_out_o ),
                 .RSRV  	( Rsrv_o ),
                 .TBEN  	( Tben_i ),
                 .TLBISYNC  	( Tlbisync_i ),
                 .QREQ 		( Qreq_o ),
                 .QACK  	( Qack_i ),
                 .SYSCLK  	( Sysclk_i ),
                 .PLL_CFG  	( Pll_cfg_i ),
                 .CLK_OUT  	( Clk_out_o ),
                 .JTAG_COP  	( Jtag_cop_io ),
                 .TEST  	( Test_io )
                 ); 

   defparam u1.FlexModelId     = `MPC740ModelId_1; 
   //********************************************************************
   // Add timing mode parameters for the MPC740 processor.
   // The default timing mode is OFF!
   // See file D:\Synopsys\cdrom\pcnt\models\mpc740_fx\mpc740_fx010007.td 
   // for processor selections of IBM and Motorola processors 
   // (different SYS_CLK and timings)
   //********************************************************************
   defparam u1.FlexTimingMode  = `FLEX_TIMING_MODE_ON;   
   defparam u1.TimingVersion   = "IBM25PPC740LGB_500";    
   defparam u1.DelayRange      = "TYP";    // "MAX" is default

/*-----------------------------------------------------------------------------
 * instantiate the model mpc750_l2, and connect the model port to the testbench
 *-----------------------------------------------------------------------------*/
   mpc750_l2   u2  (
                    .L2ADDR 	(L2addr_o),
                    .L2DATA  	(L2data_io),
                    .L2DP  	(L2dp_io),
                    .L2CE  	(L2ce_o),
                    .L2WE  	(L2we_o),
                    .L2CLK_OUTA (L2clk_outa_o),
                    .L2CLK_OUTB (L2clk_outb_o),
                    .L2SYNC_OUT (L2sync_out_o),
                    .L2SYNC_IN  	(L2sync_in_i),
                    .L2ZZ  	(L2zz_o),
                    .HRESET  	(Hreset_i),
                    .CLKIN  	(Sysclk_i)
                    ); 
   
   defparam  u2.FlexModelId     = `MPC750_L2ModelId_1;

//-----------------------------------------------------------------------------
//   Instantiate the Xilinx FPGA design, top_60x.v, and connect to testbench.
//   Testbench has little endian bit assignments for A_io, Dh_io, Dl_io, Dp_io,
//   Tt_io, and Tsiz_io. Fix this through connections to u3 design here.
// -----------------------------------------------------------------------------


  top_60x  u3  (
                .N_RESET		( Hreset_i ),  // testbench drives low to reset.
                .SYS_CLK		( Sysclk_i  ), // 100 MHz system clock
                .A		( 
                	                  /* swap A_io bit order */
                                  { A_io[0],  A_io[1],  A_io[2],  A_io[3], 
                                    A_io[4],  A_io[5],  A_io[6],  A_io[7],
                                    A_io[8],  A_io[9],  A_io[10], A_io[11],
                                    A_io[12], A_io[13], A_io[14], A_io[15],
                                    A_io[16], A_io[17], A_io[18], A_io[19], 
                                    A_io[20], A_io[21], A_io[22], A_io[23],
                                    A_io[24], A_io[25], A_io[26], A_io[27],
                                    A_io[28], A_io[29], A_io[30], A_io[31] 
                                   }
                                  ), 

                .N_TS		( Ts_io ),
                .TT		( { Tt_io[0], Tt_io[1], Tt_io[2], Tt_io[3], Tt_io[4] } ),                                                                                     
                .N_TBST		( Tbst_io ),
                .TSIZ		( { Tsiz_io[0],  Tsiz_io[1],  Tsiz_io[2] } ),                 
                .N_AACK		( N_AACK ),  
                .N_ARTRY		( Artry_io  ), 
		.N_BRA		( Br_o  ),   // Address bus grant.
		.N_BRB		( 1'b1  ),   // Deassert 2nd uP address bus request.
		.N_BGA		( N_BGA ),    
		.N_BGB		(   ),       // connect second uP to this address grant.
		.N_DBGA		( N_DBGA ), 
		.N_DBGB		(   ),       // connect second uP to this data grant.
		.DH		( 
                                  { // swap Dh_io bit order
                                    Dh_io[0],  Dh_io[1],  Dh_io[2],  Dh_io[3], 
                                    Dh_io[4],  Dh_io[5],  Dh_io[6],  Dh_io[7],
                                    Dh_io[8],  Dh_io[9],  Dh_io[10], Dh_io[11],
                                    Dh_io[12], Dh_io[13], Dh_io[14], Dh_io[15],
                                    Dh_io[16], Dh_io[17], Dh_io[18], Dh_io[19], 
                                    Dh_io[20], Dh_io[21], Dh_io[22], Dh_io[23],
                                    Dh_io[24], Dh_io[25], Dh_io[26], Dh_io[27],
                                    Dh_io[28], Dh_io[29], Dh_io[30], Dh_io[31]
                                   } ), 		 
		.DL		(                                    
                                  { // Swap Dl_io bit order
                                    Dl_io[0],  Dl_io[1],  Dl_io[2],  Dl_io[3], 
                                    Dl_io[4],  Dl_io[5],  Dl_io[6],  Dl_io[7],
                                    Dl_io[8],  Dl_io[9],  Dl_io[10], Dl_io[11],
                                    Dl_io[12], Dl_io[13], Dl_io[14], Dl_io[15],
                                    Dl_io[16], Dl_io[17], Dl_io[18], Dl_io[19], 
                                    Dl_io[20], Dl_io[21], Dl_io[22], Dl_io[23],
                                    Dl_io[24], Dl_io[25], Dl_io[26], Dl_io[27],
                                    Dl_io[28], Dl_io[29], Dl_io[30], Dl_io[31]                                 
                                   } ), 		 
		.DP		( 				  
                                  { // Swap Dp_io bit order
                                    Dp_io[0],  Dp_io[1],  Dp_io[2],  Dp_io[3],
                                    Dp_io[4],  Dp_io[5],  Dp_io[6],  Dp_io[7]
                                   } ), 		 

		.N_TA		( N_TA ),         
		.N_DRTRY	(   ),           // Not implemented in design.
		.N_TEA		(   ),           // Not implemented in design.
		.MEMA		( MEMA_io  ),    
		.MEMD		( MEMD_io ), 
		.MEMDP		( MEMDP_io ), 
		.N_MEMBEW	( N_MEMBEW_io  ), 
		.N_MEMCE	( N_MEMCE  ), 
		.N_MEMOE	( N_MEMOE  ), 
		.N_MEMADV	( N_MEMADV  ), 
		.N_MEMADSC	( N_MEMADSC ), 
		.BANKSEL	(   ),  // Only one bank of memory in testbench.
	        .N_FLASHCE	(   ),  // No external flash instantiated.
	        .N_FLASHOE	(   ), 
	        .N_FLASHWE	(   ), 
	        .FLASH_ADR	(   )
	        ) ;

//*********************************************************
// Instantiate two SSRAM flow-through memory models.
// Memory is organized as 256Kx72 bits.
//*********************************************************

mt58l256l36f u4 ( 
                  .Dq       ( { 
                                // Micron byte lanes in litte endian (31:0) format
                                // and parity bit grouped with data bit.
                                MEMDP_io[3], MEMD_io[24:31], 
                                MEMDP_io[2], MEMD_io[16:23],
                                MEMDP_io[1], MEMD_io[ 8:15],
                                MEMDP_io[0], MEMD_io[ 0: 7]
                              } ),
                  .Addr     ( MEMA_io ),
                  .Mode     ( 1'b0 ),       // linear address mode
                  .Adv_n    ( N_MEMADV ), 
                  .Clk      ( Sysclk_i ),
                  .Adsc_n   ( N_MEMADSC ), 
                  .Adsp_n   ( 1'b1 ), 
                  .Bwa_n    ( N_MEMBEW_io[0] ), // Bwa associated with MEMD_io[0:7]
                  .Bwb_n    ( N_MEMBEW_io[1] ), 
                  .Bwc_n    ( N_MEMBEW_io[2] ), 
                  .Bwd_n    ( N_MEMBEW_io[3] ), 
                  .Bwe_n    ( 1'b0 ),       // Enable byte writes
                  .Gw_n     ( 1'b1 ),       // Global write function not used.
                  .Ce_n     ( N_MEMCE ), 
                  .Ce2      (~N_MEMCE ), 
                  .Ce2_n    ( N_MEMCE ), 
                  .Oe_n     ( N_MEMOE ), 
                  .Zz       ( 1'b0 )        // Disable snooze mode.
                 );

mt58l256l36f u5 ( 
                  .Dq       ( { 
                                // Micron byte lanes in litte endian (31:0) format
                                // and parity bit grouped with data bit.
                                MEMDP_io[7], MEMD_io[56:63],
                                MEMDP_io[6], MEMD_io[48:55],
                                MEMDP_io[5], MEMD_io[40:47],
                                MEMDP_io[4], MEMD_io[32:39]
                              } ),                 
                  .Addr     ( MEMA_io ),
                  .Mode     ( 1'b0 ),       // linear address mode
                  .Adv_n    ( N_MEMADV ), 
                  .Clk      ( Sysclk_i ),
                  .Adsc_n   ( N_MEMADSC ), 
                  .Adsp_n   ( 1'b1 ), 
                  .Bwa_n    ( N_MEMBEW_io[4] ), 
                  .Bwb_n    ( N_MEMBEW_io[5] ), 
                  .Bwc_n    ( N_MEMBEW_io[6] ), 
                  .Bwd_n    ( N_MEMBEW_io[7] ), 
                  .Bwe_n    ( 1'b0 ),       // Enable byte writes
                  .Gw_n     ( 1'b1 ),       // Global write function not used.
                  .Ce_n     ( N_MEMCE ), 
                  .Ce2      (~N_MEMCE ), 
                  .Ce2_n    ( N_MEMCE ), 
                  .Oe_n     ( N_MEMOE ), 
                  .Zz       ( 1'b0 )        // Disable snooze mode.
                 );


//*******************************************************************
// Add pullups to make data bus "valid" when bus is tristated or not 
// being driven by the processor model. For byte, halfword and word 
// commands, the model only drives those bits that are valid. 
//  
// Using a synchronous data path requires data valid at all times!
// Weak pullups make bus a logical "1" as the data goes through the FPGA. 
//
// Tristated buses are bad for radiated emissions!
// FPGA bus keeper circuits eliminate pullups on PWB assembly!
//*******************************************************************
  pullups  u100 ( Dh_io );
  pullups  u101 ( Dl_io ); 
  pullups8 u102 ( Dp_io ) ;
  pullups  u103 ( MEMD_io[ 0:31]); 
  pullups  u104 ( MEMD_io[32:63]); 
  pullups8 u105 ( MEMDP_io ) ;

/*  -------------------------------------------------------------------------
    CLOCK GENERATOR : This always block generates the system clock with the |
                      time period `clock_period.                            |
    ------------------------------------------------------------------------- */

   always begin: sysclk_gen
      Sysclk_i <= #(0) 1'b1;
      #(`clk_period/2);
      Sysclk_i <= #(0) 1'b0;
      #(`clk_period/2);
   end 


//----------------------------------------------------------------------------------
//--
//-- Procedure  :   idle_cycle
//--
//-- Description:    The test bench waits for the specified number of clock periods
//--
//----------------------------------------------------------------------------------
   task idle_cycle ;
      input  cycles ; integer cycles;
      begin
         # (cycles * `clk_period);
      end
   endtask

/*  ---------------------------------------------------------------------
   | PIN INITIALIZATION : This initial begin block is used to initialize |
   |                     the pins from the test at zero simulation time. |
    --------------------------------------------------------------------- */
   initial

      begin : REG_INITIALIZATION
         // mpc740
         A_ior     <= 32'hzzzzzzzz ;
         Dl_ior    <= 32'hz;
         Dh_ior    <= 32'hz; 
         Abb_ior   <=  1'bz;
         Ts_ior    <=  1'bz;
         Ap_ior    <=  4'hz;
         Artry_ior <=  1'b1 ;
         Drtry_i   <=  1'b1 ;
         Dbb_ior   <=  1'bz;
         Dp_ior    <=  8'hzz ;
         Tt_ior    <=  5'bzzzzz ;
         Tsiz_ior  <=  3'bz ;
         Tbst_ior  <=  1'bz;
         Gbl_ior   <=  1'bz;
         Tea_i     <=  1'b1 ;
         Smi_i     <=  #(0) 1'b1 ;
         Wt_ior    <=  1'bz ;
         Ci_ior    <=  1'bz ;
         Mcp_i     <=  #(0) 1'b1 ;
         Int_i     <=  #(0) 1'b1 ;
         //mpc750_l2
         L2data_ior <= 64'hz;
         L2dp_ior   <=  8'hz;
   end // REG_INITIALIZATION

/*  -----------------------------
   | NOTES for the mpc740 model |
    -----------------------------

   The address tenure for each command (except idle and bus request commands) 
   starts with address arbitration and then to address transfer and finally 
   the address tenure termination. Data tenure for each command (except 
   address only write; idle; and bus request commands) starts with data bus 
   arbitration and then data transfer and finally data tenure termination.
 */


/*  -----------------
   |  Xfer_attr      |
    -----------------

 ------------------------------------------------------------------------------
|xfer_attr| Description                                                        |
|bits     |                                                                    |
|------------------------------------------------------------------------------|
|3-0      |Enables or disables driving/checking of odd address parity.       |
|         |If kept 4'b0000, odd parity will be driven/checked.                 |
|11-4     |Enables or disables driving/checking of odd data parity.          |
|12       |Write Through if true than Wt_o = 1'b1 else Wt_o = 1'b0             |
|14-13    |Transfer code Tc_o[0] = xfer_attr[13] and Tc_o[1] = xfer_attr[14]   |
|15       |Cache Inihibt if true than Ci_o = 1'b1 else Ci_o = 1'b0             |
|16       |Global if true than Gbl_o = 1'b1 else Gbl_o = 1'b0                  |
|18-17    |Cache Set entry Cse_o[0] = xfer_attr[17] and Cse_o[1] = xfer_attr[18|
 ------------------------------------------------------------------------------

 The following declaration is made in the mpc740_fx_tst.inc file
     reg [18:0] c_xfer ;
 The reg c_xfer can be used to pass transfer attributes.


 -------------------------
| Read comand description |
 -------------------------
    General read_req command is
    mpc740_read_req(id,address,xfer_attr_read,xfer_attr,wait,status)

           Xfer_attr_read

 ------------------------------------------------------------------------------
| Read        |    Description                                | Xfer_attr_read |
| type        |                                               | Bits           |
|------------------------------------------------------------------------------|
| ext_ctrl    | External control work read (single)           |  0             |
| normal      | Read (single or burst)                        |  1             |
| itnt_mod    | Read-with-intent-to-modify (burst)            |  2             |
| atomic      | Read-atomic (single or burst)                 |  3             |
| itnt-mod-at | Read-with-intent-to-modify-atomic (burst)     |  4             |
| no_cache    | Read-with-no-intent-to cache (single or burst)|  5             |
 ------------------------------------------------------------------------------
 following declaration is made in mpc740_fx_tst.inc file

   parameter ext_ctrl    = 7'b0000001 ;
   parameter normal      = 7'b0000010 ;
   parameter itnt_mod    = 7'b0000100 ;
   parameter atomic      = 7'b0001000 ;
   parameter itnt_mod_at = 7'b0010000 ;
   parameter no_cache    = 7'b0100000 ;


     Transfer size setting in xfer_attr_read
 ---------------------------------------------------------------
| Transfer size                              |   Xfer_attr_read |
|                                            |   Bits           |
|---------------------------------------------------------------|
| Byte (1 bytes single)                      |     7            |
| Half_word (2 bytes single)                 |     8            |
| Word (4 bytes single)                      |     9            |
| Double_word (8 bytes single)               |     10           |
| Quad_double_word (4 beats burst transfer)  |     11           |
 ---------------------------------------------------------------

  following declarations are made in mpc740_fx_tst.inc file

   parameter byte        = 5'b00001 ;
   parameter half_word   = 5'b00010 ;
   parameter word        = 5'b00100 ;
   parameter d_word      = 5'b01000 ;
   parameter q_d_word    = 5'b10000 ;

NOTE : The address is given in the form A0...A31 Thus, the A0..A3 nibble is of value 1,
       A4..A7 nibble is of value 2 and so on. Similarly for the data words.

 --------------------------
| Write command description |
 ---------------------------
    General read_req command is
       mpc740_write(id,address,xfer_attr_write,xfer_attr,data0,data1,data2,data3,wait,status);

           Xfer_attr_write

 ---------------------------------------------------------------------------
| Write      |    Description                           |  Xfer_attr_write  |
| type       |                                          |  Bits             |
|---------------------------------------------------------------------------|
| ext_ctrl   |   External control word write (single)   |    0              |
| flush      |   Write-with-flush (single or burst)     |    1              |
| kill       |   Write-with-kill (single or burst)      |    2              |
| flush_at   |   Write-with-flush-atomic (single)       |    3              |
 ---------------------------------------------------------------------------
   Following declerations are made in mpc740_fx_test.inc

   parameter flush       = 7'b0000010 ;
   parameter kill        = 7'b0000100 ;
   parameter flush_at    = 7'b0001000 ;




          Transfer size setting in xfer_attr_read
 ---------------------------------------------------------------
| Transfer size                              |   Xfer_attr_read |
|                                            |   Bits           |
|---------------------------------------------------------------|
| Byte (1 bytes single)                      |     7            |
| Half_word (2 bytes single)                 |     8            |
| Word (4 bytes single)                      |     9            |
| Double_word (8 bytes single)               |     10           |
| Quad_double_word (4 beats burst transfer)  |     11           |
 ---------------------------------------------------------------

  following declarations are made in mpc740_fx_tst.inc file

   parameter byte        = 5'b00001 ;
   parameter half_word   = 5'b00010 ;
   parameter word        = 5'b00100 ;
   parameter d_word      = 5'b01000 ;
   parameter q_d_word    = 5'b10000 ;

NOTE : The address is given in the form A0...A31 Thus, the A0..A3 nibble is of value 1,
       A4..A7 nibble is of value 2 and so on. Similarly for the data words.

 ------------------------
|  Address only commands |
 ------------------------
    General address only command is
      mpc740_write_addr_only( Id, addr,watype,xfer_attr, wait, status);

      watype : write address only type

 ----------------------------------------------
|watype     |   Desctiption            |watype |
|           |                          |bits   |
|----------------------------------------------|
|clean_blk  |   Clean block            |0      |
|flush_blk  |   Flush block            |1      |
|sync       |   Sync                   |2      |
|kill_blk   |   kill block             |3      |
|eieio      |   eieio                  |4      |
|tlb_invalid|   TLB invalidate         |5      |
|lwarx      |   lwarx reservation set  |6      |
|tlbsync    |   TLB sync               |7      |
|icbi       |   Icbi                   |8      |
 ----------------------------------------------
  Following declarations are made in mpc740_fx_test.inc

   parameter clean_blk_c    = 'b000000001;
   parameter flush_blk_c    = 'b000000010;
   parameter sync_c         = 'b000000100;
   parameter kill_blk_c     = 'b000001000;
   parameter eieio_c        = 'b000010000;
   parameter tlb_invalid_c  = 'b000100000;
   parameter lwarx_c        = 'b001000000;
   parameter tlbsync_c      = 'b010000000;
   parameter icbi_c         = 'b100000000;

NOTE : The address is given in the form A0...A31 Thus, the A0..A3 nibble is of value 1,
       A4..A7 nibble is of value 2 and so on. Similarly for the data words.
*/


/* --------------------------------------------------------------------------- 
  | CMD_AND_STIMULI_STREAM_MPC740 :                                           |
  | ------------------------------                                            |
  |             This block is used to pass commands to command core           |
  |             which are executed by the model mpc740_fx.                    |
  |             Each Block of commands in CMD_AND_STIMULI_STREAM_MPC740       |
  |             process is followed by a procedure call made which generates  |
  |             stimuli.                                                      |
   ---------------------------------------------------------------------------*/
   
   initial
      begin : CMD_AND_STIMULI_STREAM_MPC740
         $display("Start of PowerPC750 Interface to Xilinx FPGA Test Bench");
         #0 Hreset_i <= 1'b0;
         
         #(5 * `clk_period);
         /* Reset the device
          Assert the Hreset_i  pin for 4 clocks */
         Hreset_i <= 1'b0;
         idle_cycle (4);
         Hreset_i <= 1'b1;
         Abb_ior  <= #(0)  1'bz;
         Dbb_ior  <= #(0) 1'bz;
         Drtry_i  <= #(0)  1'b1;
         Ts_ior   <= #(0) 1'bz;
         Ap_ior   <= #(0) 'bz;

         
   //  Prior to using any of the FlexModel commands, you need to get a model
   //  handle. Using the FlexModelId name that you passed to the instatiated
   //  model. Assign the flexHandle to the mpcInst_1 signal for use in other
   //  processes. 
   flex_get_inst_handle(`MPC740ModelId_1, mpcInst_1, status);
      if (status != 1) begin
         $display("Error: Could not get handle for model: %s", `MPC740ModelId_1);
         $finish;
      end

   //  Upon restarting the simulation, clear all the queues in the cmd core.
   flex_clear_queue(mpcInst_1, 0, status);

   //  Notifies the command core that we are using intr_signal for
   //  interrupt handling purposes. This signal will toggle when any of the
   //  interrupt is asserted and reported by the model to the command core. 
   flex_define_intr_signal(mpcInst_1, "intr_signal", status);


   //  Enable messages from the model: warnings & command posting messages.
   //  mpc740_set_msg_level(mpcInst_1,`FLEX_ALL_MSGS,status);
   mpc740_set_msg_level(mpcInst_1,`FLEX_NO_MSGS,status);
         
   //  Set mode register:
   //  mode_reg[3]  = 0 64 bit data mode true
   //  mode_reg[2]  = 1 artry mode true
   //  mode_reg[1]  = 1 clock mode fast
   //  mode_reg[0]  = 0 drtry mode false.
   mpc740_set_reg(mpcInst_1,`MPC740_MODE_SET_REG,4'b0110,status);
          
   //  Set HID0 register:
   //  hid0_reg[3] = 1 data parity check enabled.
   //  hid0_reg[2] = 1 address parity check enabled.
   //  hid0_reg[1] = 1 data and address  parity generation enabled.
   //  hid0_reg[0] = 1 MCP interrupt enbled.
   mpc740_set_reg(mpcInst_1,`MPC740_HID0_REG,32'hffffffff,status); 

   //  Set MSR[ME] 
   mpc740_set_reg(mpcInst_1,`MPC740_MSR19_REG, 1'b1,status);   


         /* Cmd_6
          Setting the cache line 0

           Details of Cache_line registers
           ------------------------------------------------------------------
           |Value       | Bits   | Description                               |
           |-----------------------------------------------------------------|
           |Address Tag | 31-0   | Address for checking status in cache line |
           |            |        |                                           |
           |MEI status  | 33, 32 | MEI status set according to MEI protocol  |
           |            |        | 00 Modified                               |
           |            |        | 01 Exclusive                              |
           |            |        | 10 Invalid                                |
           |            |        |                                           |
           |Data Words  | 289-34 | Data for write back operation             |
           |            |        |     data0 is for bits 97-34               |
           |            |        |     data0 is for bits 161-98              |
           |            |        |     data0 is for bits 225-162             |
           |            |        |     data0 is for bits 289-226             |
            -----------------------------------------------------------------  */

   mpc740_set_reg(mpcInst_1,`MPC740_CACHE_LINE_0_REG,{cache_0_data3,cache_0_data2,cache_0_data1,cache_0_data0,cache_0_status,cache_0_address},status);
   mpc740_set_reg(mpcInst_1,`MPC740_CACHE_LINE_1_REG,{cache_0_data3,cache_0_data2,cache_0_data1,cache_0_data0,cache_0_status,cache_1_address},status);
   mpc740_set_reg(mpcInst_1,`MPC740_WRITE_BACK_REG,c_xfer,status);
   mpc740_pin_req(mpcInst_1,`MPC740_SMI_PIN,`FLEX_WAIT_F,status);
   mpc740_pin_rslt(mpcInst_1,`MPC740_SMI_PIN,read_rslt[0],status);
      if (read_rslt[0] === 1'b1) begin
          $display("Test Passed : The value of pin SMI = %0b  status = %0d",read_rslt[0],status);
      end else begin
          $display("Test Failed : The value of pin SMI = %0b  status = %0d",read_rslt[0],status);
      end
         

   mpc740_set_pin(mpcInst_1,`MPC740_QREQ_PIN,1'b0,status);

//**************************************************************************
// First external bus cycle! SRAM is memory mapped @ 4000_0000-4001_FFFF.
// This testbench demonstrates support for the 4 write commands, 6 read commands,
// and 9 address only PowerPC command types issued during the address start phase
// of the 60X bus protocol. 
//**************************************************************************


//*********************************************************
// Test 1: write_flush command: (single or burst mode)
// 32-byte write, four beats of 8 bytes each.
//*********************************************************
   mpc740_write(mpcInst_1,'h40000000,{trans_32_bytes,write_flush},transfer_attr,
      64'h0001020304050607,64'h08090a0b0c0d0e0f,64'h1011121314151617,64'h18191a1b1c1d1e1f,`FLEX_WAIT_F,status);

//*********************************************************
// Test 1: read_itnt_mod_at command: (BURST MODE ONLY!)
// 32-byte read, four beats of 8 bytes each.
//*********************************************************
   mpc740_read_req(mpcInst_1,'h40000000,{trans_32_bytes,read_itnt_mod_at},transfer_attr,`FLEX_WAIT_F,status);  

/*
   =======================================================================================
   The mpc740_read_rslt command returns maximum 64-bit data for 64-bit data bus mode).
   The data read during Quad_double_word read (4 beats burst transfer) is 256-bits. 
   Four mpc740_read_rslt commands are used to check valid write read of 256-bits of data.
   =======================================================================================
 */
    mpc740_read_rslt(mpcInst_1,32'h40000000,0,read_rslt,status);
      if (read_rslt ===  64'h0001020304050607) begin
          $display("Test Passed : The read_normal result for address 32'4000_0000 1st data beat DW0 = %0h",read_rslt); 
      end else begin 
          $display("Test Failed : The read_normal result for address 32'4000_0000 1st data beat DW0 = %0h",read_rslt);     
      end  
    mpc740_read_rslt(mpcInst_1,32'h40000000,0,read_rslt,status);
      if (read_rslt === 64'h08090a0b0c0d0e0f) begin
          $display("Test Passed : The read_normal result for address 32'4000_0000 2nd data beat DW1 = %0h",read_rslt); 
      end else begin 
          $display("Test Failed : The read_normal result for address 32'4000_0000 2nd data beat DW1 = %0h",read_rslt);    
      end
    mpc740_read_rslt(mpcInst_1,32'h40000000,0,read_rslt,status);
      if (read_rslt === 64'h1011121314151617) begin
          $display("Test Passed : The read_normal result for address 32'4000_0000 3rd data beat DW2 = %0h",read_rslt); 
      end else begin 
          $display("Test Failed : The read_normal result for address 32'4000_0000 3rd data beat DW2 = %0h",read_rslt);    
      end  
    mpc740_read_rslt(mpcInst_1,32'h40000000,0,read_rslt,status);
      if (read_rslt === 64'h18191a1b1c1d1e1f) begin
          $display("Test Passed : The read_normal result for address 32'4000_0000 4th data beat DW3 = %0h",read_rslt);  
      end else begin 
          $display("Test Failed : The read_normal result for address 32'4000_0000 4th data beat DW3 = %0h",read_rslt);  
      end

//*********************************************************
// Test 2: write_flush_at command (single beat only!)
// Eight byte single beat write. Fill another cache line.
// Write 8-bit address offset to data(address):
//*********************************************************
   mpc740_write(mpcInst_1,'h40000020,{trans_8_bytes,write_flush_at} ,transfer_attr,
      64'h2021222324252627,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000028,{trans_8_bytes,write_flush_at},transfer_attr,
      64'h28292a2b2c2d2e2f,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000030,{trans_8_bytes,write_flush_at},transfer_attr,
      64'h3031323334353637,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000038,{trans_8_bytes,write_flush_at},transfer_attr,
      64'h38393a3b3c3d3e3f,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);

//*********************************************************
// Test 2: read_itnt_mod_at command: (Single of burst mode)
// Verify 8 byte single beat writes by reading back data and 
// comparing read_rslt. Verify write_flush_at command operation.
//*********************************************************
   mpc740_read_req(mpcInst_1,'h40000020,{trans_32_bytes,read_itnt_mod_at},transfer_attr,`FLEX_WAIT_F,status);       
   mpc740_read_rslt(mpcInst_1,32'h40000020,0,read_rslt,status);
      if (read_rslt ===  64'h2021222324252627 ) begin
          $display("Test Passed : The read_itnt_mod result for address 32'4000_0020 1st data beat DW0 = %0h",read_rslt);
      end else begin 
          $display("Test Failed : The read_itnt_mod result for address 32'4000_0020 1st data beat DW0 = %0h",read_rslt);
      end           
   mpc740_read_rslt(mpcInst_1,32'h40000020,0,read_rslt,status);
      if (read_rslt === 64'h28292a2b2c2d2e2f ) begin
          $display("Test Passed : The read_itnt_mod result for address 32'4000_0020 2nd data beat DW1 = %0h",read_rslt);
      end else begin 
          $display("Test Failed : The read_itnt_mod result for address 32'4000_0020 2nd data beat DW1 = %0h",read_rslt);
      end    
   mpc740_read_rslt(mpcInst_1,32'h40000020,0,read_rslt,status);
      if (read_rslt === 64'h3031323334353637 ) begin
          $display("Test Passed : The read_itnt_mod result for address 32'4000_0020 3rd data beat DW2 = %0h",read_rslt);
      end else begin 
          $display("Test Failed : The read_itnt_mod result for address 32'4000_0020 3rd data beat DW2 = %0h",read_rslt);
      end          
   mpc740_read_rslt(mpcInst_1,32'h40000020,0,read_rslt,status);
      if (read_rslt === 64'h38393a3b3c3d3e3f ) begin
          $display("Test Passed : The read_itnt_mod result for address 32'4000_0020 4th data beat DW3 = %0h",read_rslt);
      end else begin 
          $display("Test Failed : The read_itnt_mod result for address 32'4000_0020 4th data beat DW3 = %0h",read_rslt);
      end  

//*********************************************************
// Test 3: write_kill command: (single or burst mode)                       
// Single byte write transfer test. Verify the byte write 
// capabilities of the reference design. Byte writes only
// work in single beat mode, NOT BURST!
//*********************************************************
   mpc740_write(mpcInst_1,'h40000040,{trans_1_byte,write_kill},transfer_attr,
      64'h40,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000041,{trans_1_byte,write_kill},transfer_attr,
      64'h41,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000042,{trans_1_byte,write_kill},transfer_attr,
      64'h42,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000043,{trans_1_byte,write_kill},transfer_attr,
      64'h43,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000044,{trans_1_byte,write_kill},transfer_attr,
      64'h44,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000045,{trans_1_byte,write_kill},transfer_attr,
      64'h45,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000046,{trans_1_byte,write_kill},transfer_attr,
      64'h46,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000047,{trans_1_byte,write_kill},transfer_attr,
      64'h47,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);         

//*********************************************************
// Test 3: read_no_cache command: (Single or burst mode)
// Verify byte writes to SSRAM by reading back the data and
// comparing read_rslt to write values.
//*********************************************************
   mpc740_read_req(mpcInst_1,'h40000040,{trans_8_bytes,read_no_cache},transfer_attr,`FLEX_WAIT_F,status);       
   mpc740_read_rslt(mpcInst_1,32'h40000040,0,read_rslt,status);
     if (read_rslt ===  64'h4041424344454647 ) begin
         $display("Test Passed : The read_no_cache result for address 32'4000_0040 1st data beat DW0 = %0h",read_rslt);
     end else begin 
          $display("Test Failed : The read_no_cache result for address 32'4000_0040 1st data beat DW0 = %0h",read_rslt);
     end           

//*********************************************************
// Test 4: write_flush command: (Single or burst mode)
// Halfword write transfer test. Byte writes only
// work in single beat mode, NOT BURST!
//*********************************************************
   mpc740_write(mpcInst_1,'h40000048,{trans_2_bytes,write_flush},transfer_attr,
      64'h4849,64'h11,64'h22,64'h33,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h4000004a,{trans_2_bytes,write_flush},transfer_attr,
      64'h4a4b,64'h11,64'h22,64'h33,`FLEX_WAIT_F,status);                  
   mpc740_write(mpcInst_1,'h4000004c,{trans_2_bytes,write_flush},transfer_attr,
      64'h4c4d,64'h11,64'h22,64'h33,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h4000004e,{trans_2_bytes,write_flush},transfer_attr,
      64'h4e4f,64'h11,64'h22,64'h33,`FLEX_WAIT_F,status); 

//*********************************************************
// Test 4: rd_atomic command: (Single or burst mode)
// Verify halfword writes to SSRAM by reading back the data
// and comparing read_rslt to write values.
//*********************************************************
// Verify halfword writes by reading back data and comparing read_rslt.
   mpc740_read_req(mpcInst_1,'h40000048,{trans_8_bytes,read_normal},transfer_attr,`FLEX_WAIT_F,status);       
   mpc740_read_rslt(mpcInst_1,32'h40000048,0,read_rslt,status);
     if (read_rslt ===  64'h48494a4b4c4d4e4f ) begin
         $display("Test Passed : The read_normal result for address 32'4000_0048 1st data beat DW0 = %0h",read_rslt);
     end else begin 
         $display("Test Failed : The read_normal result for address 32'4000_0048 1st data beat DW0 = %0h",read_rslt);
     end 

//*********************************************************
// Test 5: write_ext_ctrl command: (Single beat 4-BYTE ONLY!)
// Word transfer test. Use external control word write.
//*********************************************************
   mpc740_write(mpcInst_1,'h40000050,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h50515253,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000054,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h54555657,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000058,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h58595a5b,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status); 
   mpc740_write(mpcInst_1,'h4000005c,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h5c5d5e5f,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);       

//*********************************************************
// Use both the read_ext_ctrl and read_normal commands. 
// Tests 1-5 use all four write and all six read commands.
//
// Verify word writes by reading back the data & comparing
// read_rslt to write data values.
//*********************************************************
   mpc740_read_req(mpcInst_1,'h40000050,{trans_8_bytes,read_normal},transfer_attr,`FLEX_WAIT_F,status);       
   mpc740_read_rslt(mpcInst_1,32'h40000050,0,read_rslt,status);
     if (read_rslt === 64'h5051525354555657 ) begin
         $display("Test Passed : The read result for address 32'4000_0050 1st data beat DW0 = %0h",read_rslt);
     end else begin 
         $display("Test Failed : The read result for address 32'4000_0050 1st data beat DW0 = %0h",read_rslt);
     end 
   mpc740_read_req(mpcInst_1,'h40000058,{trans_8_bytes,read_normal},transfer_attr,`FLEX_WAIT_F,status);       
   mpc740_read_rslt(mpcInst_1,32'h40000058,0,read_rslt,status);
     if (read_rslt === 64'h58595a5b5c5d5e5f ) begin
         $display("Test Passed : The read result for address 32'4000_0058 1st data beat DW0 = %0h",read_rslt);
     end else begin 
         $display("Test Failed : The read result for address 32'4000_0058 1st data beat DW0 = %0h",read_rslt);
     end 


   mpc740_read_req(mpcInst_1,'h40000050,{trans_4_bytes,read_ext_ctrl},transfer_attr,`FLEX_WAIT_F,status);       
   mpc740_read_rslt(mpcInst_1,32'h40000050,0,read_rslt,status);
     if (read_rslt === 64'hZZZZZZZZ50515253 ) begin
         $display("Test Passed : The read result for address 32'4000_0050 1st data beat DW0 = %0h",read_rslt);
     end else begin 
         $display("Test Failed : The read result for address 32'4000_0050 1st data beat DW0 = %0h",read_rslt);
     end 
   mpc740_read_req(mpcInst_1,'h40000054,{trans_4_bytes,read_ext_ctrl},transfer_attr,`FLEX_WAIT_F,status);       
   mpc740_read_rslt(mpcInst_1,32'h40000054,0,read_rslt,status);
     if (read_rslt === 64'hZZZZZZZZ54555657 ) begin
         $display("Test Passed : The read result for address 32'4000_0054 1st data beat DW0 = %0h",read_rslt);
     end else begin 
         $display("Test Failed : The read result for address 32'4000_0054 1st data beat DW0 = %0h",read_rslt);
     end 


//*********************************************************
// Test 6: write_flush command: (Single or burst mode)
// Three byte write transfer test. 
// Write a doubleword using a sequence of two (3B and 1B write) 
// Write a doubleword using a sequence of two (1B and 3B write)
// Byte writes only work in single beat mode, NOT BURST!
//*********************************************************
   mpc740_write(mpcInst_1,'h40000060,{trans_3_bytes,write_flush},transfer_attr,
      64'h606162,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000063,{trans_1_byte,write_flush},transfer_attr,
      64'h63,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000064,{trans_3_bytes,write_flush},transfer_attr,
      64'h646566,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000067,{trans_1_byte,write_flush},transfer_attr,
      64'h67,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);

   mpc740_write(mpcInst_1,'h40000068,{trans_1_byte,write_flush},transfer_attr,
      64'h68,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000069,{trans_3_bytes,write_flush},transfer_attr,
      64'h696a6b,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h4000006c,{trans_1_byte,write_flush},transfer_attr,
      64'h6c,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h4000006d,{trans_3_bytes,write_flush},transfer_attr,
      64'h6d6e6f,64'h0,64'h0,64'h0,`FLEX_WAIT_F,status);

//*********************************************************
// Test 6: read_no_cache command: (Single mode) 
// Verify misaligned three byte writes by reading back the
// SRAM data and comparing to expected results.
//*********************************************************        
   mpc740_read_req(mpcInst_1,'h40000060,{trans_8_bytes,read_no_cache},transfer_attr,`FLEX_WAIT_F,status);  
   mpc740_read_req(mpcInst_1,'h40000068,{trans_8_bytes,read_no_cache},transfer_attr,`FLEX_WAIT_F,status);  
   mpc740_read_rslt(mpcInst_1,32'h40000060,0,read_rslt,status);
     if (read_rslt === 64'h6061626364656667 ) begin
         $display("Test Passed : The read result for address 32'4000_0060 1st data beat DW0 = %0h",read_rslt);
     end else begin 
         $display("Test Failed : The read result for address 32'4000_0060 1st data beat DW0 = %0h",read_rslt);
     end 
   mpc740_read_rslt(mpcInst_1,32'h40000068,0,read_rslt,status);
     if (read_rslt === 64'h68696a6b6c6d6e6f ) begin
         $display("Test Passed : The read result for address 32'4000_0068 1st data beat DW0 = %0h",read_rslt);
     end else begin 
         $display("Test Failed : The read result for address 32'4000_0068 1st data beat DW0 = %0h",read_rslt);
     end 











// Test the six read operations that require data tenures:
// Verify FPGA accepts all read commands and presents data correctly.
   // read_normal: (Single or burst mode supported)
   mpc740_read_req(mpcInst_1,'h40000000,{trans_32_bytes,read_normal},transfer_attr,`FLEX_WAIT_F,status);  
           
   // read_itnt_mod: (BURST mode ONLY!!)
   mpc740_read_req(mpcInst_1,'h40000020,{trans_32_bytes,read_itnt_mod},transfer_attr,`FLEX_WAIT_F,status);  

   // read_ext_ctrl: (SINGLE WORD ONLY!!) 
   mpc740_read_req(mpcInst_1,'h40000040,{trans_4_bytes,read_ext_ctrl},transfer_attr,`FLEX_WAIT_F,status);  
   mpc740_read_req(mpcInst_1,'h40000044,{trans_4_bytes,read_ext_ctrl},transfer_attr,`FLEX_WAIT_F,status);  

   // read_atomic: (Single or burst mode supported)
   mpc740_read_req(mpcInst_1,'h40000048,{trans_8_bytes,read_atomic},transfer_attr,`FLEX_WAIT_F,status);  

   // read_itnt_mod_at: (BURST mode ONLY!!)
   mpc740_read_req(mpcInst_1,'h40000000,{trans_32_bytes,read_itnt_mod_at},transfer_attr,`FLEX_WAIT_F,status);  

   // read_no_cache (Single or burst mode supported)
   mpc740_read_req(mpcInst_1,'h40000040,{trans_8_bytes,read_no_cache},transfer_attr,`FLEX_WAIT_F,status);  

// Test the FOUR supported write operations that REQUIRE data tenures:
   //
   // write_ext_ctrl (SINGLE 32-bit WORD only!)
   // External control write only supports a single beat, 32-bit, write!
   // Fill up an entire cache line for later read verify.
   mpc740_write(mpcInst_1,'h40000100,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h00010203,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000104,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h04050607,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000108,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h08090a0b,64'h00, 64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h4000010c,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h0c0d0e0f,64'h00, 64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000110,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h10111213,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000114,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h14151617,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000118,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h18191a1b,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h4000011c,{trans_4_bytes,write_ext_ctrl},transfer_attr,
      64'h1c1d1e1f,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);                  
      
   // write_flush (single or burst mode supported)
   // 32-byte write, four beats of 8 bytes each.
   mpc740_write(mpcInst_1,'h40000120,{trans_32_bytes,write_flush},transfer_attr,
      64'h2021222324252627,64'h28292a2b2c2d2e2f,64'h3031323334353637,64'h38393a3b3c3d3e3f,`FLEX_WAIT_F,status);

   // write_kill (single or burst mode supported)
   // 32-byte write, four beats of 8 bytes each.
   mpc740_write(mpcInst_1,'h40000140,{trans_32_bytes,write_kill},transfer_attr,
      64'h4041424344454647,64'h48494a4b4c4d4e4f,64'h5051525354555657,64'h58595a5b5c5d5e5f,`FLEX_WAIT_F,status);

   // write_flush_at (single beat transfer only!)
   // 32-byte write, 1 beat of 8 bytes.
   mpc740_write(mpcInst_1,'h40000160,{trans_8_bytes,write_flush_at},transfer_attr,
      64'h6061626364656667,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000168,{trans_8_bytes,write_flush_at},transfer_attr,
      64'h68696a6b6c6d6e6f,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000170,{trans_8_bytes,write_flush_at},transfer_attr,
      64'h7071727374757677,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);
   mpc740_write(mpcInst_1,'h40000178,{trans_8_bytes,write_flush_at},transfer_attr,
      64'h78797a7b7c7d7e7f,64'h00,64'h00,64'h00,`FLEX_WAIT_F,status);


// Burst read from SSRAM.
   mpc740_read_req(mpcInst_1,'h40000100,{trans_32_bytes,read_itnt_mod},transfer_attr,`FLEX_WAIT_F,status);  

// Single beat read.
   mpc740_read_req(mpcInst_1,'h40000120,{trans_8_bytes,read_no_cache},transfer_attr,`FLEX_WAIT_F,status);  
  

// Test the address only operations that DO NOT REQUIRE data tenures:
// FPGA must assert N_AACK even if no data tenure is required.
// Bus snooper may assert N_ARTRY to indicate need to update data in memory.
//*********************
   // clean_blk
   mpc740_write_addr_only(mpcInst_1,'h4000F000,clean_blk,transfer_attr,`FLEX_WAIT_F,status);
   // flush_blk
   mpc740_write_addr_only(mpcInst_1,'h4000F000,flush_blk,transfer_attr,`FLEX_WAIT_F,status);
   // sync
   mpc740_write_addr_only(mpcInst_1,'h4000F000,sync,transfer_attr,`FLEX_WAIT_F,status);
   // kill_blk
   mpc740_write_addr_only(mpcInst_1,'h4000F000,kill_blk,transfer_attr,`FLEX_WAIT_F,status);

   mpc740_read_req(mpcInst_1,'h40000100,{trans_32_bytes,read_itnt_mod},transfer_attr,`FLEX_WAIT_F,status);  
   mpc740_read_req(mpcInst_1,'h40000120,{trans_32_bytes,read_itnt_mod},transfer_attr,`FLEX_WAIT_F,status);  
   mpc740_read_req(mpcInst_1,'h40000140,{trans_32_bytes,read_itnt_mod},transfer_attr,`FLEX_WAIT_F,status);  
   mpc740_read_req(mpcInst_1,'h40000160,{trans_32_bytes,read_itnt_mod},transfer_attr,`FLEX_WAIT_F,status);  

         
   // eieio_addr
   mpc740_write_addr_only(mpcInst_1,'h4000F000,eieio_addr,transfer_attr,`FLEX_WAIT_F,status);
   // tlb_invalib
   mpc740_write_addr_only(mpcInst_1,'h4000F000,tlb_invalib,transfer_attr,`FLEX_WAIT_F,status);
   // lwarx
   mpc740_write_addr_only(mpcInst_1,'h4000F000,lwarx,transfer_attr,`FLEX_WAIT_F,status);
   // tlbsync
   mpc740_write_addr_only(mpcInst_1,'h4000F000,tlbsync,transfer_attr,`FLEX_WAIT_F,status);
   // icbi
   mpc740_write_addr_only(mpcInst_1,'h4000F000,icbi,transfer_attr,`FLEX_WAIT_F,status);


   mpc740_read_req(mpcInst_1,'hFFFF0100,{trans_8_bytes,read_normal},transfer_attr,`FLEX_WAIT_F,status);  
    
   idle_cycle(1000);

   $display("Finished PowerPC 60X Bus Interface Test Bench");
   idle_cycle(10);

  #5000 $stop;
   $finish;         
   end //CMD_AND_STIMULI_STREAM_MPC740
   



/*  -------------------------------
   | NOTES FOR THE mpc750_l2 model |
    -------------------------------

   The basic purpose of this description is to help the user
   to understand the command execution. 
  
 A]L2CR Cache Control Register(L2CR): The register can be configured to control L2 operation.
    This register can be set through command set_reg.This register is of 32 bit.But only following
    bits are modelled.
          ---------------------------------------------------------------------
         |  Bit      |	Name	|	Function                               |
         |---------------------------------------------------------------------|
         |   1	     |	L2PE	|	L2 data parity generation and checking |
         |   2-3     |	L2SIZ	|	L2 size                                |
         |   7-8     |	L2RAM	|       L2 RAM type                            |
         |   17      |	L2DF	|	L2 differential clock                  |
         |           |          |                                              |
          ----------------------------------------------------------------------
 
 B]Following commands can be given for controlling :-
  1) mpc750_l2_read_burst_req
  2) mpc750_l2_write_burst
  3) mpc750_l2_read_rslt
  4) mpc750_l2_abort
  5) mpc750_l2_idle
  6) mpc750_l2_read_modify_write
  7) mpc750_l2_output_enable ***
  8) mpc750_l2_set_pin
  9) mpc750_l2_pin_req
 10) mpc750_l2_pin_rslt
 11) mpc750_l2_set_msg_level**
 12) mpc750_l2_set_timing_control**
 13) mpc750_l2_set_reg
 14) mpc750_l2_reg_req **
 15) mpc750_l2_reg_rslt**
 
 The control functions described in this section describe the command set that will be
 implemented for the model.

 MODEL COMMANDS :-
 Bus commands such as read_burst, write_burst and idle, cause the model to become active in 
 simulation. When the model seeks its next command, it will execute as many non-bus commands
 (set_reg, get_reg, etc.) as necessary to reach the next bus command. The state machine in the
 mpc750_l2_bfm will support model commands that generate bus cycles, consisting of a name and 
 an argument list.
 
 Generic Rules/Parameters in Commands:
 1)  Id  This gives the instance Id# for the model. 
 
 2)  wait if true, the command source will pause until the completion of this command. This can
  be set to true `to break the pipelining in the model.
 

1) Idle Command :-

     mpc750_l2_idle(Id, number_of_idles, wait, status)

 Idle is a special type of bus command. It asks the model to behave as if the device has no bus
 operations to perform. Both address and data buses are in idle model during the execution of
 this command.
          The "idle" command causes the specified number of idle cycles to appear on the bus.
 An idle cycle causes the model to idle for one clock cycle. 

 Idle Argument Description:
 Id          = This gives the instance ID # for the model.
 No_of_idles = Number of idle cycles to perform.
 Status      = if the command core successfully receives the command, a positive status value
 will be returned.
 Wait        = if true, the command source will pause until the completion of this  command.

 2)Burst Read Command and Read result Command :-

     mpc750_l2_read_burst_req( Id, address, wait, status)

     mpc750_l2_read_rslt (Id, address, tag, result, status)

          The mpc750_l2_read_burst_req command is used to generate read_cycles on the L2 cache
 bus, it does not return any results except command status which indicates successful communication
 with the command core. The results from read_req are accessed using the mpc750_l2_read_rslt command.
 This command always prevents the command source from providing new commands to the model until it is 
 complete. If all previous commands have completed, and the desired result has not become available 
 then the result command completes with an error. 
	The mpc750_l2_read_burst_req performs the burst read. And for each beat in burst address will
 be driven by L2 cache interface, which will be in incremental fashion.
	The mpc750_l2_read_burst_req command will generate the an extra read cycle to keep the burst 
 RAM driving the data bus for the last read.
        This extra read cycle will be generated for Flow through Synchronous burst SRAM and Pipelined 
 Synchronous burst SRAM.

 Read Argument Description:
 Id      = This gives the instance ID # for the model.
 Address = Address location to read from.
 Tag     = The tag number from the corresponding read_req.
 Status  = If the command core successfully receives the command, a positive status value will be
           returned.
 
 Result  = Data returned from the read_rslt command.
 Wait    = If true, the command source will pause until the completion of this command.

 3)Burst Write command :-

       mpc750_l2_burst_write(Id, address, data0, data1, data2 data3, wait, status)

 The mpc750_l2_write_burst command is used to generate write_cycles on the L2 cache bus, it does
 return command status which indicates successful communication with the command core. 
	The mpc750_l2_write_burst performs the burst write. And for each beat in burst address will
 be driven by L2 cache interface, which will be in incremental fashion.

 Write Argument Description: 
 Id            = This gives the instance ID # for the model.
 Address       = Address location to be written
 Data0, Data1, Data2, Data3  Description: The data parameters are bit_vector values that the operation
                 must write.  All the four data  arguments must be specified for a burst write transfer. 
 Wait          = If true, the command source will pause until the completion of this command.
 Status        = If the command core successfully receives the command, a positive status value
                 will be returned.


4)Read modified write command:-

        mpc750_l2_read_modified_write(Id, data0,wait, status)

 This command is used to perform the read-modify-write L2 cache operation.In this command read is done 
 at given address.Then at same address data is written.Hence this is called read modify write.
 
 Read modified write command Argument Description: 
 Id      =  This gives the instance ID # for the model.
 Address = Address location to be read and then written
 Data0 Description = The data parameters are bit_vector values that the operation must write.  
        All the four data arguments must be specified for a burst write transfer. 
 Wait    = If true, the command source will pause until the completion of this command.
 Status  = If the command core successfully receives the command, a positive status value will be returned.

 Note: In this read-modify-write cycle the data at address R8 is read and then at W9 (R8) write is performed. 

5)Abort command :-
 
        mpc750_l2_abort(Id, wait, status)
	
	This command is used to perform abort read This command should be given after read_burst_req 
 to perform abort cycle. If this command is given after any other command except mpc750_l2_read_burst_req
 then message will be flashed and this command will be executed as idle command.

 Abort Argument Description: 
 Id     = This gives the instance ID # for the model.
 Wait   = If true, the command source will pause until the completion of this command.
 Status = If the command core successfully receives the command, a positive status value will be
          returned.

6)Set Register Commands.
  
        mpc750_l2_set_reg(instance, reg_name, reg_value, status)

          The mpc750_l2_set_reg command sets the value of a specified register. You can set an
 entire register, or individual bit of a register by specifying the specific bit appended to the 
 register name.


 Set reg Argument Description: 
 Instance   =  A positive integer that identifies the model instance, as defined by the model's id 
 generic/defparam.
 Reg_name   =  A constant that indicates the name of the register whose value is to be set. 
 Allowed values of reg_name are listed in Table 3.1.5. You can access the register by specifying 
 the register name.


7)Get_Pin Commands

            mpc750_l2_pin_req(Id, pin, wait, status)

            mpc750_l2_pin_rslt(Id, pin, rslt, status)

           The mpc750_l2_get_pin_req command is used to read the data from the model, it does not
 return any  results except the command status which indicates successful communication with the 
 command core  program. The results from mpc750_l2_get_pin_req are accessed using the 
 mpc750_l2_get_pin_rslt command.This command always prevents the command source from providing new
 commands to the model until it is  complete. If all previous commands have completed, and the desired
 result has not become available then the result command completes with an error.

Get_pin argument description:-
Id     = This gives the instance ID # for the model.
pin    = Identifies the pin.
wait   = If true, the command source will pause until the completion of this command.
Status = If the command core successfully receives the command, a positive status value will be returned.

8)Set_Pin command.

                mpc750_l2_set_pin(Id, pin, pin_value, wait, status)
 
                Sets the specified pin to the supplied value. 1 drives the pin high, 0 drives the pin low,
 anything else drives it unknown.
 Note that not all pins will be driven to "pin value" at the same time. The model will drive the 
 specified pin to the specified value at the appropriate time as specified in the data sheet.
 
 set_pin argument description:
 Id        = This gives the instance ID # for the model.
 Pin       = Identifies the pin.
 Pin_value = The level to set the pin to.
 Wait      = If true, the command source will pause until the completion of this command.
 Status    = If the command core successfully receives the command, a positive status value will be 
 returned.
 */

/* -----------------------------------------------------------------
  | CMD_AND_STIMULI_STREAM_MPC750_L2 :                              |
  | ---------------------------------                               |
  |     This block is used to pass commands to command core         |
  |     which are executed by the model mpc750_l2_fx.               |
  |     Each Block of commands in CMD_AND_STIMULI_STREAM_MPC750_L2  |
  |     process is followed by a procedure call made which generates| 
  |     stimuli.                                                    |
   ----------------------------------------------------------------- */
   

   initial 
      begin :CMD_AND_STIMULI_STREAM_MPC750_L2

         #(5 * `clk_period);
         idle_cycle(4);

         /* Prior to using any of the FlexModel commands, you need to get a model
          handle. Using the FlexModelId name that you passed to the instatiated
          model. Assign the flexHandle to the `MPC750_L2ModelId_1 signal for use in other
          processes. */

         flex_get_inst_handle(`MPC750_L2ModelId_1, mpc750_l2Inst_1, status_mpc750_l2);

         if (status_mpc750_l2 != 1) begin
            $display("Error: Could not get handle for model: %s" ,`MPC750_L2ModelId_1);
            $finish;
         end // if (status_mpc750_l2 != 1)

         /* The set_msg_level command enables the messages from the model like warnings and also 
          command posting messages */

         //mpc750_l2_set_msg_level(mpc750_l2Inst_1,`FLEX_ALL_MSGS,status_mpc750_l2);
         mpc750_l2_set_msg_level(mpc750_l2Inst_1,`FLEX_NO_MSGS,status_mpc750_l2);
         /* Now set reg l2cr.
          ---------------------------------------------------------------------------------|
         |  Bit      |	Name	|	Function                               | Value     |
         |---------------------------------------------------------------------|-----------|
         |   1	     |	L2PE	|	L2 data parity generation and checking | enable    |
         |   2-3     |	L2SIZ	|	L2 size                                | 512 kbytes|
         |   7-8     |	L2RAM	|       L2 RAM type                            | late_write|
         |   17      |	L2DF	|	L2 differential clock                  | enable    |
         |           |          |                                              |           |
          --------------------------------------------------------------------------------- */
         
         mpc750_l2_set_reg(mpc750_l2Inst_1,`MPC750_L2_L2CR_SET_REG_REG,{14'h0,l2df,8'h0,ssram_late,3'h0,one_512kb,l2pe,1'h0},status_mpc750_l2);

         // Give read reg request to read l2cr register contents.
         mpc750_l2_reg_req(mpc750_l2Inst_1,`MPC750_L2_L2CR_SET_REG_REG,`FLEX_WAIT_F,status_mpc750_l2);

         // Give reg_rslt to read display register value.
         mpc750_l2_reg_rslt(mpc750_l2Inst_1,`MPC750_L2_L2CR_SET_REG_REG,reg_val,status_mpc750_l2);
         $display(" Value of register L2cr is %b",reg_val);

      end // block: CMD_AND_STIMULI_STREAM_MPC750_L2
   

/*-----------------------------------------------------------------
 * continuous assignments for the Bidirectionals  of mpc740
 *-----------------------------------------------------------------*/
   assign (weak1, strong0) Ts_io = 1'b1;
   assign (weak1, strong0) Gbl_io = 1'b1;   
   assign Abb_io = Abb_ior;
   assign Ts_io = Ts_ior;
   assign A_io = A_ior;
   assign Ap_io = Ap_ior;
   assign Tt_io = Tt_ior;
   assign Tbst_io = Tbst_ior;
   assign Tsiz_io = Tsiz_ior;
   assign Gbl_io = Gbl_ior;
   assign Wt_io = Wt_ior;
   assign Ci_io = Ci_ior;
   assign Artry_io = Artry_ior;
   assign Dbb_io = Dbb_ior;
   assign Dl_io = Dl_ior;
   assign Dh_io = Dh_ior;
   assign Dp_io = Dp_ior;
   assign Dbdis_io = Dbdis_ior;
   assign Jtag_cop_io = Jtag_cop_ior;
   assign Test_io = Test_ior;
/*-----------------------------------------------------------------
 * continuous assignments for the Bidirectionals  of mpc750_l2
 *-----------------------------------------------------------------*/
   assign L2data_io = L2data_ior;
   assign L2dp_io = L2dp_ior;

endmodule // top





