library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

--`timescale 1ns/1ns
entity ADR_ARB is
port (
 	CLK :		in	STD_LOGIC;  -- Buffered system clock.
 	N_RESET : 	in	STD_LOGIC;  -- Power on reset to initialize logic.
 	N_TS_DLY :	in	STD_LOGIC;  -- Transfer start output from processors.
 	N_BRA :	in	STD_LOGIC;  -- Bus request processor A.
 	N_BRB :	in	STD_LOGIC;  -- Bus request processor B.
 	ADR_BRA :	out	STD_LOGIC;  -- Address qualifier source: processor A.
 	ADR_BRB :	out	STD_LOGIC;  -- Address qualifier source: processor B.
 	N_BGA :	out	STD_LOGIC;  -- Bus grant to processor A.
 	N_BGB :	out	STD_LOGIC   -- Bus grant to processor B.
      );
end ADR_ARB;

architecture ADR_ARB of ADR_ARB is

-- Mealy state machine with synchronous outputs. This design provides a two
-- processor address bus arbiter to support pipelined address transactions
-- for a PowerPC 60X bus system.

  -- ONE HOT state machine state assignments:
  type STATE_TYPE is (s0, s1, s2, s3, s4, s5, s6 );
  attribute ENUM_ENCODING: STRING;
  attribute ENUM_ENCODING of STATE_TYPE: type is "0000001 0000010 0000100  0001000 0010000 0100000 1000000";
  signal current:  STATE_TYPE;

  -- synchronize bus request inputs.
  signal N_BRA_IN :	STD_LOGIC;
  signal N_BRB_IN : 	STD_LOGIC;
begin

IOB_DFF:
process ( CLK, N_RESET )
 begin
  if ( N_RESET = '0' ) then
    N_BRA_IN <= '1';
    N_BRB_IN <= '1';
  elsif (CLK'event and CLK = '1' ) then
    N_BRA_IN <= N_BRA;
    N_BRB_IN <= N_BRB;
  end if;
end process;


STATE_MACHINE:
process ( CLK, N_RESET )
 begin
 -- State machine code:
  if ( N_RESET = '0' ) then
    current <= s0; ADR_BRA <= '0'; ADR_BRB <= '0'; N_BGA <= '1'; N_BGB <= '1';  -- deassert outputs    
  elsif (CLK'event and CLK = '1' ) then
  -- #2 add output switching delay to emulate DFF switching delays.
  -- start out in IDLE state after first rising edge of clk.
  
  case(current) is

  when s0 =>

     if( N_BRA_IN = '0' ) then
     -- uP_A has requested an ADDRESS BUS tenure. Grant it now.
       current <= s1; ADR_BRA <= '1'; ADR_BRB <= '0'; N_BGA <= '0'; N_BGB <= '1';     
     elsif(( (not N_BRB_IN) and N_BRA_IN ) = '1' )  then
     -- uP_B has requested an ADDRESS BUS tenure. Grant it now.       
       current <= s2; ADR_BRA <= '0'; ADR_BRB <= '1'; N_BGA <= '1'; N_BGB <= '0';    
     else
     -- Neither uP has requested an ADDRESS BUS tenure.        
       current <= s0; ADR_BRA <= '0'; ADR_BRB <= '0'; N_BGA <= '1'; N_BGB <= '1';          
     end if;

  when s1 =>
     if( N_TS_DLY = '0' ) then
     -- uP_A has started an ADDRESS BUS tenure.
       current <= s3; ADR_BRA <= '1'; ADR_BRB <= '0'; N_BGA <= '0'; N_BGB <= '1';        
     elsif(( N_TS_DLY and ( not N_BRB_IN) ) = '1' ) then 
     -- No transfer start from uP_A, uP_B request asserted.      
       current <= s5; ADR_BRA <= '1'; ADR_BRB <= '0'; N_BGA <= '0'; N_BGB <= '1';          
     else     
     -- Wait for uP_A to start the ADDRESS BUS tenure.        
       current <= s1; ADR_BRA <= '1'; ADR_BRB <= '0'; N_BGA <= '0'; N_BGB <= '1';          
     end if;

  when s2 =>
     if( N_TS_DLY = '0' ) then
     -- uP_B has started an ADDRESS BUS tenure.       
       current <= s4; ADR_BRA <= '0'; ADR_BRB <= '1'; N_BGA <= '1'; N_BGB <= '0';             
     elsif(( N_TS_DLY and (not N_BRA_IN) ) = '1' ) then
     -- No transfer start from uP_B, uP_A request asserted.        
       current <= s6; ADR_BRA <= '0'; ADR_BRB <= '1'; N_BGA <= '1'; N_BGB <= '0';          
     else     
     -- Wait for uP_B to start the ADDRESS BUS tenure.        
       current <= s2; ADR_BRA <= '0'; ADR_BRB <= '1'; N_BGA <= '1'; N_BGB <= '0';          
     end if;

  when s3 =>
  -- this state stores address qualifiers associated with uP_A address tenure.
     if( N_BRB_IN = '0' ) then
     -- uP_B has requested an ADDRESS BUS tenure.
     -- Store address & bus qualifiers in address pipeline and deassert N_BGA.
       current <= s2; ADR_BRA <= '1'; ADR_BRB <= '0'; N_BGA <= '1'; N_BGB <= '0';         
     else     
     -- Park address grant on uP_A.        
       current <= s1; ADR_BRA <= '1'; ADR_BRB <= '0'; N_BGA <= '0'; N_BGB <= '1';           
     end if;

  when s4 =>
     if( N_BRA_IN = '0' ) then
     -- uP_A has requested an ADDRESS BUS tenure.
     -- Store address & bus qualifiers in address pipeline and deassert N_BGB.
       current <= s1; ADR_BRA <= '0'; ADR_BRB <= '1'; N_BGA <= '0'; N_BGB <= '1';     
     else     
     -- Park address grant on uP_B.
       current <= s2; ADR_BRA <= '0'; ADR_BRB <= '1'; N_BGA <= '1'; N_BGB <= '0';         
     end if;

  when s5 =>
     if( N_TS_DLY = '0' ) then
     -- uP_A has started the ADDRESS BUS tenure while grant is transitioning.
     -- Go to s3 to store uP_A address bus qualifiers.
       current <= s3; ADR_BRA <= '1'; ADR_BRB <= '0'; N_BGA <= '1'; N_BGB <= '0';      
     else     
     -- Transition grant to uP_B.        
       current <= s2; ADR_BRA <= '0'; ADR_BRB <= '0'; N_BGA <= '1'; N_BGB <= '0';           
     end if;

  when s6 =>
     if( N_TS_DLY = '0' ) then
     -- uP_B has started the ADDRESS BUS tenure while grant is transitioning.
     -- Go to S4 to store uP_B address bus qualifiers.
       current <= s4; ADR_BRA <= '0'; ADR_BRB <= '1'; N_BGA <= '0'; N_BGB <= '1';        
     else     
     -- Transition grant to uP_A.
       current <= s2; ADR_BRA <= '0'; ADR_BRB <= '0'; N_BGA <= '0'; N_BGB <= '1';           
     end if;

  end case;
end if;

end process;
end ADR_ARB;
