--The IEEE standard 1164 package, declares std_logic, rising_edge(), etc.
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

--`timescale 1ns/1ns
entity ADRPIPE is
port (  

 A : 			in  	STD_LOGIC_VECTOR(0 to 31); -- 60x Address bus
 N_TS :		in 	STD_LOGIC; -- 60x uP starting address bus tenure.
 TT:	 		in	STD_LOGIC_VECTOR(0 to 4); -- 60x transfer type
 TSIZ :		in	STD_LOGIC_VECTOR(0 to 2); -- 60x transfer size
 N_TBST :		in	STD_LOGIC; -- 60x transfer burst
 ENABLE_XFR :	in	STD_LOGIC; -- Move address from input to output stage.
 DONE : 		in	STD_LOGIC; -- SRAM controller completed data bus tenure.
 DONE_FLASH : 	in 	STD_LOGIC; -- Flash controller completed data bus tenure.
 CLK: 		in  	STD_LOGIC; -- 60x system clock
 N_RESET: 		in    STD_LOGIC; -- Active high Power on reset
 N_TS_DLY :		out	STD_LOGIC; -- Synchronized transfer start signal.
 MEMA : 	buffer 	STD_LOGIC_VECTOR(0 to 17); -- Synchronized PPC address bus.
 BURST : 	buffer	STD_LOGIC; -- Synchronized burst.
 R_nW : 	buffer	STD_LOGIC; -- Syncrhonized transfer size.
 BWE : 	buffer	STD_LOGIC_VECTOR(0 to 7); -- Synchronized transfer type bits.
 BUS_REQ : 	buffer	STD_LOGIC; -- SSRAM memory request. 
 BUSREQ_FLASH : buffer	STD_LOGIC; -- Flash memory request.
 DATAXFR : 	buffer	STD_LOGIC; -- Data bus tenure required based upon TT(0:4) inputs.
 BANKSEL: 	buffer	STD_LOGIC_VECTOR(0 to 1)-- SRAM memory bank ( A(9:10) )
 ) ;
end ADRPIPE;


architecture ADRPIPE of ADRPIPE is    

-- Declare local signals used in the design: 
   -- Input pipeline signals:  
   signal ADR:   		STD_LOGIC_VECTOR(0 to 31);
   signal TX_TYPE : 	STD_LOGIC_VECTOR(0 to 4);
   signal TX_SIZE :	STD_LOGIC_VECTOR(0 to 2);

   -- Synchronize external 60X bus inputs before using in pipeline. 
   signal A_IN :   	STD_LOGIC_VECTOR(0 to 31);
   signal TT_IN : 	STD_LOGIC_VECTOR(0 to 4);
   signal TSIZ_IN :	STD_LOGIC_VECTOR(0 to 2);
   signal N_TBST_IN: 	STD_LOGIC;
   signal N_TS_IN:	STD_LOGIC;
   
   signal BYTE_ADR :	STD_LOGIC_VECTOR(0 to 2);
   signal BUS_BURST:	STD_LOGIC;
   signal READ :		STD_LOGIC;
   signal SRAM :		STD_LOGIC;
   signal FLASH :		STD_LOGIC;
   signal TBST :		STD_LOGIC;
--******************************************************************       
-- If transfer start is not asserted, hold prior address information.
-- If transfer start is asserted, clock in new address information.
-- If ENABLE_XFR then move from input to output stage DFF.
-- Multiplexor permits data to be held without gated clock design.
--******************************************************************
   signal MUX_ADDR :	STD_LOGIC_VECTOR(0 to 31); 
   signal MUX_TX_TYPE :	STD_LOGIC_VECTOR(0 to  4);
   signal MUX_TX_SIZE : STD_LOGIC_VECTOR(0 to  2);  
   signal MUX_MEMA : 	STD_LOGIC_VECTOR(0 to 17);
   signal MUX_BWE :	STD_LOGIC_VECTOR(0 to 7);  
   signal MUX_BANKSEL :	STD_LOGIC_VECTOR(0 to 1);  
   signal MUX_BRST :	STD_LOGIC;
   signal MUX_BURST:	STD_LOGIC;
   signal MUX_RnW :	STD_LOGIC;
   signal MUX_SRAM :	STD_LOGIC;
   signal MUX_FLASH :	STD_LOGIC;
   signal BYTE_WE :	STD_LOGIC_VECTOR(0 to 7); 
   signal BYTE_ADDR :   STD_LOGIC_VECTOR(0 to 2);
   signal WHAT: STD_LOGIC_VECTOR(0 to 6);

begin

IOB_SYNC_INPUTS:    
 process ( CLK, N_RESET )
  begin    
-- Synchronize 60X bus inputs before using internally.
    if ( N_RESET = '0') then 
   -- Point to 1st instruction address but don't enable transfers.
       A_IN 	 <= (others => '0');
       TT_IN       <= (others => '0');
       TSIZ_IN 	 <= (others => '0');
       N_TBST_IN   <= '1';
       N_TS_IN     <= '1';         
    elsif ( CLK'event and CLK = '1' ) then
       A_IN 	 <= A;
       TT_IN       <= TT;
       TSIZ_IN 	 <= TSIZ;
       N_TBST_IN   <= N_TBST;
       N_TS_IN     <= N_TS;     
    end if;
 end process;
 
 --***********************************************************************************
 -- Output N_TS_OUT for state machine usage. 
 -- Transfer qualifiers, Address, and transfer start all delayed one clock internally.
 --***********************************************************************************
 N_TS_DLY <= N_TS_IN;
 
  INPUT_STAGE_MUX:
  -- input stage address multiplexor.
  process ( N_TS_IN, A_IN, TT_IN, TSIZ_IN, N_TBST_IN , ADR, TX_TYPE, TX_SIZE, TBST)
  begin
    if (N_TS_IN = '1' ) then
      MUX_ADDR    <= ADR;
      MUX_TX_TYPE <= TX_TYPE;
      MUX_TX_SIZE <= TX_SIZE; 
      MUX_BRST    <= TBST;
    else
      MUX_ADDR    <= A_IN;
      MUX_TX_TYPE <= TT_IN;
      MUX_TX_SIZE <= TSIZ_IN; 
      MUX_BRST    <= not N_TBST_IN; 
    end if;
  end process;  
  
  OUTPUT_MUX:      
   -- output stage address multiplexor.
  process ( ENABLE_XFR, ADR, BUS_BURST, BYTE_WE, READ, SRAM, FLASH, MEMA, BURST, BWE, R_nW, BANKSEL, BUS_REQ, BUSREQ_FLASH, DONE, DONE_FLASH)
  begin
    if ( ENABLE_XFR = '1' ) then 
      MUX_MEMA    <=  ADR(11 to 28);
      MUX_BURST   <=  BUS_BURST;
      MUX_BWE     <=  BYTE_WE;
      MUX_RnW     <=  READ;
      MUX_BANKSEL <=  ADR(9 to 10);
      MUX_SRAM    <=  SRAM;   
      MUX_FLASH   <=  FLASH;     
    else
      MUX_MEMA    <=  MEMA;
      MUX_BURST   <=  BURST;
      MUX_BWE     <=  BWE;
      MUX_RnW     <=  R_nW;
      MUX_BANKSEL <=  BANKSEL;
      -- Use feedback from SRAMCTL and FLASH_IF to disable bus request
      -- when the state machine has signaled bus access completed. Conversely
      -- enable bus request to flow through if state machines are IDLE.
      MUX_SRAM    <=  BUS_REQ and (not DONE);   
      MUX_FLASH   <=  BUSREQ_FLASH and (not DONE_FLASH);         
    end if;
  end process;    
    

ASYNC_RST_DFF:    
process ( CLK, N_RESET )
  begin    
-- D Flip Flop (asynchronous Reset) to hold incoming address & qualifiers
-- and output stage for memory address and memory request signals.
-- Always store address transaction when N_TS is asserted. Data bus arbiter
-- asserts address acknowledge N_AACK, so the address won't change until the
-- decode logic has a chance to determine if a data bus tenure is needed.

   if ( N_RESET = '0') then --asynchronous RESET
   -- Point to 1st instruction address but don't enable transfers.
   -- Address Pipeline: INPUT stage.     
     ADR      <=  "11111111111100000000000100000000";  -- 1st instruction fetched after HRESET.
     TX_TYPE  <=  (others => '0'); -- Flush block address only.
     TX_SIZE  <=  "010"; -- Eight words. 
     TBST     <=  '0';   -- Burst disabled.
   -- Address Pipeline: OUTPUT stage. 
     MEMA     <=  (others => '0'); -- SRAM memory address.
     BURST    <=  '0' ;  -- disable 4 beat request.
     BWE      <=  (others => '0'); -- disable byte write enables.
     R_nW     <=  '1'; -- Read not write.
     BANKSEL  <=  "00"; -- SRAM Bank 0
     BUS_REQ  <=  '0';  -- SSRAM memory access request.
     BUSREQ_FLASH <= '0'; -- Flash memory access request.
   elsif (CLK'event and CLK='1') then           
   --  #2 simulate DFF with output switching delays.
       -- at CLK rising edge save old qualifiers or load next.
       -- Enable transfer from input to output stage if ENABLE_XFR asserted.
       -- input stage.
       ADR 		<=  MUX_ADDR;
       TX_TYPE  	<=  MUX_TX_TYPE;
       TX_SIZE  	<=  MUX_TX_SIZE;
       TBST 	<=  MUX_BRST;
       -- output stage.
       MEMA     	<=  MUX_MEMA ; -- SRAM memory address.
       BURST    	<=  MUX_BURST ; -- disable 4 beat request.
       BWE      	<=  MUX_BWE ; -- disable byte write enables.
       R_nW     	<=  MUX_RnW ; -- Read not write.
       BANKSEL  	<=  MUX_BANKSEL; -- SRAM Bank 0:3 selected by address inputs.
       BUS_REQ  	<=  MUX_SRAM; -- Request SRAM memory access.
       BUSREQ_FLASH <= MUX_FLASH; -- Request Flash memory access.
   end if;
end process;

BYTE_ADDR  <= ADR(29 to 31); 



Decode_Transfer_Type:
-- Decode address bus qualifiers and return values to other modules.
-- Must determine if ADDRESS ONLY bus tenure and ignore. Address only
-- transfer returns DATAXFR='0' so N_AACK can be asserted without a
-- memory cycle generated to SSRAM or flash.
process ( TX_TYPE )
  begin
    case ( TX_TYPE ) is
      when "10100" => DATAXFR <= '1';
      when "11100" => DATAXFR <= '1';
      when "10010" => DATAXFR <= '1';
      when "11010" => DATAXFR <= '1';
      when "11110" => DATAXFR <= '1';
      when "00010" => DATAXFR <= '1';
      when "00110" => DATAXFR <= '1';
      when "01010" => DATAXFR <= '1';
      when "01110" => DATAXFR <= '1';
      when "01011" => DATAXFR <= '1';

      when others  => DATAXFR <= '0';
    end case;
    
    READ <= TX_TYPE(1);
end process;


Transfer_SIZE:
-- This process decodes the Transfer Size Qualifers to determine 
-- if a burst cycle is requested and for write cycles what the 
-- byte enables should be. This process also supports the 
-- misaligned data transfers.
-- Note that BYTE_WE's get inverted in the SRAMCTL module.

WHAT <= TBST & TX_SIZE & ADR(29 to 31);

process ( TBST, WHAT )
  begin
    case ( WHAT ) is
    
      when "1010000" => BYTE_WE <= "11111111"; -- 4 beat 8 byte burst   
      
      when "0000000" => BYTE_WE <= "11111111"; -- 1 beat 8 byte
      
      when "0001000" => BYTE_WE <= "10000000"; -- 1 beat byte 0
      when "0001001" => BYTE_WE <= "01000000"; -- 1 beat byte 1
      when "0001010" => BYTE_WE <= "00100000"; -- 1 beat byte 2
      when "0001011" => BYTE_WE <= "00010000"; -- 1 beat byte 3
      when "0001100" => BYTE_WE <= "00001000"; -- 1 beat byte 4
      when "0001101" => BYTE_WE <= "00000100"; -- 1 beat byte 5
      when "0001110" => BYTE_WE <= "00000010"; -- 1 beat byte 6
      when "0001111" => BYTE_WE <= "00000001"; -- 1 beat byte 7
      
      when "0010000" => BYTE_WE <= "11000000"; -- 1 beat bytes 0&1 
      when "0010001" => BYTE_WE <= "01100000"; -- 1 beat bytes 1&2
      when "0010010" => BYTE_WE <= "00110000"; -- 1 beat bytes 2&3
      when "0010100" => BYTE_WE <= "00001100"; -- 1 beat bytes 4&5 
      when "0010101" => BYTE_WE <= "00000110"; -- 1 beat bytes 5&6
      when "0010110" => BYTE_WE <= "00000011"; -- 1 beat bytes 6&7 
      
      -- if 3B transfer crosses word boundry it gets
      -- broken up into two smaller transfers. Only
      -- odd bytes in word boundries work in 1 beat.     
      when "0011000" => BYTE_WE <= "11100000"; -- 3 bytes 0&1&2
      when "0011001" => BYTE_WE <= "01110000"; -- 3 bytes 0&1&2
      when "0011100" => BYTE_WE <= "00001110"; -- 3 bytes 0&1&2
      when "0011101" => BYTE_WE <= "00000111"; -- 3 bytes 0&1&2
      
      when "0100000" => BYTE_WE <= "11110000"; -- 4 bytes 0&1&2&3
      when "0100100" => BYTE_WE <= "00001111"; -- 4 bytes 4&5&6&7
      -- 60X bus interface won't produce 5, 6, or 7 byte write requests.
      when others => BYTE_WE <= "00000000"; -- don't write.

    end case;
    
    BUS_BURST <= TBST ;
  end process;


ADR_DCD:
-- Decode address bits A(0:7) to determine if N_AACK should be asserted.
process ( ADR , DATAXFR )
  begin
  --SRAM @0x40 and FLASH @0xFF.
    SRAM     <= DATAXFR and (not ADR(0)) and ADR(1) and (not ADR(2)) and (not ADR(3)) and
                             (not ADR(4)) and (not ADR(5)) and (not ADR(6)) and (not ADR(7)) ;
    FLASH    <= DATAXFR and  ADR(0) and  ADR(1) and  ADR(2) and  ADR(3) and   
                              ADR(4) and  ADR(5) and   ADR(6) and  ADR(7) ;
  end process;

end ADRPIPE;

