library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

entity DATABUS is
   port (
	BUS_REQ:		in 		STD_LOGIC;
 	BUSREQ_FLASH:	in 		STD_LOGIC;
 	CLK:			in 		STD_LOGIC;
 	N_RESET:		in 		STD_LOGIC;
 	R_nW:			in 		STD_LOGIC;
 	BURST:		in 		STD_LOGIC;
 	N_MEMOE:		in 		STD_LOGIC;
   	FLASH_DB:		in		STD_LOGIC_VECTOR(0 to 63);
 	FLASH_DBP:		in		STD_LOGIC_VECTOR(0 to  7);
 	FLASH_TA:		in		STD_LOGIC;
 	SRAM_TA:		in		STD_LOGIC;
 	DONE:			in		STD_LOGIC;
 	DONE_FLASH:		in		STD_LOGIC;  
      DH:  			inout 	STD_LOGIC_VECTOR(0 to 31);
      DL:  			inout 	STD_LOGIC_VECTOR(0 to 31); 
      DP:  			inout 	STD_LOGIC_VECTOR(0 to  7);
      MEMDP:		inout 	STD_LOGIC_VECTOR(0 to  7);
      MEMD:	 		inout 	STD_LOGIC_VECTOR(0 to 63);
      
	-- add flash data output from IOB DFF synchronization logic
      FLASH_D_IN :	out   	STD_LOGIC_VECTOR(0 to 15);
 	N_TA:			out		STD_LOGIC	
	);
end DATABUS;

-- `timescale 1ns/1ns
architecture DATABUS of DATABUS is

-- Declare internal signals used in this design:
  signal DH_O: 		STD_LOGIC_VECTOR(0 to 31);
  signal DL_O: 		STD_LOGIC_VECTOR(0 to 31);
  signal DP_O: 		STD_LOGIC_VECTOR(0 to 7); 
  signal MEMDP_O:		STD_LOGIC_VECTOR(0 to 7);
  signal MEMD_O:		STD_LOGIC_VECTOR(0 to 63);
  signal OUT_DH:		STD_LOGIC_VECTOR(0 to 31); 
  signal OUT_DL:		STD_LOGIC_VECTOR(0 to 31);
  signal OUT_DP:		STD_LOGIC_VECTOR(0 to  7); 
  signal OUT_MEMDP:	STD_LOGIC_VECTOR(0 to  7);
  signal OUT_MEMD: 	STD_LOGIC_VECTOR(0 to 63);
  signal N_TAO:		STD_LOGIC;
  signal DATA_OE: 	STD_LOGIC;
  -- create separate DATA_OEW signals to lighten load on OE.
  signal DATA_OEW: 	STD_LOGIC_VECTOR(0 to 8);

  -- add another stage of DFF for input synchronization
  -- device doesn't meet timing specs when multiplexor on input data paths.  
  signal MEMDP_IN:	STD_LOGIC_VECTOR(0 to 7);
  signal MEMD_IN:		STD_LOGIC_VECTOR(0 to 63);
  signal DH_IN:		STD_LOGIC_VECTOR(0 to 31);
  signal DL_IN: 		STD_LOGIC_VECTOR(0 to 31);
  signal DP_IN:		STD_LOGIC_VECTOR(0 to 7);

  begin
  

PPC_DATA_BUS:
--*********************************************************************
-- Tristate outputs controlled by DATA_OE and DATA_OEW signals for 
-- the current bus tenure. PowerPC 60X data bus I/O and memory I/O
-- Enable tristate I/O one clock before data valid to provide enough
-- time for the buffer to turn on. 
--*********************************************************************
 process ( DATA_OE, OUT_DH, OUT_DL, OUT_DP )
  begin
  -- If read cycle route MEMD and MEMDP to DH, DL, and DP outputs. 
    if DATA_OE = '1' then   
      DH_O    <= OUT_DH;
      DL_O    <= OUT_DL;
      DP_O    <= OUT_DP;   
    else 
      DH_O <= (others => 'Z');
      DL_O <= (others => 'Z');
      DP_O <= (others => 'Z');
    end if;
 end process;

MEM_DATA_BUS0:
 process ( DATA_OEW, OUT_MEMD )
  begin
     if ( DATA_OEW(0) = '1' ) then  
       MEMD_O(0 to 7)  <= OUT_MEMD(0 to 7); 
     else 
       MEMD_O(0 to 7)  <= (others => 'Z');
     end if;      
 end process;

MEM_DATA_BUS1:
 process ( DATA_OEW, OUT_MEMD )
  begin
     if ( DATA_OEW(1)  = '1' ) then  
       MEMD_O(8 to 15)  <= OUT_MEMD(8 to 15); 
     else 
       MEMD_O(8 to 15)  <= (others => 'Z');
     end if;      
 end process;

MEM_DATA_BUS2:
 process ( DATA_OEW, OUT_MEMD )
  begin
     if ( DATA_OEW(2) = '1' ) then  
       MEMD_O(16 to 23)  <= OUT_MEMD(16 to 23); 
     else 
       MEMD_O(16 to 23)  <= (others => 'Z');
     end if;      
 end process;

MEM_DATA_BUS3:
 process ( DATA_OEW, OUT_MEMD )
  begin
     if ( DATA_OEW(3) = '1' ) then  
       MEMD_O(24 to 31)  <= OUT_MEMD(24 to 31); 
     else 
       MEMD_O(24 to 31)  <= (others => 'Z');
     end if;      
 end process;

MEM_DATA_BUS4:
 process ( DATA_OEW, OUT_MEMD )
  begin
     if ( DATA_OEW(4) = '1' ) then  
       MEMD_O(32 to 39)  <= OUT_MEMD(32 to 39); 
     else 
       MEMD_O(32 to 39)  <= (others => 'Z');
     end if;      
 end process;

MEM_DATA_BUS5:
 process ( DATA_OEW, OUT_MEMD )
  begin
     if ( DATA_OEW(5) = '1' ) then  
       MEMD_O(40 to 47)  <= OUT_MEMD(40 to 47); 
     else 
       MEMD_O(40 to 47)  <= (others => 'Z');
     end if;      
 end process;

MEM_DATA_BUS6:
 process ( DATA_OEW, OUT_MEMD )
  begin
     if ( DATA_OEW(6) = '1' ) then  
       MEMD_O(48 to 55)  <= OUT_MEMD(48 to 55); 
     else 
       MEMD_O(48 to 55)  <= (others => 'Z');
     end if;      
 end process;

MEM_DATA_BUS7:
 process ( DATA_OEW, OUT_MEMD )
  begin
     if ( DATA_OEW(7) = '1' ) then  
       MEMD_O(56 to 63)  <= OUT_MEMD(56 to 63); 
     else 
       MEMD_O(56 to 63)  <= (others => 'Z');
     end if;      
 end process;

MEM_DATA_BUSP:
 process ( DATA_OEW, OUT_MEMDP )
  begin
     if ( DATA_OEW(8) = '1' ) then  
       MEMDP_O(0 to 7)  <= OUT_MEMDP(0 to 7); 
     else 
       MEMDP_O(0 to 7)  <= (others => 'Z');
     end if;      
 end process;



-- Original code with 72 loads on DATA_OEW: 
--MEM_DATA_BUS:
-- process ( DATA_OEW, OUT_MEMD(0 to 7, OUT_MEMDP )
--  begin
  -- SRAM and FLASH memory data bus I/O
  -- If write cycle output DH, DL, DP on MEMD and MEMDP outputs.
--     if DATA_OEW = '1' then  
--       MEMD_O  <= OUT_MEMD; 
--       MEMDP_O <= OUT_MEMDP;
--     else 
--       MEMD_O  <= (others => 'Z');
--       MEMDP_O <= (others => 'Z');
--     end if;      
-- end process;

ASYNC_DFF:
process ( N_RESET, CLK )
--process (CLK, BUS_REQ, DONE, R_nW, BURST, N_RESET)
 begin
   if ( N_RESET='0') then	
     --asynchronous RESET
     OUT_DH    	<= (others => '0');
     OUT_DL    	<= (others => '0');
     OUT_DP    	<= (others => '0');
 	-- tristate 60x bus output driver.
     DATA_OE   	<= '0';  
	-- tristate memory output drivers.      
     DATA_OEW  	<= (others => '0');
     N_TA	   	<= '1';  
     N_TAO     	<= '1';

   elsif (CLK'event and CLK='1') then
   -- multiplexer for SRAM or FLASH data presentation to uP.
     if (BUS_REQ = '1') then
       -- use IOB sync'ed data 11/30
       OUT_DH    <=  MEMD_IN( 0 to 31);  
       OUT_DL    <=  MEMD_IN(32 to 63); 
       OUT_DP    <=  MEMDP_IN;
     else
       OUT_DH    <= FLASH_DB( 0 to 31);
       OUT_DL    <= FLASH_DB(32 to 63);
       OUT_DP    <= FLASH_DBP;
     end if;
            
   DATA_OE   	<=  R_nW and (( DONE_FLASH or FLASH_TA ) or (not N_MEMOE)); 

   DATA_OEW(0)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));
   DATA_OEW(1)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));
   DATA_OEW(2)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));
   DATA_OEW(3)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));
   DATA_OEW(4)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));
   DATA_OEW(5)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));
   DATA_OEW(6)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));
   DATA_OEW(7)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));
   DATA_OEW(8)  	<= (not R_nW) and ( (BUSREQ_FLASH and ((not N_TAO) or FLASH_TA )) 
                 	    or (BUS_REQ and (not DONE)));

   --********************************************************************************
   -- delay N_TA to line up with data valid on bus.
   -- replicate N_TAO and N_TA to avoid pad feedback into CLB for DATA_OE() creation.
   -- without duplicate logic N_TA goes to output pad and feeds into CLB for DATA_OE.
   -- which causes timing problems and potential board level signal concerns.
   --********************************************************************************
     if ( R_nW = '1' ) then
       N_TAO <= not( FLASH_TA or SRAM_TA );
	 N_TA  <= not( FLASH_TA or SRAM_TA );

     elsif (BURST = '1') then
       N_TAO <= not( BUS_REQ and (not(SRAM_TA or DONE)) );
       N_TA  <= not( BUS_REQ and (not(SRAM_TA or DONE)) );
     else
       N_TAO <= not( BUS_REQ and (not DONE) );
       N_TA  <= not( BUS_REQ and (not DONE) );

     end if; 
   end if;
 end process;

-- Add output delay to emulate switching characteristics of FPGA.
  DH    <= DH_O;
  DL    <= DL_O;
  DP    <= DP_O;
  MEMD  <= MEMD_O;
  MEMDP <= MEMDP_O;
  
  
IOB_DFFS:
process (CLK, N_RESET)
--*********************************************************************
-- Access time of SSRAM is 7.5ns
-- This only leaves 2.5ns (at 100MHz clock) setup time before CLK
-- Use IOB flip flop to store data at chip boundary before processing.
-- Send 16 most significant bits to flash_if to be assembled into words.
--**********************************************************************
 begin
   if N_RESET='0' then	
      --asynchronous RESET
     	DH_IN		<= (others => '0');
     	DL_IN		<= (others => '0');
     	DP_IN		<= (others => '0');
     	OUT_MEMD  	<= (others => '0');
     	OUT_MEMDP 	<= (others => '0');
     	MEMD_IN  	<= (others => '0');
     	MEMDP_IN 	<= (others => '0'); 
     
   elsif (CLK'event and CLK='1') then 
	-- Synchronize 60x data bus inputs
     	DH_IN		<= DH;
     	DL_IN		<= DL;
     	DP_IN		<= DP;
      -- delay DH, DL, DP one more clock cycle to align with control inputs that are
	-- also synchronized internally.     
	OUT_MEMD  	<= DH_IN & DL_IN; -- concatenate bus structures.
     	OUT_MEMDP 	<= DP_IN;
	-- synchronize memory data bus inputs before multiplexing with flash data.
     	MEMD_IN    	<=  MEMD;  
     	MEMDP_IN   	<=  MEMDP;
   end if;
end process;

-- route synchronized MEMD_IN to flash-if for data assmembly. 11/30 ST
FLASH_D_IN <= MEMD_IN(0 to 15);

  end DATABUS;
