--The IEEE standard 1164 package, declares std_logic, rising_edge(), etc.
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

--`timescale 1ns/1ns

entity DATAFLOW is
 port(
 	N_TS_DLY : 	in 	STD_LOGIC; -- Transfer start.
 	N_ARTRY :	in 	STD_LOGIC; -- Address retry asserted by bus snooper.
 	ADR_BRA :	in	STD_LOGIC; -- Address bus grant A and N_TS asserted.
 	ADR_BRB :	in	STD_LOGIC; -- Address bus grant B and N_TS asserted.
 	DATAXFR :	in	STD_LOGIC; -- Address decode indicates data bus required for transfer.
 	CLK :		in	STD_LOGIC; -- System clock (globally buffered).
 	N_RESET :	in	STD_LOGIC; -- System reset input.
 	N_DBB :	in	STD_LOGIC; -- Data bus busy (optional input).
 	BUS_REQ :	in	STD_LOGIC; -- SRAM bus request.
 	BUSREQ_FLASH :in	STD_LOGIC; -- Flash bus request.
 	DONE :	in	STD_LOGIC; -- SRAM access complete.
 	N_AACK :	out	STD_LOGIC; -- Address acknowledge.
 	N_DBGA :	out  	STD_LOGIC; -- Data bus grant uP_A.
 	N_DBGB :	out  	STD_LOGIC; -- Data bus grant uP_B. 
 	ENABLE_XFR :out  	STD_LOGIC  -- Move address from input stage to output stage.
      );
end DATAFLOW;


architecture DATAFLOW of DATAFLOW is

  type STATE_TYPE is (s0, s1, s2, s3, s4, s5);
  -- ONE HOT state machine state assignments:
  attribute ENUM_ENCODING: STRING;
  attribute ENUM_ENCODING of STATE_TYPE: type is "000001 000010 000100  001000 010000 100000";

  signal current : 	STATE_TYPE;
  signal N_ARTRY_IN :	STD_LOGIC;

begin

-- Syncrhonize address retry to provide higher bus clock rate capabilities.
IOB_SYNC_ARTRY_INPUT:
 process ( CLK, N_RESET)

 begin 
   if (N_RESET = '0') then
     N_ARTRY_IN <= '1';
   elsif (CLK'event and CLK = '1') then
     N_ARTRY_IN <= N_ARTRY;
   end if;
 end process;

-- If a bus cycle is in progress, don't assert N_AACK until bus cycle done.

STATE_MACHINE:
process ( CLK, N_RESET)

begin 
  if (N_RESET = '0') then
    current <= s0; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';
  elsif (CLK'event and CLK = '1') then

  case (current) is

  when s0 =>   
     if( ((not N_TS_DLY) and not(BUS_REQ or BUSREQ_FLASH) ) ='1' )then
     -- Start of address bus tenure. Proceed to state 1.
       current <= s1; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';                   
     elsif ( ((not N_TS_DLY) and (BUS_REQ or BUSREQ_FLASH) and ADR_BRA ) = '1' ) then
     -- Start of address bus tenure. Data bus is busy.
       current <= s4; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';        
     elsif (( (not N_TS_DLY) and (BUS_REQ or BUSREQ_FLASH) and ADR_BRB ) = '1' ) then 
     -- Start of address bus tenure. Data bus is busy.
       current <= s5; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';                
     else
     -- wait for a next address bus tenure start.
       current <= s0; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';
     end if;       

  when s1 =>
   -- Need information from address arbiter to determine which
   -- data bus grant to issue. Use ADR_BRA and ADR_BRB inputs.
     if (( DATAXFR and not(BUS_REQ or BUSREQ_FLASH) ) = '1' ) then
     -- Address decoder determined data bus transaction needed.
     -- SRAM and Flash state machines are not busy processing a bus request.
     -- Assert data bus grant to address bus master now.
       current <= s2; ENABLE_XFR <= '0'; N_DBGA <= ADR_BRB; N_DBGB <= ADR_BRA; N_AACK <= '0';        
     else        
     -- Address only transaction.
     -- Wait for the next address bus tenure to start.
       current <= s0; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '0';         
     end if;   

  when s2 =>
   -- Deassert data bus grant and address acknowledge.
   -- This is the clock that N_ARTRY gets asserted by bus snooper.
   --****************************************************************
   -- Note: if address retry support is NOT required, the ENABLE_XFR
   -- signal could be asserted in state:s1 and the memory latency
   -- would be reduced by one clock period. Since ARTRY is supported
   -- for this design the memory request queue will not be filled
   -- until after the ARTRY clock phase.  
   --****************************************************************
     if ( N_ARTRY_IN = '0' ) then
     -- Bus Snooper has this location in cache.
     -- Abort data transfer and return to IDLE.
     -- Cancel pending data transfer in queue.
     current <= s0; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';        
     else
     -- move address qualifiers to output stage.
     -- make room for next address qualifer at input stage.
       current <= s3; ENABLE_XFR <= '1'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';
     end if;

  when s3 =>
     if(( (not N_TS_DLY) and ADR_BRA) = '1' ) then
     -- Start of pipelined address bus tenure.
     -- Can't assert N_AACK until N_TA asserted for data tenure.
       current <= s4; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';         
     elsif (( (not N_TS_DLY) and ADR_BRB) = '1' ) then
     -- Start of pipelined address bus tenure.
     -- Can't assert N_AACK until N_TA asserted for data tenure.
       current <= s5; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';                    
     else
     -- wait for address bus tenure to start.
       current <= s3; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';
     end if;       

  when s4 =>
  -- Waiting for data transfer to complete before asserting N_AACK and N_DBGA
  -- data bus grant to issue. If the data bus is busy, wait until the cycle has
  -- finished before asserting address acknowledge and data bus grant. If the
  -- cycle is an address only transaction, return to IDLE state, without waiting
  -- for data tenure to complete. Need to check busy at exit of s0 to inhibit
  -- further transactions if busy.
     if ( DATAXFR = '0' ) then
     -- address only transaction. No data bus tenure required. Return to IDLE.
       current <= s0; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '0';   
     elsif (( DATAXFR and not(BUS_REQ or BUSREQ_FLASH) )  = '1' ) then
     -- Address decoder determined data bus transaction needed.
     -- SRAM and Flash state machines are not busy processing a bus request.
     -- Assert data bus grant to address bus master now.
       current <= s2; ENABLE_XFR <= '0'; N_DBGA <= '0'; N_DBGB <= '1'; N_AACK <= '0';        
     else  
     -- wait for data bus tenure to finish.
       current <= s4; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';
     end if;       

  when s5 =>
  -- Waiting for data transfer to complete before asserting N_AACK and N_DBGB
  -- data bus grant. Data bus grant can only be asserted one clock prior to
  -- start of processor driving bus. Must wait for bus request to deassert. If the
  -- cycle is an address only transaction, return to IDLE state, without waiting
  -- for data tenure to complete. Need to check busy at exit of s0 to inhibit
  -- further transactions if busy.
    if( DATAXFR = '0' ) then
    -- address only transaction. No data bus tenure required. Return to IDLE.
       current <= s0; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '0';  
    elsif (( DATAXFR and  not(BUS_REQ or BUSREQ_FLASH) ) = '1' ) then
    -- Address decoder determined data bus transaction needed.
    -- SRAM and Flash state machines are not busy processing a bus request.
    -- Assert data bus grant to address bus master now.
       current <= s2; ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '0'; N_AACK <= '0';        
    else  
    -- wait for data bus tenure to finish.
      current <= s5;  ENABLE_XFR <= '0'; N_DBGA <= '1'; N_DBGB <= '1'; N_AACK <= '1';
    end if;

  end case;
end if;
  
end process;

end DATAFLOW;
