-- The IEEE standard 1164 package, declares std_logic, rising_edge(), etc.
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

--`timescale 1ns/1ns
entity FLASH_IF is
   port(
	 CLK :		in 	STD_LOGIC;  -- Buffered system clock.
	 N_RESET : 		in 	STD_LOGIC;  -- Power on reset to initialize chip.
	 R_nW :		in 	STD_LOGIC;  -- Read not write inferred from TT[1]. 
	 BUSREQ_FLASH :	in 	STD_LOGIC;  -- Bus Request for Flash memory access.
	 --MEMD :		in	STD_LOGIC_VECTOR(0 to 15); -- Flash data bus read input
	 FLASH_D_IN :	in	STD_LOGIC_VECTOR(0 to 15); -- synchronized IOB DFF 11/30
	 DONE_FLASH :	buffer STD_LOGIC; -- Flash bus cycle completed.
	 N_FLASHCE : 	out 	STD_LOGIC; -- Flash chip enable.
	 N_FLASHWE : 	out 	STD_LOGIC; -- Flash write enable.
	 N_FLASHOE : 	out 	STD_LOGIC; -- Flash  enable.
	 FLASH_TA :		out 	STD_LOGIC; -- Flash transfer acknowledge signal.
	 FLASH_ADR :	out	STD_LOGIC_VECTOR(0 to 1);  -- Flash address bits 0:1 (FLASH_ADR[2]=ADR[29])
	 FLASH_DB :		buffer STD_LOGIC_VECTOR(0 to 63); -- Flash double word data input to uP
	 FLASH_DBP :	out	STD_LOGIC_VECTOR(0 to 7)   -- Odd byte lane parity generated for flash reads.
        );
end FLASH_IF;

architecture FLASH_IF of FLASH_IF is

  type STATE_TYPE is (s0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11);
  -- ONE HOT state machine state assignments:
  attribute ENUM_ENCODING: STRING;
  attribute ENUM_ENCODING of STATE_TYPE: type is "000000000001 000000000010 000000000100 000000001000 000000010000 000000100000 000001000000 000010000000 000100000000 001000000000 010000000000 100000000000";

  signal current:  STATE_TYPE;

  signal DIN :	STD_LOGIC_VECTOR(0 to 63) ;
  signal WHAT: 	STD_LOGIC_VECTOR(0 to  2); 
  signal CNT :	STD_LOGIC_VECTOR(0 to  1);
  signal CNT_UP, LOAD, RD_EN, CNT_THREE:   STD_LOGIC;

begin
-- PowerPC interface to flash memory. This design assumes a flash
-- memory with 70ns access times. The specific device used in the
-- design is an AMD29LV400B, organized 512Kx8 or 256Kx16 (Width of
-- flash is selected by strapping BYTE# flash input low or high).

STATE_MACHINE:
process ( N_RESET, CLK )
begin
-- Mealy state machine for control of boot flash.
if ( N_RESET = '0' ) then
  current <= s0;  DONE_FLASH <= '0';  LOAD<= '1'; CNT_UP <= '0';         
  N_FLASHCE <= '1'; N_FLASHOE <= '1'; N_FLASHWE <= '1';  RD_EN <= '0';
elsif ( CLK'event and CLK='1' ) then

  case (current) is

  when s0 =>
    if(( R_nW and BUSREQ_FLASH ) = '1' ) then
    -- Start access. Addresses are stable before asserting CE & OE!
      current <= s1;  DONE_FLASH <= '0';  LOAD <= '1'; CNT_UP<= '0'; 
      N_FLASHCE <= '0'; N_FLASHOE <= '0'; N_FLASHWE <= '1';  RD_EN <= '0';    
    elsif(( (not R_nW) and BUSREQ_FLASH ) = '1' ) then
    -- Addresses must be stable before asserting CE & OE.
      current <= s1;  DONE_FLASH <= '0';  LOAD <= '1'; CNT_UP<= '0'; 
      N_FLASHCE <= '0'; N_FLASHOE <= '1'; N_FLASHWE <= '1';  RD_EN <= '0';
    else    
    -- wait for a valid bus request.
      current <= s0;  DONE_FLASH <= '0'; LOAD <= '1'; CNT_UP<= '0';   
      N_FLASHCE <= '1'; N_FLASHOE <= '1'; N_FLASHWE <= '1';  RD_EN <= '0';
    end if;
   
  when s1 =>    
  -- Delay 1 clk from CE to OE to meet Toeh after a write cycle.
    current <= s2;  DONE_FLASH <= '0';  LOAD <= '0'; CNT_UP<= '0';   
    N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <= R_nW;  RD_EN <= '0';
      
  when s2 =>
    current <= s3;  DONE_FLASH <= '0';  LOAD <= '0'; CNT_UP<= '0';       
    N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <= R_nW; RD_EN <= '0';
   
  when s3 =>
    current <= s4;  DONE_FLASH <= '0';  LOAD <= '0'; CNT_UP<= '0';      
    N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <=  R_nW;  RD_EN <= '0';         
   
  when s4 =>
    current <= s5;  DONE_FLASH <= '0'; LOAD <= '0'; CNT_UP<= '0';          
    N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <= R_nW;  RD_EN <= '0';     
   
  when s5 =>
    current <= s6;  DONE_FLASH <= '0';  LOAD <= '0'; CNT_UP<= '0';      
    N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <= R_nW;  RD_EN <= '0';

  when s6 =>
    current <= s7;  DONE_FLASH <= '0';  LOAD <= '0'; CNT_UP<= '0';      
    N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <= R_nW;  RD_EN <= '1';

  when s7 =>
    if ( CNT_THREE = '1' ) then
    -- Assert DONE for completion of flash access cycle.
    -- Deasert chip enable, output enable, write enable.
      current <= s8; DONE_FLASH <= '1'; LOAD <= '0'; CNT_UP<= '0';      
      N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <= R_nW; RD_EN <= '0';
    else
      current <= s8;  DONE_FLASH <= '0'; LOAD <= '0'; CNT_UP<= '0';      
      N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <= R_nW;  RD_EN <= '0';
    end if;        
   
  when s8 =>
   if ( CNT_THREE = '1' ) then 
   -- Assert DONE for completion of flash access cycle.
   -- Deasert chip enable, output enable, write enable.
     current <= s0; DONE_FLASH <= '0'; LOAD <= '0'; CNT_UP<= '0';      
     N_FLASHCE <= '1'; N_FLASHOE <= '1'; N_FLASHWE <= '1'; RD_EN <= '0';
   else 
   -- After first read cycle enable counter to count up now.
   -- Need next address to assemble 64-bit word for read from flash.
   -- Give the counter one clock cycle to increment before accessing flash again.
     current <= s9; DONE_FLASH<= '0'; LOAD <= '0'; CNT_UP <= '1';
     N_FLASHCE <= '1'; N_FLASHOE <= '1'; N_FLASHWE <= '1'; RD_EN <= '0';
   end if;
   
  when s9 =>
   -- Flash output goes high-Z 25ns after N_FLASHOE deasserted.
   -- Don't enable flash until output driver has turned off.    
   -- Wait 30ns (at 100MHz SYS_CLK) rate.
   -- Delay needs adjusted as SYS_CLK frequency changes.  
     current <= s10; DONE_FLASH <= '0'; LOAD <= '0'; CNT_UP <= '0';
     N_FLASHCE <= '1'; N_FLASHOE <= '1'; N_FLASHWE <= '1'; RD_EN <= '0';

  when s10 =>
    current <= s11; DONE_FLASH <= '0'; LOAD <= '0'; CNT_UP <= '0';
    N_FLASHCE <= '1'; N_FLASHOE <= '1'; N_FLASHWE <= '1'; RD_EN <= '0';

  when s11 =>
    current <= s1; DONE_FLASH <= '0'; LOAD <= '0'; CNT_UP <= '0';
    N_FLASHCE <= '0'; N_FLASHOE <= not R_nW; N_FLASHWE <= R_nW; RD_EN <= '0';

  end case;
end if;
end process;         


COUNTER:
--*********************************************************************
-- The counter interfaces with the flash_if state machine and keeps 
-- track of the number of times the state machine has accessed flash 
-- during a bus request. The flash interface must form a 32 or 64-bit 
-- word to present to the processor during boot time. The counter
-- provides the least significant address bits to the byte/halfword 
-- organized flash (depending upon external system implementation).
--
-- The counter provides a mechanism to steer the byte or halfword
-- flash data to the correct byte lanes for the PowerPC750 read:
--
-- B0..B1..B2..B3..B4..B5..B6..B7  PowerPC byte format
-- ..00......01......02......03..  CNT incrementing: 16-bit flash
-- 00..01..02..03..04..05..06..07  CNT incrementing:  8-bit flash 
--
-- Note: The MEMA outputs are aligned to a 8 byte boundary.
-- For 16-bit flash: use A[29] as 3rd least significant address bit.
--*********************************************************************
process ( N_RESET, CLK )
 begin
 if ( N_RESET = '0' ) then CNT <= "00";  
 elsif ( CLK'event and CLK ='1' ) then 
   if ( LOAD = '1' ) then 
     CNT <= "00";
   elsif ( CNT_UP = '1' ) then 
     CNT <= CNT + '1';
   else 
     CNT <= CNT;
   end if;
 end if;  
end process;

 FLASH_ADR <= CNT ;
 CNT_THREE <= CNT(1) and CNT(0);
 WHAT <= RD_EN & CNT(0) & CNT(1);


FLASH_DATA_ASSEMBLER:
--********************************************************************
-- This process stores each sequential 16-bit read cycle in a holding
-- register until the entire double word has been assembled. Once
-- assembled Transfer Acknowledge (N_TA) can be asserted to the uP.
-- Note: ODD PARITY is generated on the byte lanes!
--*********************************************************************
process( WHAT, FLASH_D_IN, FLASH_DB )
 begin

 case ( WHAT ) is
 -- Note: counter bits are big endian format. Bit 0 = most significant bit.

 when "100" =>
   DIN( 0 to 15) <=  FLASH_D_IN; 
   DIN(16 to 31) <=  FLASH_DB(16 to 31);
   DIN(32 to 47) <=  FLASH_DB(32 to 47);
   DIN(48 to 63) <=  FLASH_DB(48 to 63);
 when "101" =>
   DIN( 0 to 15) <=  FLASH_DB( 0 to 15); 
   DIN(16 to 31) <=  FLASH_D_IN ;
   DIN(32 to 47) <=  FLASH_DB(32 to 47);
   DIN(48 to 63) <=  FLASH_DB(48 to 63);
 when "110" =>
   DIN( 0 to 15) <=  FLASH_DB( 0 to 15); 
   DIN(16 to 31) <=  FLASH_DB(16 to 31);
   DIN(32 to 47) <=  FLASH_D_IN;
   DIN(48 to 63) <=  FLASH_DB(48 to 63);
 when "111" =>
   DIN( 0 to 15) <=  FLASH_DB( 0 to 15); 
   DIN(16 to 31) <=  FLASH_DB(16 to 31);
   DIN(32 to 47) <=  FLASH_DB(32 to 47);
   DIN(48 to 63) <=  FLASH_D_IN ;

 when others => DIN <= FLASH_DB;
 end case;

end process;


DFF:
process( N_RESET, CLK )
 begin
 if (N_RESET = '0') then
   FLASH_DB <= (others => '0' ); 
   FLASH_TA <= '0';
 elsif ( CLK'event and CLK='1' ) then 
 --  # 2 emulate output switching delays.
   FLASH_DB <= DIN;
   FLASH_TA <= DONE_FLASH;  
 end if;
end process; -- DFF

ODD_PARITY_GEN:
process( FLASH_DB )
 begin
 --******************************************************************** 
 -- Generate odd parity on flash byte lanes.
 -- SRAM stores odd parity with data. May need a datapath multiplexor
 -- to select parity generation(flash) or flowthrough (SRAM).
 --*********************************************************************
 
 FLASH_DBP(0) <= not( FLASH_DB(0) xor FLASH_DB(1) xor FLASH_DB(2) xor FLASH_DB(3) xor
 		      FLASH_DB(4) xor FLASH_DB(5) xor FLASH_DB(6) xor FLASH_DB(7));

 FLASH_DBP(1) <= not( FLASH_DB(8) xor FLASH_DB(9) xor FLASH_DB(10) xor FLASH_DB(11) xor
 		      FLASH_DB(12) xor FLASH_DB(13) xor FLASH_DB(14) xor FLASH_DB(15));

 FLASH_DBP(2) <= not( FLASH_DB(16) xor FLASH_DB(17) xor FLASH_DB(18) xor FLASH_DB(19) xor
 		      FLASH_DB(20) xor FLASH_DB(21) xor FLASH_DB(22) xor FLASH_DB(23));

 FLASH_DBP(3) <= not( FLASH_DB(24) xor FLASH_DB(25) xor FLASH_DB(26) xor FLASH_DB(27) xor
 		      FLASH_DB(28) xor FLASH_DB(29) xor FLASH_DB(30) xor FLASH_DB(31));

 FLASH_DBP(4) <= not( FLASH_DB(32) xor FLASH_DB(33) xor FLASH_DB(34) xor FLASH_DB(35) xor
 		      FLASH_DB(36) xor FLASH_DB(37) xor FLASH_DB(38) xor FLASH_DB(39));

 FLASH_DBP(5) <= not( FLASH_DB(40) xor FLASH_DB(41) xor FLASH_DB(42) xor FLASH_DB(43) xor
 		      FLASH_DB(44) xor FLASH_DB(45) xor FLASH_DB(46) xor FLASH_DB(47));

 FLASH_DBP(6) <= not( FLASH_DB(48) xor FLASH_DB(49) xor FLASH_DB(50) xor FLASH_DB(51) xor
 		      FLASH_DB(52) xor FLASH_DB(53) xor FLASH_DB(54) xor FLASH_DB(55));

 FLASH_DBP(7) <= not( FLASH_DB(56) xor FLASH_DB(57) xor FLASH_DB(58) xor FLASH_DB(59) xor
 		      FLASH_DB(60) xor FLASH_DB(61) xor FLASH_DB(62) xor FLASH_DB(63));

end process;   
end FLASH_IF;
