--The IEEE standard 1164 package, declares std_logic, rising_edge(), etc.
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
--`timescale 1ns/1ns
entity SRAMCTL is
port (
	N_RESET :   in    STD_LOGIC; -- Power on reset to initialize chip logic.
	RnW :    	in    STD_LOGIC; -- Read not write derived from TT[4:0] bus qualifiers.
	BUSREQ :    in    STD_LOGIC; -- Active bus request from pipeline controller.
	BURST :    	in    STD_LOGIC; -- 60X burst (fixed: @4/8 transfers of 64/32 bits).
	BWE :   	in    STD_LOGIC_VECTOR(0 to 7) ;  -- Byte writes derived from TBST & TSIZ[0:2].
	CLK :    	in    STD_LOGIC; -- buffered SYS_CLK.
	N_CE :    	out    STD_LOGIC; -- SSRAM chip enable.
 	N_OE :    	out    STD_LOGIC; -- SSRAM  enable.
	N_WE :    	out    STD_LOGIC_VECTOR(0 to 7); -- SSRAM byte write enables.
	N_ADV :    	out    STD_LOGIC; -- SSRAM advance for read bursting.
	N_ADSC :    out    STD_LOGIC; -- SSRAM address strobe.
	DONE :    	out    STD_LOGIC; -- Signal that SRAM bus cycle has completed. 
	SRAM_TA :   out    STD_LOGIC  -- Signal that SRAM data transfer beat has completed. 
      );
end SRAMCTL;


architecture SRAMCTL of SRAMCTL is

--*********************************************************************
-- Mealy state machine with synchronous outputs.
-- Controls 1 bank of synchronous SRAM memory (256Kx72 bits).
-- Not implemented: PPC603e configured as a 32-bit bus device. 
--
-- To support a cache line read burst that doesn't start on a cache line
-- boundary, the MODE input of the SSRAM must a logic 0 to select linear 
-- burst addressing. Linear address mode works as follows:
--
-- If the External Address ADR[1:0] is:
-- ADR[1:0] = 0x00, burst internal address = 0x01..0x10..0x11.
-- ADR[1:0] = 0x01, burst internal address = 0x10..0x11..0x00.
-- ADR[1:0] = 0x10, burst internal address = 0x11..0x00..0x01.
-- ADR[1:0] = 0x11, burst internal address = 0x00..0x01..0x10.
--
--*********************************************************************

-- Module declarations:
-- ONE HOT state machine state assignments
  type STATE_TYPE is (s0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10);
  -- ONE HOT state machine state assignments:
  attribute ENUM_ENCODING: STRING;
  attribute ENUM_ENCODING of STATE_TYPE: type is 
     "00000000001 00000000010 00000000100 00000001000 00000010000 00000100000 00001000000 00010000000 00100000000 01000000000 10000000000";

  signal current:  STATE_TYPE;    

begin

STATE_MACHINE:
process ( CLK, N_RESET )
begin
-- Start state machine at positive edge of system clock (CLK).
if ( N_RESET = '0' ) then
  current <= s0;  DONE <= '0';  SRAM_TA <= '0';        
  N_CE <= '1'; N_OE <= '1'; N_ADV <= '1'; N_ADSC <= '1'; N_WE <= "11111111";
elsif ( CLK'event and CLK = '1' ) then
  
  case(current) is

  when s0 =>
     if(( BUSREQ and RnW) = '1' ) then
     -- Start a read cycle. Pipeline controller issues valid BUSREQ.
     -- Assert N_OE at same time to get valid data on next clock.
       current <= s1; DONE <= '0';  SRAM_TA <= '0'; 
       N_CE <= '0'; N_OE <= '0'; N_ADV <= '1'; N_ADSC <= '0'; N_WE <= "11111111";             
     elsif(( BUSREQ and (not RnW) and (not BURST)) = '1' ) then
     -- Start a single beat write cycle.
       current <= s6; DONE <= '1'; SRAM_TA <= '1';   
       N_CE <= '0'; N_OE <= '1'; N_ADV <= '1'; N_ADSC <= '0'; N_WE <= not BWE;        
     elsif (( BUSREQ and (not RnW ) and BURST) = '1' ) then
     -- Start a four beat write cycle. All byte lanes asserted.
       current <= s7; DONE <= '0';  SRAM_TA <= '0'; 
       N_CE <= '1'; N_OE <= '1'; N_ADV <= '1'; N_ADSC <= '1'; N_WE <= "11111111";                
     else
     -- wait for a valid bus request.
       current <= s0; DONE <= '0';  SRAM_TA <= '0';
       N_CE <= '1'; N_OE <= '1'; N_ADV <= '1'; N_ADSC <= '1'; N_WE <= "11111111";         
     end if;       
   
  when s1 =>
     if( BURST = '1') then               
     -- Start a four beat burst read ( 8 bytes per beat).
       current <= s2; DONE <= '0'; SRAM_TA <= '1';
       -- External address selects starting address for read burst.
       -- After initial address is latched, use SSRAM internal address counter.    
       N_CE <= '1'; N_OE <= '0'; N_ADV <= '0'; N_ADSC <= '1'; N_WE <= "11111111";              
     else
     -- Start a single beat read ( 8 bytes ).
     -- External address selects which location to read.
       current <= s5; DONE <= '1'; SRAM_TA <= '1';     
       N_CE <= '1'; N_OE <= '0'; N_ADV <= '1'; N_ADSC <= '1'; N_WE <= "11111111";               
     end if;
   
  when s2 =>
     -- continue 4 beat read cycle.
     current <= s3; DONE <= '0'; SRAM_TA <= '1';
     -- deassert chip enable, assert output enable.
     -- assert advance signal to increment internal SSRAM address counter.           
     N_CE <= '1'; N_OE <= '0'; N_ADV <= '0'; N_ADSC <= '1'; N_WE <= "11111111"; 
   
  when s3 =>
     -- continue 4 beat read cycle.
     current <= s4; DONE <= '0'; SRAM_TA <= '1';
     -- increment internal SSRAM address counter.           
     N_CE <= '1'; N_OE <= '0'; N_ADV <= '0'; N_ADSC <= '1'; N_WE <= "11111111";  
   
  when s4 =>
     -- continue 4 beat read cycle.
     current <= s5; DONE <= '1'; SRAM_TA <= '1';         
     -- assert advance signal to increment internal SSRAM address counter.
     N_CE <= '1'; N_OE <= '0'; N_ADV <= '1'; N_ADSC <= '1'; N_WE <= "11111111";    
   
  when s5 =>
     -- continue 4 beat read cycle or deassert OE for single beat read.
     current <= s0; DONE <= '0';  SRAM_TA <= '0';       
     -- deassert advance signal to complete 4 beat burst.
     N_CE <= '1'; N_OE <= '1'; N_ADV<= '1'; N_ADSC <= '1'; N_WE <= "11111111";
     -- SRAM read cycle has completed. Ready for current bus access.      


-- WRITE STATE MACHINE STATEMENTS FOLLOW:
  
  when s6 =>
     -- Single beat write cycle complete. Deassert CE, OE & return to IDLE.
     current <= s0; DONE <= '0'; SRAM_TA <= '0';        
     N_CE <= '1'; N_OE <= '1'; N_ADV<= '1'; N_ADSC <= '1'; N_WE <= "11111111";
     -- SRAM write cycle has completed. Return to IDLE and wait for next request.          

  when s7 =>
     -- Second beat of four beat write cycle, deassert OE for write cycles.
     current <= s8; DONE <= '0'; SRAM_TA <= '0';       
     N_CE <= '0'; N_OE <= '1'; N_ADV<= '1'; N_ADSC <= '0'; N_WE <= "00000000";     

  when s8 =>
     -- Third beat of four beat read cycle.
     current <= s9; DONE <= '0'; SRAM_TA <= '0';     
     N_CE <= '1'; N_OE <= '1'; N_ADV<= '0'; N_ADSC <= '1'; N_WE <= "00000000";   

  when s9 =>
     -- Fourth beat of four beat read cycle.
     -- SRAM_TA is used to deassert N_TA on next clock edge.
     current <= s10; DONE <= '0'; SRAM_TA <= '1';       
     N_CE <= '1'; N_OE <= '1'; N_ADV<= '0'; N_ADSC <= '1'; N_WE <= "00000000";

  when s10 =>
     -- Assert DONE for completion of four beat write cycle.
     current <= s6; DONE <= '1';  SRAM_TA <= '0';     
     N_CE <= '1'; N_OE <= '1'; N_ADV<= '0'; N_ADSC <= '1'; N_WE <= "00000000";
     -- SRAM write cycle has completed. Ready for next bus access.     

  end case;
end if;
end process;

end SRAMCTL; 
