library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

-- pragma translate_off
library unisim;
use unisim.vcomponents.all;
-- pragma translate_on


entity top_60x is
  port (
	N_RESET :	in	STD_LOGIC;  -- system reset input.
	SYS_CLK :	in	STD_LOGIC;  -- System clock input.
 	N_BRA :	in	STD_LOGIC;  -- uP_A address bus request.
 	N_BRB :	in	STD_LOGIC;  -- uP_B address bus request. 
 	N_TS :	in	STD_LOGIC;  -- Transfer start asserted by bus master.
 	N_TBST :	in	STD_LOGIC; -- Transfer burst.
 	N_ARTRY :	in	STD_LOGIC; -- Address retry asserted by address bus snooper.
	N_DBB :	in	STD_LOGIC; -- Data bus busy (not used).
	
 	A :		in	STD_LOGIC_VECTOR(0 to 31); -- Processor address bus.
 	TT :		in	STD_LOGIC_VECTOR(0 to 4); -- Transfer type bits.
 	TSIZ :	in	STD_LOGIC_VECTOR(0 to 2); -- Transfer size.

	DH : 		inout STD_LOGIC_VECTOR(0 to 31);  -- Most significant uP data bus word.
	DL : 		inout STD_LOGIC_VECTOR(0 to 31);  -- Least significant uP data bus word.
	DP : 		inout STD_LOGIC_VECTOR(0 to 7);   -- Odd data parity.
	MEMD : 	inout STD_LOGIC_VECTOR(0 to 63);  -- SSRAM data bus to/from external memory.
	MEMDP : 	inout STD_LOGIC_VECTOR(0 to 7);   -- SSRAM data parity to/from external memory.

	N_BGA :	out	STD_LOGIC; -- Address bus grant for uP_A.
	N_BGB :	out	STD_LOGIC; -- Address bus grant for uP_B. 
	N_AACK :	out	STD_LOGIC; -- Address acknowledge to address bus master.
	N_DBGA :	out	STD_LOGIC; -- Data bus grant for uP_A.
	N_DBGB :	out	STD_LOGIC; -- Data bus grant for uP_B.
	N_TA :	out	STD_LOGIC; -- Data transfer acknowledge. Each data beat.
	N_DRTRY :	out	STD_LOGIC; -- Retry data transfer.
	N_TEA :	out	STD_LOGIC; -- Data transfer error acknowledge. Assert with N_TA.

	BANKSEL:	out	STD_LOGIC_VECTOR(0 to 1) ; -- SRAM bank selection (address bits A[9:10]).
	MEMA :	out	STD_LOGIC_VECTOR(0 to 17); -- SSRAM memory address.
	N_MEMBEW :	out	STD_LOGIC_VECTOR(0 to 7);  -- SSRAM byte write enables.

	N_MEMCE : 	out	STD_LOGIC; -- SSRAM chip enable.
	N_MEMOE :	buffer STD_LOGIC; -- SSRAM output enable.   
	N_MEMADV :	out	STD_LOGIC; -- SSRAM advance internal address counter.
	N_MEMADSC :	out	STD_LOGIC; -- SSRAM store external address.

	N_FLASHCE :	out	STD_LOGIC; -- Flash memory chip enable.
	N_FLASHOE :	out	STD_LOGIC; -- Flash memory output enable.
	N_FLASHWE :	out	STD_LOGIC; -- Flash memory write enable.

	FLASH_ADR :	out	STD_LOGIC_VECTOR(0 to 1) -- Flash address bits (0:1).
    );
end TOP_60X;

architecture TOP_60X of TOP_60X is

-- This is the top level VHDL module for the PPC_60X bus reference design.
-- It assumes an external flash memory and external SSRAM memory.
-- The bus structures are all big endian format.

-- DLL clock signals: 
 signal SYSCLK_ibufg : 	std_logic; 
 signal CLK0_dll : 	std_logic;
 signal RST_in :		std_logic;

-- Signal declarations:
  signal ADR_BRA :	STD_LOGIC;
  signal ADR_BRB :	STD_LOGIC;
  signal BURST :		STD_LOGIC;
  signal CLK :		STD_LOGIC;
  signal R_nW :		STD_LOGIC;
  signal BUS_REQ :	STD_LOGIC;
  signal BUSREQ_FLASH :	STD_LOGIC;
  signal DONE :		STD_LOGIC;
  signal DONE_FLASH :	STD_LOGIC;
  signal FLASH_TA :	STD_LOGIC;
  signal ENABLE_XFR:	STD_LOGIC;
  signal DATAXFR :	STD_LOGIC;
  signal SRAM_TA :	STD_LOGIC;  
  signal BWE :		STD_LOGIC_VECTOR(0 to 7);
  signal FLASH_DB :	STD_LOGIC_VECTOR(0 to 63);
  signal FLASH_DBP :	STD_LOGIC_VECTOR(0 to 7);

  -- add synchronized IOB DFF memory data for timing improvement:
  signal FLASH_D_IN : 	STD_LOGIC_VECTOR(0 to 15);
  signal N_TS_DLY :	STD_LOGIC;

--*******************************
-- Component declarations follow:
--*******************************

-- DLL Clock circuit:
 component IBUFG 
    port ( 
          I : in 	std_logic; 
          O : out std_logic
         ); 
 end component; 

 component BUFG 
    port ( 
          I : in 		std_logic; 
          O : buffer 	std_logic
         ); 
 end component; 

 component CLKDLL 
    port ( 
          CLKIN : 	in std_logic; 
          CLKFB : 	in std_logic; 
          RST : 		in std_logic; 
          CLK0 : 		out std_logic; 
          CLK90 : 	out std_logic; 
          CLK180 : 	out std_logic; 
          CLK270 : 	out std_logic; 
          CLKDV : 	out std_logic; 
          CLK2X : 	out std_logic; 
          LOCKED :	out std_logic
         ); 
 end component; 

-- Address bus arbiter state machine:
component ADR_ARB is
   port (
   	CLK :		in	STD_LOGIC;
   	N_RESET : 	in	STD_LOGIC;
      N_TS_DLY :	in	STD_LOGIC;
      N_BRA :	in	STD_LOGIC;
      N_BRB :	in	STD_LOGIC; 
      ADR_BRA:	out	STD_LOGIC; 
      ADR_BRB :	out	STD_LOGIC; 
      N_BGA :	out	STD_LOGIC; 	
      N_BGB :	out	STD_LOGIC   	
   	) ;
end component ADR_ARB ;

-- Address decoder and address pipeline logic:
component ADRPIPE is
   port(
   	A :		in	STD_LOGIC_VECTOR(0 to 31); 
	N_TS :	in	STD_LOGIC; 
	TT :		in	STD_LOGIC_VECTOR	(0 to 4); 
	TSIZ :	in	STD_LOGIC_VECTOR	(0 to 2); 
	N_TBST :	in	STD_LOGIC; 
      CLK :		in	STD_LOGIC; 
      N_RESET :	in	STD_LOGIC;
      ENABLE_XFR :in	STD_LOGIC;
      DONE :	in	STD_LOGIC;
      DONE_FLASH :in	STD_LOGIC;
      N_TS_DLY :	out	STD_LOGIC;
      BURST :	out	STD_LOGIC;  
      DATAXFR :	out	STD_LOGIC;
      BUS_REQ :	out	STD_LOGIC;
      BUSREQ_FLASH :out	STD_LOGIC;
      R_nW :	out	STD_LOGIC;
      BANKSEL :	out	STD_LOGIC_VECTOR(0 to 1);
      MEMA :	out	STD_LOGIC_VECTOR	(0 to 17);                   
      BWE :		out	STD_LOGIC_VECTOR(0 to 7)
       );
end component ADRPIPE;

-- Databus (60X and memory) tristate I/O control:
-- BUFFER is used where signal is used in feedback path.
component DATABUS is
   port (
       CLK :		in	STD_LOGIC;
       N_RESET :		in	STD_LOGIC;
       BURST :		in	STD_LOGIC;
	 R_nW :		in	STD_LOGIC;
	 BUS_REQ :		in	STD_LOGIC;
	 BUSREQ_FLASH :	in	STD_LOGIC; 
	 FLASH_TA :		in	STD_LOGIC;
	 SRAM_TA :		in	STD_LOGIC;
	 DONE :		in	STD_LOGIC;
	 DONE_FLASH :	in	STD_LOGIC;
       N_MEMOE :		in	STD_LOGIC;
       N_TA :		out	STD_LOGIC;
       FLASH_DB :		in	STD_LOGIC_VECTOR(0 to 63);
       FLASH_DBP :	in	STD_LOGIC_VECTOR(0 to 7);
       DH : 		inout	STD_LOGIC_VECTOR(0 to 31);
       DL :			inout	STD_LOGIC_VECTOR(0 to 31);
       DP :			inout	STD_LOGIC_VECTOR(0 to 7);
       FLASH_D_IN :	out	STD_LOGIC_VECTOR(0 to 15);
       MEMD :		inout	STD_LOGIC_VECTOR(0 to 63);
       MEMDP :		inout	STD_LOGIC_VECTOR(0 to 7)
        );
end component DATABUS;

-- Data bus arbitration state machine.
component DATAFLOW is
   port (  
   	  N_TS_DLY:		in	STD_LOGIC;
        CLK :		in	STD_LOGIC; 
        DATAXFR :		in	STD_LOGIC;
        N_ARTRY	:	in	STD_LOGIC; 
        ADR_BRA :		in	STD_LOGIC; 
        ADR_BRB :		in	STD_LOGIC; 
        N_RESET :		in	STD_LOGIC; 
        N_DBB : 		in	STD_LOGIC;
        BUS_REQ : 	in	STD_LOGIC;
        BUSREQ_FLASH : 	in	STD_LOGIC;
        DONE :		in	STD_LOGIC;
        ENABLE_XFR :	out	STD_LOGIC; 
        N_AACK :		out	STD_LOGIC; 
        N_DBGA :		out	STD_LOGIC; 
        N_DBGB :		out	STD_LOGIC
         );
end component DATAFLOW;

-- Flash memory control state machine and odd parity generator.
component FLASH_IF is
   port (
	CLK :			in	STD_LOGIC;
	N_RESET :		in	STD_LOGIC;
	R_nW :		in 	STD_LOGIC;
	BUSREQ_FLASH :	in	STD_LOGIC;	
	FLASH_D_IN :	in	STD_LOGIC_VECTOR(0 to 15);
	DONE_FLASH :	out	STD_LOGIC;
	N_FLASHCE :		out	STD_LOGIC;
	N_FLASHWE :		out	STD_LOGIC;
	N_FLASHOE :		out	STD_LOGIC;
	FLASH_TA :		out	STD_LOGIC;
	FLASH_ADR :		out	STD_LOGIC_VECTOR(0 to 1);
	FLASH_DB :		out	STD_LOGIC_VECTOR(0 to 63);
	FLASH_DBP:		out	STD_LOGIC_VECTOR(0 to 7)
	 );
end component FLASH_IF;

-- SSRAM memory control state machine:
component SRAMCTL is
   port(
   	N_RESET :	in	STD_LOGIC; 
	RnW :		in	STD_LOGIC; 
	BUSREQ :	in	STD_LOGIC; 
	BURST :	in	STD_LOGIC; 
	BWE :     	in	STD_LOGIC_VECTOR(0 to 7); 
	CLK :		in	STD_LOGIC; 
	N_CE :	out	STD_LOGIC; 
	N_OE :	buffer STD_LOGIC; 
	N_WE :	out	STD_LOGIC_VECTOR(0 to 7);
	N_ADV :	out	STD_LOGIC; 
	N_ADSC :	out	STD_LOGIC; 
	DONE :	out	STD_LOGIC;
	SRAM_TA :	out	STD_LOGIC  
	  );
end component SRAMCTL;


-- End of component declarations.


begin

-- Instantiate DLL clock circuitry for 1X clock design:
 ACLK_ibufg1 : IBUFG 
    port map ( 
              I => SYS_CLK, 
              O => SYSCLK_ibufg 
              ); 

 ACLK_bufg1 : BUFG 
    port map ( 
              I => CLK0_dll, 
              O => CLK 
              ); 
 
 RST_in <= '0';
 
 ACLK_dlla : CLKDLL 
    port map ( 
              CLKIN 	=> SYSCLK_ibufg, 
              CLKFB 	=> CLK, 
              RST 	=> RST_IN,   -- '0', 
              CLK2X 	=> OPEN, 
              CLK0 	=> CLK0_dll, 
              CLK90 	=> OPEN, 
              CLK180 	=> OPEN, 
              CLK270 	=> OPEN, 
              CLKDV 	=> OPEN, 
              LOCKED 	=> OPEN
              ); 

-- Instantiate address bus arbiter module:
my_arbb:  ADR_ARB  
   port map (
   		CLK		=> CLK,
   		N_RESET	=> N_RESET, 
   		N_TS_DLY	=> N_TS_DLY, 
   		N_BRA		=> N_BRA,
   		N_BRB		=> N_BRB, 
   		ADR_BRA	=> ADR_BRA, 
   		ADR_BRB	=> ADR_BRB, 
   		N_BGA		=> N_BGA, 	
   		N_BGB		=> N_BGB
             );

--Instantiate global buffer on SYS_CLK input pad. 
--********************************************************
-- Replace BUFG clock buffer with DLL clock circuitry.
-- Global buffer adds 4ns skew to clock!!
--********************************************************
--U1: BUFG 
--  port map (
--   		I  => SYS_CLK,
--   		O  => CLK
--  	     );
-- CLK <= SYS_CLK; -- use this clock for behavioral simulation.

-- Instantiate address pipeline and address decoder logic.
my_pipe: ADRPIPE
   port map (
   		A		=> A, 
		N_TS		=> N_TS,
		N_TS_DLY	=> N_TS_DLY,
		TT		=> TT, 
		TSIZ		=> TSIZ, 
		N_TBST	=> N_TBST, 
            MEMA		=> MEMA,                   
            BURST		=> BURST,  
            CLK		=> CLK, 
            R_nW		=> R_nW,
            BWE		=> BWE(0 to 7), 
            N_RESET	=> N_RESET,
            ENABLE_XFR 	=> ENABLE_XFR,
            DATAXFR 	=> DATAXFR,
            BUS_REQ  	=> BUS_REQ,
            BUSREQ_FLASH => BUSREQ_FLASH,
            BANKSEL    	=> BANKSEL(0 to 1),
            DONE 		=> DONE,
            DONE_FLASH	=> DONE_FLASH
             );

-- Instantiate the databus module.
my_dbus: DATABUS 
   port map (
   		DH		=> DH,
   		DL		=> DL,
   		DP		=> DP,
   		MEMD		=> MEMD,
   		MEMDP		=> MEMDP, 
            FLASH_DB	=> FLASH_DB, 
            FLASH_DBP 	=> FLASH_DBP,
            CLK		=> CLK, 
            N_RESET	=> N_RESET, 
            BURST		=> BURST,
            R_nW		=> R_nW, 
            BUS_REQ	=> BUS_REQ, 
            BUSREQ_FLASH => BUSREQ_FLASH, 
            FLASH_TA 	=> FLASH_TA, 
            SRAM_TA	=> SRAM_TA, 
            DONE		=> DONE, 
            DONE_FLASH	=> DONE_FLASH,
            FLASH_D_IN	=> FLASH_D_IN,
            N_MEMOE	=> N_MEMOE, 
            N_TA		=> N_TA 
             );

-- Instantiate data bus arbiter.
my_data: DATAFLOW 
   port map (
   		N_TS_DLY	=> N_TS_DLY,
       	N_AACK	=> N_AACK, 
        	N_ARTRY	=> N_ARTRY, 
        	ADR_BRA	=> ADR_BRA, 
        	ADR_BRB	=> ADR_BRB, 
        	N_DBGA	=> N_DBGA, 
       	N_DBGB	=> N_DBGB, 
        	DATAXFR     => DATAXFR,
        	ENABLE_XFR  => ENABLE_XFR, 
        	CLK		=> CLK, 
        	N_RESET	=> N_RESET, 
        	N_DBB		=> N_DBB,
       	BUS_REQ     => BUS_REQ,
        	BUSREQ_FLASH => BUSREQ_FLASH,
        	DONE         => DONE
              );

-- Instantiate flash memory controller module:
-- Flash organized as 256Kx16-bit, big endian data numbering.
my_boot: FLASH_IF
   port map (
        	CLK		=> CLK, 
		N_RESET	=> N_RESET,
		R_nW	  	=> R_nW, 
		BUSREQ_FLASH => BUSREQ_FLASH,
		DONE_FLASH  => DONE_FLASH, 
		N_FLASHCE	=> N_FLASHCE, 
		N_FLASHWE	=> N_FLASHWE, 
		N_FLASHOE	=> N_FLASHOE,
		FLASH_ADR   => FLASH_ADR(0 to 1),
		FLASH_D_IN	=> FLASH_D_IN,
		FLASH_DB   	=> FLASH_DB,
		FLASH_DBP  	=> FLASH_DBP,
		FLASH_TA   	=> FLASH_TA
		 );

-- Instantiate the SSRAM memory controller module:
-- Only one bank of memory currently supported for reference design.
my_sram: SRAMCTL
   port map (
   		N_RESET 	=> N_RESET, 
		RnW     	=> R_nW, 
		BUSREQ  	=> BUS_REQ, 
		BURST		=> BURST, 
		BWE		=> BWE(0 to 7), 
		CLK     	=> CLK, 
		N_CE    	=> N_MEMCE, 
		N_OE    	=> N_MEMOE, 
		N_WE    	=> N_MEMBEW,
		N_ADV   	=> N_MEMADV, 
		N_ADSC  	=> N_MEMADSC, 
		DONE    	=> DONE,
		SRAM_TA 	=> SRAM_TA
		);

end TOP_60X;

