	Page	58,132
	Title	DEBUG.ASM	Apple Emulator Debugger
;******************************************************************************
;
;   Name:	DEBUG.ASM	Apple Emulator Debugger
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the Apple
;	emulator resident debugger.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Debug			; Apple emulator resident debugger
	Public	Debug_Init		; Debugger configuration routine
	Public	Value			; Current value storage area
	Public	Dump_Code		; Dump code listing routine
	Public	Dump_Data		; Dump RAM data routine
	Public	Dump_Stack		; Dump stack data routine
	Public	Print_Title		; Print debugger title routine
	Public	Print_Status		; Print status routine
	Public	Print_Register		; Print registers routine
	Public	Title_Window		; Debugger title window structure
	Public	Data_Window		; Debugger data window structure
	Public	Status_Window		; Debugger status window structure
	Public	Code_Window		; Debugger code window structure
	Public	Register_Window 	; Debugger register window structure
	Public	Stack_Window		; Debugger stack window structure
	Public	Command_Window		; Debugger command window structure
	Public	Code_Address		; Current code dump address
	Public	Data_Address		; Current data dump address
	Public	Current_Address 	; Current address value
	Public	Compare_Address 	; Compare address value
	Public	Break_Address		; Breakpoint address value
	Public	Last_Code		; Last code dump address
	Public	Last_Data		; Last data dump address
	Public	Dump_Format		; Current data dump format
	Public	Enter_Format		; Current data entry format
	Public	Color_Table		; Debugger color table structure
	Public	Opcode_Table		; Debugger opcode table
	Public	Debug_Table		; Debugger command table
	Public	Break_Table		; Debugger breakpoint table
	Public	Code_Table		; Debugger code address table
	Public	Data_Table		; Debugger data address table
	Public	Type_Table		; Debugger type table
	Public	Local_Break		; Local breakpoint structure
	Public	Value			; Current value storage area
	Public	Compare 		; Comparison value storage area
	Public	Code_Count		; Code breakpoints count value
	Public	Data_Count		; Data breakpoints count value
	Public	Trace_Count		; Trace count value storage area
;
;  External Declarations
;
	Extrn	TTY_Init:Near		; Initialize TTY mode routine	  (TTY)
	Extrn	TTY_Reset:Near		; Reset TTY mode routine	  (TTY)
	Extrn	Write_TTY:Near		; Write TTY routine		  (TTY)
	Extrn	Set_Position:Near	; Set current position routine	  (TTY)
	Extrn	Set_Row:Near		; Set current row routine	  (TTY)
	Extrn	Set_Column:Near 	; Set current column routine	  (TTY)
	Extrn	Set_Attribute:Near	; Set current attribute routine   (TTY)
	Extrn	Set_Foreground:Near	; Set current foreground routine  (TTY)
	Extrn	Set_Background:Near	; Set current background routine  (TTY)
	Extrn	Set_Cursor:Near 	; Set cursor position routine	  (TTY)
	Extrn	Set_Type:Near		; Set cursor type routine	  (TTY)
	Extrn	Get_Position:Near	; Get current position routine	  (TTY)
	Extrn	Get_Row:Near		; Get current row routine	  (TTY)
	Extrn	Get_Column:Near 	; Get current column routine	  (TTY)
	Extrn	Get_Attribute:Near	; Get current attribute routine   (TTY)
	Extrn	Get_Foreground:Near	; Get current foreground routine  (TTY)
	Extrn	Get_Background:Near	; Get current background routine  (TTY)
	Extrn	Get_Cursor:Near 	; Get cursor position routine	  (TTY)
	Extrn	Get_Type:Near		; Get cursor type routine	  (TTY)
	Extrn	Input_Data:Near 	; Input data routine		(INPUT)
	Extrn	Error:Near		; Apple emulator error routine	(APPLE)
	Extrn	Exit:Near		; Apple emulator exit routine	(APPLE)
	Extrn	Read_Memory:Near	; Read 65C02 memory routine	(APPLE)
	Extrn	Write_Memory:Near	; Write 65C02 memory routine	(APPLE)
	Extrn	Break_Setup:Near	; Breakpoint setup routine	(BREAK)
	Extrn	Break_Reset:Near	; Breakpoint reset routine	(BREAK)
	Extrn	Break_Check:Near	; Breakpoint check routine	(BREAK)
	Extrn	Assignment:Near 	; Assignment test routine      (CONFIG)
	Extrn	Unknown:Near		; Unknown parameter routine    (CONFIG)
	Extrn	Noise:Near		; Skip noise routine	       (CONFIG)
	Extrn	Get_Window:Near 	; Get current window routine   (WINDOW)
	Extrn	Set_Window:Near 	; Set current window routine   (WINDOW)
	Extrn	Scroll_Up:Near		; Scroll window up routine     (WINDOW)
	Extrn	Scroll_Down:Near	; Scroll window down routine   (WINDOW)
	Extrn	Clear_Window:Near	; Clear window routine	       (WINDOW)
	Extrn	Save_Window:Near	; Save window routine	       (WINDOW)
	Extrn	Restore_Window:Near	; Restore window routine       (WINDOW)
	Extrn	Get_Parameter:Near	; Get parameter routine       (SUPPORT)
	Extrn	Upper_Case:Near 	; Convert to upper case       (SUPPORT)
	Extrn	Match_Parameter:Near	; Match parameter routine     (SUPPORT)
	Extrn	ASCII_Binary:Near	; ASCII to binary conversion  (SUPPORT)
	Extrn	Clear_Keyboard:Near	; Clear keyboard routine     (KEYBOARD)
	Extrn	Flush_Keyboard:Near	; Flush keyboard routine     (KEYBOARD)
	Extrn	Check_Key:Near		; Check for input routine    (KEYBOARD)
	Extrn	Get_Key:Near		; Get keyboard code routine  (KEYBOARD)
	Extrn	Printf:Near		; Generic Printf routine       (PRINTF)
	Extrn	Input_Data:Near 	; Input data routine		(INPUT)
	Extrn	Get_Expression:Near	; Get expression value routine(EXPRESS)
	Extrn	Execute_Command:Near	; Debugger execute command    (COMMAND)
	Extrn	Go_Command:Near 	; Debugger go command	      (COMMAND)
	Extrn	Eval_Command:Near	; Debugger eval command       (COMMAND)
	Extrn	Dump_Command:Near	; Debugger dump command       (COMMAND)
	Extrn	Dbyte_Command:Near	; Debugger dump byte command  (COMMAND)
	Extrn	Dword_Command:Near	; Debugger dump word command  (COMMAND)
	Extrn	Code_Command:Near	; Debugger unassemble command (COMMAND)
	Extrn	Reg_Command:Near	; Debugger register command   (COMMAND)
	Extrn	Radix_Command:Near	; Debugger radix command      (COMMAND)
	Extrn	Help_Command:Near	; Debugger help command       (COMMAND)
	Extrn	Trace_Command:Near	; Debugger trace command      (COMMAND)
	Extrn	Enter_Command:Near	; Debugger enter command      (COMMAND)
	Extrn	Ebyte_Command:Near	; Debugger enter byte command (COMMAND)
	Extrn	Eword_Command:Near	; Debugger enter word command (COMMAND)
	Extrn	Fill_Command:Near	; Debugger fill command       (COMMAND)
	Extrn	Move_Command:Near	; Debugger move command       (COMMAND)
	Extrn	Compare_Command:Near	; Debugger compare command    (COMMAND)
	Extrn	Search_Command:Near	; Debugger search command     (COMMAND)
	Extrn	Bpoint_Command:Near	; Debugger set breakpoint     (COMMAND)
	Extrn	Bdsable_Command:Near	; Debugger disable breakpoint (COMMAND)
	Extrn	Benable_Command:Near	; Debugger enable breakpoint  (COMMAND)
	Extrn	Bclear_Command:Near	; Debugger clear breakpoint   (COMMAND)
	Extrn	Blist_Command:Near	; Debugger list breakpoint    (COMMAND)
	Extrn	Proceed_Command:Near	; Debugger process command    (COMMAND)
	Extrn	Flip_Command:Near	; Debugger fill command       (COMMAND)
	Extrn	Key_Status:Byte 	; Keyboard status byte	     (KEYBOARD)
	Extrn	RAM_Space:Word		; RAM space segment value	 (DATA)
	Extrn	Input_Buffer:Byte	; Input buffer storage area	 (DATA)
	Extrn	Parm_Buffer:Byte	; Parameter buffer storage area  (DATA)
	Extrn	Delimit_Input:Byte	; Input delimiter table 	 (DATA)
	Extrn	Delimit_Debug:Byte	; Debug delimiter table 	 (DATA)
	Extrn	Delimit_Express:Byte	; Expression delimiter table	 (DATA)
	Extrn	Flag_Encode:Byte	; CPU flag encoding table	 (DATA)
	Extrn	Flag_Decode:Byte	; CPU flag decoding table	 (DATA)
	Extrn	Emulate_Flag:Byte	; Emulator system flag byte	 (DATA)
	Extrn	System_Flag:Byte	; Apple emulator system flag byte(DATA)
	Extrn	Video_Flag:Byte 	; Video system flag byte	 (DATA)
	Extrn	Screen_Window:Word	; Entire video screen window	 (DATA)
	Extrn	ERR_COLOR:Abs		; Invalid color error code	 (DATA)
	Extrn	ERR_SCHEME:Abs		; Invalid color scheme error code(DATA)
	Extrn	Invalid_Command:Byte	; Invalid command format      (MESSAGE)
;
;  LOCAL Equates
;
TITLE_UL_ROW	Equ	00h		; Debug
TITLE_UL_COL	Equ	00h		;	title
TITLE_LR_ROW	Equ	01h		;	      window
TITLE_LR_COL	Equ	27h		;		     definition
DATA_UL_ROW	Equ	02h		; Debug
DATA_UL_COL	Equ	00h		;	data
DATA_LR_ROW	Equ	06h		;	     window
DATA_LR_COL	Equ	27h		;		    definition
STAT_UL_ROW	Equ	07h		; Debug
STAT_UL_COL	Equ	00h		;	status
STAT_LR_ROW	Equ	07h		;	       window
STAT_LR_COL	Equ	27h		;		      definition
CODE_UL_ROW	Equ	08h		; Debug
CODE_UL_COL	Equ	00h		;	code
CODE_LR_ROW	Equ	12h		;	     window
CODE_LR_COL	Equ	20h		;		    definition
REG_UL_ROW	Equ	13h		; Debug
REG_UL_COL	Equ	00h		;	register
REG_LR_ROW	Equ	13h		;		 window
REG_LR_COL	Equ	27h		;			definition
STACK_UL_ROW	Equ	08h		; Debug
STACK_UL_COL	Equ	21h		;	stack
STACK_LR_ROW	Equ	12h		;	      window
STACK_LR_COL	Equ	27h		;		     definition
COMM_UL_ROW	Equ	14h		; Debug
COMM_UL_COL	Equ	00h		;	command
COMM_LR_ROW	Equ	18h		;		window
COMM_LR_COL	Equ	27h		;		       definition
MNEMONIC_COLUMN Equ	10h		; Debug mnemonic starting column value
ADDRESS_COLUMN	Equ	15h		; Debug address starting column value
BREAK_COLUMN	Equ	1Dh		; Debug breakpoint starting column value
LINE_COUNT	Equ	08h		; Number of bytes per data line
BYTE_COUNT	Equ	03h		; Number of bytes per code line (Max.)
FLAG_COUNT	Equ	08h		; Number of 65C02 flag bits to print
RANGE_MASK	Equ	80h		; ASCII translate range mask value
FORMAT_BYTE	Equ	00h		; Data byte format value
FORMAT_WORD	Equ	02h		; Data word format value
INTENSE 	Equ	08h		; Color intensity bit value
MAX_COLOR	Equ	0Fh		; Maximum color value
PENDING_INT	Equ	(SYS_DEBUG + SYS_RESET + SYS_REQUEST)
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include 	Strucs.inc	; Include the structure definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Debug		Apple Emulator Debugger
	Page	+
;******************************************************************************
;
;	Debug(RAM_Space, Registers)
;
;		Save the required registers
;		Clear break and trace interrupts
;		If not already in debug mode
;			Put keyboard into input mode
;			Call routine to reset breakpoints
;			Call routine to initialize the TTY interface
;			Setup the debugger background color value
;			Call routine to clear the window (Clear screen)
;		Endif
;		Call routine to clear the keyboard
;		Call routine to print debugger title
;		Call routine to dump the data values
;		Call routine to print the status line
;		If current address within last code dumped
;			Use last code address for dump
;		Else
;			Use current address for dump
;		Endif
;		Call routine to dump the code
;		Call routine to print the 65C02 registers
;		Call routine to dump the 65C02 stack
;		Call routine to process debug commands
;		If command requests to leave debug mode
;			Call routine to reset TTY mode
;			Call routine to setup breakpoints
;			Take the keyboard out of input mode
;		Endif
;		Call routine to clear the keyboard
;		Clear any pending system interrupts
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DL    - 65C02 Accumulator
;		DH    - 65C02 processor flags
;		CL    - 65C02 Y index register
;		CH    - 65C02 X index register
;		BX    - 65C02 Stack pointer
;		SI    - 65C02 Program counter
;		DS    - 65C02 RAM space
;		ES    - Video RAM segment
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Debug		Proc	Near		; Apple emulator debugger procedure
	Save	ax,bx,cx,dx,si,di,bp	; Save the required registers
	mov	bp,sp			; Setup access to the stack frame
	Save	ds,es			; Save the emulator segment values
	mov	ax,cs			; Setup to access
	mov	ds,ax			;		  the data segment
	and	cs:[Emulate_Flag],Not PENDING_INT
	test	ds:[System_Flag],INPUT	; Check for already in debug mode
	jnz	Debug_Continue		; Jump if already in debug mode
	or	ds:[System_Flag],INPUT	; Put keyboard into input mode
	call	Break_Reset		; Call routine to reset breakpoints
	call	TTY_Init		; Call routine to initialize TTY
	mov	ax,ds:[Color_Table.Color_Back]
	call	Set_Background		; Setup the debugger background color
	call	Clear_Window		; Call routine to clear window (Screen)
Debug_Continue:
	call	Clear_Keyboard		; Call routine to clear the keyboard
	mov	ax,ss:[bp.SI_Register]	; Get the 65C02
	dec	ax			;		program counter (SI-1)
	cmp	ax,ds:[Last_Code]	; Check against last code address
	jb	Code_Update		; Jump if outside last code dump
	cmp	ax,ds:[Code_Address]	; Check against last code address
	jae	Code_Update		; Jump if outside last code dump
	mov	ax,ds:[Last_Code]	; Use last code dump address value
Code_Update:
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Print_Title		; Call routine to print debug title
	call	Dump_Data		; Call routine to dump the data
	call	Print_Status		; Call routine to print status line
	call	Dump_Code		; Call routine to dump the code
	call	Print_Register		; Call routine to print register line
	call	Dump_Stack		; Call routine to dump the stack
	call	Command 		; Call routine to process debug commands
	Restore ds,es			; Restore the emulator segment values
	jz	Debug_Exit		; Jump if not to exit debug mode
	mov	al,cs:[Key_Status]	; Get current keyboard status
	push	ax			; Save current keyboard status
	and	cs:[Key_Status],Not TYPE_SPECIAL
	call	TTY_Reset		; Call routine to reset TTY
	pop	ax			; Restore the original
	mov	cs:[Key_Status],al	;		       keyboard status
	call	Break_Setup		; Call routine to setup breakpoints
	and	cs:[System_Flag],Not INPUT
Debug_Exit:
	call	Clear_Keyboard		; Call routine to clear the keyboard
	and	cs:[Emulate_Flag],Not PENDING_INT
	Restore ax,bx,cx,dx,si,di,bp	; Restore the required registers
	ret				; Return to the caller
Debug		Endp			; End of the Debug procedure
	Subttl	Debug_Init	Debugger Configuration Routine
	Page	+
;******************************************************************************
;
;	Debug_Init(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		While not at the next section
;			Call routine to get the next parameter
;			If a null parameter was given
;				If delimiter type is comment
;					Call routine to eat this comment
;				Endif
;				If delimiter type is section
;					Exit the while loop
;				Endif
;			Else valid parameter was given
;				Call routine to convert to uppercase
;				Get pointer to parameter table
;				Call routine to check for parameter match
;				If no parameter match
;					Call routine for unknown parameter
;				Endif
;				Save the actual matching value
;				Call routine to check for assignment
;				Call routine to process the parameter
;			Endif
;		Endwhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;		ES:DI - Pointer to parameter buffer
;
;	Registers on Exit:
;
;		AX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Debug_Init	Proc	Near		; Debugger configuration procedure
	Save	si,di,es		; Save the required registers
	mov	ax,cs			; Setup access to
	mov	es,ax			;		  the parse tables
Debug_Loop:
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	jc	Config_Exit		; Jump if no more parameters
	jnz	Check_Parameter 	; Jump if a non-null parameter
Check_Type:
	test	ah,SECTION_TYPE 	; Check for section type delimiter
	jnz	Config_Exit		; Jump if next section reached
	test	ah,COMMENT_TYPE 	; Check for a comment type delimiter
	jz	Debug_Loop		; Jump if this is not a comment
	call	Noise			; Call routine to skip this noise
	jmp	Short Debug_Loop	; Go continue the parsing
Check_Parameter:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Match_Table]	; Get pointer to start of match table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the match table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jnc	Process_Parameter	; Jump if a parameter match was found
Parameter_Error:
	call	Unknown 		; Call the unknown parameter routine
	jmp	Short Config_Exit	; Go return control to the caller
Process_Parameter:
	xor	bh,bh			; Convert match number
	shl	bx,1			;		       to table index
	push	bx			; Save the matching parameter value
	call	Assignment		; Call routine to check for assignment
	pop	bx			; Restore the matching parameter value
	call	cs:[Jump_Table + bx]	; Call routine to handle parameter
	jmp	Short Check_Type	; Go check the delimiter type value
Config_Exit:
	dec	dx			; Correct the current parse position
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Debug_Init	Endp			; End of the Debug_Init procedure
	Subttl	Back_Option	Background Parameter Option
	Page	+
;******************************************************************************
;
;	Back_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set background color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Back_Option	Proc	Near		; Background parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Back		; Jump if valid color given
Back_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Back:
	mov	Byte Ptr cs:[Color_Table.Color_Back],al
	ret				; Return to the caller
Back_Option	Endp			; End of the Back_Option procedure
	Subttl	Title_Option	Title Parameter Option
	Page	+
;******************************************************************************
;
;	Title_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set title color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Title_Option	Proc	Near		; Title parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Title		; Jump if valid color given
Title_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Title:
	mov	Byte Ptr cs:[Color_Table.Color_Title],al
	ret				; Return to the caller
Title_Option	Endp			; End of the Title_Option procedure
	Subttl	Command_Option	Command Parameter Option
	Page	+
;******************************************************************************
;
;	Command_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set command color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Command_Option	Proc	Near		; Command parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Command		; Jump if valid color given
Command_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Command:
	mov	Byte Ptr cs:[Color_Table.Color_Command],al
	ret				; Return to the caller
Command_Option	Endp			; End of the Command_Option procedure
	Subttl	Address_Option	Address Parameter Option
	Page	+
;******************************************************************************
;
;	Address_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set address color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Address_Option	Proc	Near		; Address parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Address		; Jump if valid color given
Address_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Address:
	mov	Byte Ptr cs:[Color_Table.Color_Address],al
	ret				; Return to the caller
Address_Option	Endp			; End of the Address_Option procedure
	Subttl	Value_Option	Value Parameter Option
	Page	+
;******************************************************************************
;
;	Value_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set value color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Value_Option	Proc	Near		; Value parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Value		; Jump if valid color given
Value_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Value:
	mov	Byte Ptr cs:[Color_Table.Color_Value],al
	ret				; Return to the caller
Value_Option	Endp			; End of the Value_Option procedure
	Subttl	Mnemonic_Option Mnemonic Parameter Option
	Page	+
;******************************************************************************
;
;	Mnemonic_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set opcode color to given value
;			Set mnemonic color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Mnemonic_Option Proc	Near		; Mnemonic parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Mnemonic		; Jump if valid color given
Mnemonic_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Mnemonic:
	mov	Byte Ptr cs:[Color_Table.Color_Opcode],al
	mov	Byte Ptr cs:[Color_Table.Color_Mnemonic],al
	ret				; Return to the caller
Mnemonic_Option Endp			; End of the Mnemonic_Option procedure
	Subttl	Flags_Option	Flags Parameter Option
	Page	+
;******************************************************************************
;
;	Flags_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set flag off color to given value
;			Set flag on color to toggled given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Flags_Option	Proc	Near		; Flags parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Flags		; Jump if valid color given
Flags_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Flags:
	mov	Byte Ptr cs:[Color_Table.Color_Off],al
	xor	al,INTENSE		; Toggle the intensity value
	mov	Byte Ptr cs:[Color_Table.Color_On],al
	ret				; Return to the caller
Flags_Option	Endp			; End of the Flags_Option procedure
	Subttl	Data_Option	Data Parameter Option
	Page	+
;******************************************************************************
;
;	Data_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set data color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Data_Option	Proc	Near		; Data parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Data		; Jump if valid color given
Data_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Data:
	mov	Byte Ptr cs:[Color_Table.Color_Data],al
	ret				; Return to the caller
Data_Option	Endp			; End of the Data_Option procedure
	Subttl	Stack_Option	Stack Parameter Option
	Page	+
;******************************************************************************
;
;	Stack_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set stack color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Stack_Option	Proc	Near		; Stack parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Stack		; Jump if valid color given
Stack_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Stack:
	mov	Byte Ptr cs:[Color_Table.Color_Stack],al
	ret				; Return to the caller
Stack_Option	Endp			; End of the Stack_Option procedure
	Subttl	Register_Option Register Parameter Option
	Page	+
;******************************************************************************
;
;	Register_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the color value
;		If a valid color value is given
;			Set register color to given value
;		Else invalid color value
;			Get invalid color error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Register_Option Proc	Near		; Register parameter option procedure
	call	Get_Color		; Call routine to get color value
	jnc	Set_Register		; Jump if valid color given
Register_Error:
	mov	al,ERR_COLOR		; Get bad color error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Set_Register:
	mov	Byte Ptr cs:[Color_Table.Color_Register],al
	ret				; Return to the caller
Register_Option Endp			; End of the Register_Option procedure
	Subttl	Scheme_Option	Debugger Color Scheme Option
	Page	+
;******************************************************************************
;
;	Scheme_Option(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;
;
;
;
;
;
;
;
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Scheme_Option	Proc	Near		; Debugger color scheme option procedure
	Save	si,di,es		; Save the required registers
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	Save	ax,dx			; Save the required registers
	ja	Get_Scheme		; Jump if valid color scheme file
Scheme_Error:
	Restore ax,dx			; Restore the required registers
	mov	al,ERR_SCHEME		; Get bad color scheme error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Get_Scheme:
	mov	di,ds			; Save 65C02 RAM space segment value
	mov	ax,cs			; Setup to access
	mov	ds,ax			;		  the parameter value
	mov	ah,OPEN_FILE		; Get the open file function code
	mov	al,READ_ONLY		; Get the read file access code
	lea	dx,cs:[Parm_Buffer]	; Get pointer to color scheme filename
	int	DOS			; Try to open the color scheme file
	mov	ds,di			; Restore 65C02 RAM space segment value
	mov	bx,ax			; Move file handle to BX register
	jc	Scheme_Error		; Jump if no color scheme file present
	mov	ax,cs			; Setup to access
	mov	ds,ax			;		  the debug color table
Read_Scheme:
	mov	ah,READ_FILE		; Get read file function code
	lea	dx,cs:[Color_Table]	; Setup the data buffer address
	mov	cx,Size Debug_Color	; Get debug color table size (Bytes)
	int	DOS			; Try to read color scheme file
	mov	ds,di			; Restore 65C02 RAM space segment value
	jc	Scheme_Error		; Jump if bad color scheme file
	cmp	ax,cx			; Check the color scheme file size
	jne	Scheme_Error		; Jump if bad color scheme file
	mov	ah,CLOSE_FILE		; Get close file function code
	int	DOS			; Close the color scheme file
	Restore ax,dx			; Restore the required registers
	clc				; Clear carry indicating no error
Scheme_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return control to the caller
Scheme_Option	Endp			; End of the Scheme_Option procedure
	Subttl	Get_Color	Get Color Value
	Page	+
;******************************************************************************
;
;	Get_Color(Debugger_Data, Pointer, Offset)
;
;		Save the required registers
;		Initialize to non-intense value
;		While there are more parameters
;			Call routine to get the next parameter
;			If there is a valid parameter
;				Call routine to convert to uppercase
;				Call routine to check for parameter match
;				If there is a parameter match
;					If this is color value
;						Set intensity flag
;						Exit the while loop (Return)
;					Else intensity selection
;						Toggle the intensity flag
;					Endif
;				Else invalid color value
;					Set carry indicating color error
;				Endif
;			Else no/invalid parameter
;				Set carry to indicate color error
;			Endif
;		EndWhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		SI    - Pointer to configuration data (In 65C02 RAM Space)
;
;	Registers on Exit:
;
;		AL    - Color value found
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;		FL    - Carry set if no/invalid color value
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Color	Proc	Near		; Get color value procedure
	Save	si,di,es		; Save the required registers
	xor	ax,ax			; Initialize to
	mov	cs:[Value],ax		;		non-intense value
Get_Next:
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	jc	Color_Exit		; Jump if invalid parameter given
	stc				; Set carry in case of null parameter
	jz	Color_Exit		; Jump if null (Invalid) parameter
Check_Color:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Parm_Table]	; Get pointer to parameter table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get parameter table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jc	Color_Exit		; Jump if no parameter match found
	xor	bx,cs:[Value]		; Toggle intensity if needed
	mov	al,bl			; Get the actual color value
	cmp	al,MAX_COLOR + 1	; Check the color value
	cmc				; Set carry flag to correct state
	jnc	Color_Exit		; Jump if a good color value
	xor	cs:[Value],INTENSE	; Toggle the intensity setting
	jmp	Short Get_Next		; Go try to get next parameter
Color_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Get_Color	Endp			; End of the Get_Color procedure
	Subttl	Command 	Apple Emulator Debug Command Processor
	Page	+
;******************************************************************************
;
;	Command(Data_Segment, Stack_Frame)
;
;		Save the required registers
;		Setup command window as the current window
;		Setup the command color values
;		Initialize the command row and column values
;		While there are commands to execute
;			Call routine to print debug prompt
;			Call routine to get user input
;			Call routine to get first user parameter (Command)
;			If there is a valid parameter (non-null)
;				Call routine to convert to uppercase
;				Check for a valid debug command
;				If this is a valid debug command
;					Call routine to process this command
;				Else invalid command
;					Call routine to print help message
;				Endif
;			Endif
;		EndWhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;		FL    - Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Command 	Proc	Near		; Debug command processor procedure
	Save	ds,es			; Save the required registers
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	lea	si,ds:[Command_Window]	; Get pointer to debug command window
	call	Set_Window		; Call routine to set current window
	mov	ax,ds:[Color_Table.Color_Command]
	mov	bx,ds:[Color_Table.Color_Back]
	call	Set_Attribute		; Setup the command color values
	mov	ax,ds:[si.Lower_Right_Row]
	sub	ax,ds:[si.Upper_Left_Row]
	xor	bx,bx			; Setup the command column value
	call	Set_Position		; Call routine to set row and column
Command_Loop:
	lea	si,ds:[Debug_Prompt]	; Get pointer to debug prompt string
	call	Printf			; Call routine to print debug prompt
	mov	cx,INPUT_SIZE		; Get the maximum line input size
	lea	di,es:[Input_Buffer]	; Get pointer to the input buffer
	call	Input_Data		; Call routine to get user command
	jz	Command_Loop		; Jump if no input data
	xor	dx,dx			; Initialize the parse offset
Parse_Command:
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,CR			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Debug]	; Get pointer to debug delimiter table
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	lea	si,ds:[Input_Buffer]	; Get pointer to the input buffer
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	call	Get_Parameter		; Call routine to get next parameter
	jbe	Command_Loop		; Jump if no/null parameter
	mov	bx,ax			; Save parameter length/delimiter type
	call	Upper_Case		; Call routine to convert to uppercase
	push	si			; Save the input parameter pointer
	lea	si,ds:[Debug_Table]	; Get pointer to start of debug table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the match table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	pop	si			; Restore the input parameter pointer
	jnc	Process_Command 	; Jump if command match is found
Bad_Command:
	lea	si,ds:[Invalid_Command] ; Get pointer to invalid command message
	call	Printf			; Call routine to print error message
	jmp	Short Command_Loop	; Go try to get the next command
Process_Command:
	xor	ah,ah			; Convert match number to full word
	shl	ax,1			; Convert match number to table index
	xchg	bx,ax			; Setup to access command jump table
	call	ds:[Command_Table + bx] ; Call routine to process this command
	jnc	Command_Loop		; Jump if more commands to process
	Restore ds,es			; Restore the required registers
	ret				; Return to the caller
Command 	Endp			; End of the Command procedure
	Subttl	Print_Title	Print Debug Title Bar
	Page	+
;******************************************************************************
;
;	Print_Title(Data_Segment)
;
;		Save the required registers
;		Get and save the current window
;		Setup title window as current window
;		Setup the title color values
;		Call routine to clear title window
;		Position to title row and column
;		Get pointer to debug title
;		Call routine to print debugger title
;		Restore the original window
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Print_Title	Proc	Near		; Print debug title bar procedure
	Save	ax,bx,si		; Save the required registers
	call	Get_Window		; Call routine to get current window
	push	si			; Save the current window pointer
	lea	si,ds:[Title_Window]	; Get pointer to debug title window
	call	Set_Window		; Call routine to set current window
	mov	ax,ds:[Color_Table.Color_Title]
	mov	bx,ds:[Color_Table.Color_Back]
	call	Set_Attribute		; Setup the title color values
	call	Clear_Window		; Call routine to clear the window
	xor	ax,ax			; Setup the title
	xor	bx,bx			;		  row and column values
	call	Set_Position		; Call routine to set row and column
	lea	si,ds:[Debug_Title]	; Get pointer to debug title
	call	Printf			; Call routine to print the title
	pop	si			; Restore the original window pointer
	call	Set_Window		; Call routine to set current window
	Restore ax,bx,si		; Restore the required registers
	ret				; Return to the caller
Print_Title	Endp			; End of the Print_Title procedure
	Subttl	Print_Status	Print Current Status Routine
	Page	+
;******************************************************************************
;
;	Print_Status(Data_Segment, Stack_Frame)
;
;		Save the required registers
;		Get and save the current window
;		Setup status window as current window
;		Setup the status color values
;		Call routine to clear status window
;		Position to status row and column
;		Get the 65C02 program counter value
;		Call routine to print the program counter value
;		Get the 65C02 stack pointer value
;		Call routine to print the stack pointer value
;		Restore the original window
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Print_Status	Proc	Near		; Print current status procedure
	Save	ax,bx,si		; Save the required registers
	call	Get_Window		; Call routine to get current window
	push	si			; Save the current window pointer
	lea	si,ds:[Status_Window]	; Get pointer to debug status window
	call	Set_Window		; Call routine to set current window
	mov	ax,ds:[Color_Table.Color_Register]
	mov	bx,ds:[Color_Table.Color_Back]
	call	Set_Attribute		; Setup the status color values
	call	Clear_Window		; Call routine to clear the window
	xor	ax,ax			; Setup the status
	xor	bx,bx			;		   row and column values
	call	Set_Position		; Call routine to set row and column
	mov	ax,ss:[bp.SI_Register]	; Get the
	dec	ax			;	  65C02 program
	mov	ds:[Value],ax		;			counter (SI-1)
	lea	si,ds:[PC_Format]	; Get pointer to PC format string
	call	Printf			; Call routine to print the string
	mov	ax,ds:[Color_Table.Color_Value]
	call	Set_Foreground		; Setup the value foreground color
	call	Print_Word		; Call routine to print the word value
	mov	ax,ds:[Color_Table.Color_Register]
	call	Set_Foreground		; Setup the register foreground color
	mov	ax,ss:[bp.BX_Register]	; Get the 65C02
	mov	ds:[Value],ax		;		stack pointer (BX)
	lea	si,ds:[SP_Format]	; Get pointer to SP format string
	call	Printf			; Call routine to print the string
	mov	ax,ds:[Color_Table.Color_Value]
	call	Set_Foreground		; Setup the value foreground color
	call	Print_Word		; Call routine to print the word value
	mov	ax,ds:[Color_Table.Color_Register]
	call	Set_Foreground		; Setup the register foreground color
	pop	si			; Restore the original window pointer
	call	Set_Window		; Call routine to set current window
	Restore ax,bx,si		; Restore the required registers
	ret				; Return to the caller
Print_Status	Endp			; End of the Print_Status procedure
	Subttl	Print_Register	Print Registers Routine
	Page	+
;******************************************************************************
;
;	Print_Register(Data_Segment, Stack_Frame)
;
;		Save the required registers
;		Get and save the current window
;		Setup register window as current window
;		Setup the register color values
;		Call routine to clear register window
;		Position to register row and column
;		Get the 65C02 accumulator value
;		Call routine to print the accumulator value
;		Get the 65C02 X index value
;		Call routine to print the X index value
;		Get the 65C02 Y index value
;		Call routine to print the Y index value
;		Get the 65C02 flags value
;		Set the r (Reserved) flag bit
;		Encode flag bits from 80x86 to 65C02
;		While there are more flags to print
;			Setup the flag off color
;			Shift flag bit into carry
;			If the flag bit is set
;				Setup the flag on color
;			Endif
;			Call routine to print the flag
;		EndWhile
;		Restore the original window
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Print_Register	Proc	Near		; Print registers procedure
	Save	ax,bx,si		; Save the required registers
	call	Get_Window		; Call routine to get current window
	push	si			; Save the current window pointer
	lea	si,ds:[Register_Window] ; Get pointer to debug register window
	call	Set_Window		; Call routine to set current window
	mov	ax,ds:[Color_Table.Color_Register]
	mov	bx,ds:[Color_Table.Color_Back]
	call	Set_Attribute		; Setup the register color values
	call	Clear_Window		; Call routine to clear the window
	xor	ax,ax			; Setup register
	xor	bx,bx			;		 row and column values
	call	Set_Position		; Call routine to set row and column
	lea	si,ds:[ACC_Format]	; Get pointer to ACC format string
	call	Printf			; Call routine to print the string
	mov	ax,ds:[Color_Table.Color_Value]
	call	Set_Foreground		; Setup the value foreground color
	mov	al,ss:[bp.DL_Register]	; Get the 65C02
	mov	Byte Ptr ds:[Value],al	;		accumulator register
	call	Print_Byte		; Call routine to print the byte value
	mov	ax,ds:[Color_Table.Color_Register]
	call	Set_Foreground		; Setup the register foreground color
	lea	si,ds:[X_Format]	; Get pointer to X format string
	call	Printf			; Call routine to print the string
	mov	ax,ds:[Color_Table.Color_Value]
	call	Set_Foreground		; Setup the value foreground color
	mov	al,ss:[bp.CH_Register]	; Get the 65C02
	mov	Byte Ptr ds:[Value],al	;		X index register
	call	Print_Byte		; Call routine to print the byte value
	mov	ax,ds:[Color_Table.Color_Register]
	call	Set_Foreground		; Setup the register foreground color
	lea	si,ds:[Y_Format]	; Get pointer to Y format string
	call	Printf			; Call routine to print the string
	mov	ax,ds:[Color_Table.Color_Value]
	call	Set_Foreground		; Setup the value foreground color
	mov	al,ss:[bp.CL_Register]	; Get the 65C02
	mov	Byte Ptr ds:[Value],al	;		Y index register
	call	Print_Byte		; Call routine to print the byte value
	mov	ax,ds:[Color_Table.Color_Register]
	call	Set_Foreground		; Setup the register foreground color
	lea	si,ds:[Flags_Format]	; Get pointer to flags format string
	call	Printf			; Call routine to print the string
	lea	ax,ds:[Flag_String]	; Initialize flag
	mov	ds:[Char_Address],ax	;		  character pointer
	mov	bl,ss:[bp.DH_Register]	; Get the 65C02 flags register
	or	bl,CPU_R		; Make sure reserved bit is set
	xor	bh,bh			; Convert flag value to full word
	mov	dl,cs:[bx + Flag_Encode]; Get the encoded flag value (65C02)
	mov	cx,FLAG_COUNT		; Get the flag count value
Flag_Loop:
	mov	ax,ds:[Color_Table.Color_Off]
	shl	dl,1			; Put flag bit into carry flag
	jnc	Set_Color		; Jump if flag bit is off
	mov	ax,ds:[Color_Table.Color_On]
Set_Color:
	call	Set_Foreground		; Call routine to set correct color
	lea	bx,ds:[Char_Arg]	; Get pointer to character argument list
	lea	si,ds:[Char_Format]	; Get pointer to character format string
	call	Printf			; Call routine to print the character
	inc	Word Ptr ds:[Char_Address]
	loop	Flag_Loop		; Loop till all flags are printed
	mov	ax,ds:[Color_Table.Color_Register]
	call	Set_Foreground		; Setup the register foreground color
	pop	si			; Restore the original window pointer
	call	Set_Window		; Call routine to set current window
	Restore ax,bx,si		; Restore the required registers
	ret				; Return to the caller
Print_Register	Endp			; End of the Print_Register procedure
	Subttl	Print_Address	Print Current Address Routine
	Page	+
;******************************************************************************
;
;	Print_Address()
;
;		Save the required registers
;		Setup the address foreground color
;		Call routine to print the current address value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Print_Address	Proc	Near		; Print current address procedure
	Save	ax,bx,si,ds		; Save the required registers
	mov	ax,cs			; Setup the access
	mov	ds,ax			;		   the data segment
	mov	ax,ds:[Color_Table.Color_Address]
	call	Set_Foreground		; Setup the address foreground color
	lea	bx,ds:[Address_Arg]	; Get pointer to address argument list
	lea	si,ds:[Address_Format]	; Get pointer to address format string
	call	Printf			; Call routine to print the address
	Restore ax,bx,si,ds		; Restore the required registers
	ret				; Return to the caller
Print_Address	Endp			; End of the Print_Address procedure
	Subttl	Print_Opcode	Print Opcode Bytes Routine
	Page	+
;******************************************************************************
;
;	Print_Opcode(Data_Segment)
;
;		Setup the opcode foreground color (Highlight if necessary)
;		Get the actual opcode value
;		Get the opcode addressing type value
;		Get the actual opcode length (Bytes)
;		Get pointer to correct opcode format
;		Call routine to print the opcode bytes
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX    - Number of bytes in opcode (1-3)
;		BX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Print_Opcode	Proc	Near		; Print opcode bytes procedure
	mov	ax,ds:[Color_Table.Color_Opcode]
	xor	ax,ds:[Highlight]	; Turn highlighting on if necessary
	call	Set_Foreground		; Setup the opcode foreground color
	mov	bl,ds:[Value_0] 	; Get the actual opcode byte value
	mov	al,Size Opcode_Debug	; Get size of opcode debug structure
	mul	bl			; Compute the offset to opcode entry
	mov	bx,ax			; Setup to access the opcode table
	mov	al,ds:[Opcode_Table + bx.Op_Address]
	and	ax,ADDRESS_MASK 	; Mask off all but the addressing type
	mov	bx,ax			; Setup to address the length table
	mov	bx,ds:[Length_Table + bx]
	mov	ax,bx			; Save the actual opcode length value
	dec	bx			; Convert length value
	shl	bx,1			;		       to table index
	mov	si,ds:[Byte_Table + bx] ; Get pointer to correct format string
	lea	bx,ds:[Code_Arg]	; Get pointer to opcode argument list
	call	Printf			; Call routine to print opcode bytes
Opcode_Exit:
	ret				; Return to the caller
Print_Opcode	Endp			; End of the Print_Opcode procedure
	Subttl	Print_Mnemonic	Print Opcode Mnemonic Routine
	Page	+
;******************************************************************************
;
;	Print_Mnemonic(Data_Segment)
;
;		Setup the mnemonic foreground color (Highlight if necessary)
;		Position to mnemonic column (Use current row)
;		Get the actual opcode value
;		Get the opcode type value
;		Get pointer to mnemonic string
;		Call routine to print mnemonic string
;		Position to address column (Use current row)
;		Get the opcode addressing type value
;		Call correct routine to print address value
;		Get the current code address
;		Save the current foreground color
;		Call routine to check for breakpoint
;		If this address is a breakpoint
;			Save the breakpoint number value
;			If this breakpoint is enabled
;				Set foreground to enabled color value
;			Else breakpoint disabled
;				Set foreground to disabled color value
;			Endif
;			Call routine to print breakpoint number
;		Endif
;		Restore the current foreground color
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Print_Mnemonic	Proc	Near		; Print opcode mnemonic procedure
	mov	ax,ds:[Color_Table.Color_Mnemonic]
	xor	ax,ds:[Highlight]	; Turn highlighting on if necessary
	call	Set_Foreground		; Setup the mnemonic foreground color
	mov	ax,MNEMONIC_COLUMN	; Setup starting mnemonic
	call	Set_Column		;			  column value
	mov	bl,ds:[Value_0] 	; Get the actual opcode byte value
	mov	al,Size Opcode_Debug	; Get size of opcode debug structure
	mul	bl			; Compute the offset to opcode entry
	mov	bx,ax			; Setup to access the opcode table
	mov	dx,ax			; Save the opcode table offset value
	mov	al,ds:[Opcode_Table + bx.Op_Type]
	mov	bl,Size Opcode_Mnemonic ; Get size of opcode mnemonic structure
	mul	bl			; Compute the offset to opcode string
	lea	bx,ds:[Mnemonic_Table]	; Get pointer to start of mnemonic table
	add	ax,bx			; Compute actual mnemonic string address
	mov	ds:[Pointer],ax 	; Save pointer to mnemonic string
	lea	bx,ds:[Mnemonic_Arg]	; Get pointer to mnemonic argument list
	lea	si,ds:[Mnemonic_Format] ; Get pointer to mnemonic format string
	call	Printf			; Call routine to print the mnemonic
	mov	ax,ADDRESS_COLUMN	; Setup starting address
	call	Set_Column		;			 column value
	mov	bx,dx			; Restore the opcode table offset value
	mov	bl,ds:[Opcode_Table + bx.Op_Address]
	and	bx,ADDRESS_MASK 	; Convert addressing type to a full word
	call	cs:[Address_Table + bx] ; Call correct routine to print address
	call	Get_Foreground		; Call routine to get foreground color
	mov	cx,ax			; Save the current foreground color
	mov	ax,ds:[Code_Address]	; Get the current code address
	call	Break_Check		; Call routine to check for a breakpoint
	jc	Mnemonic_Exit		; Jump if this is not a breakpoint
	mov	ds:[Value],ax		; Save the actual breakpoint number
	mov	ax,ds:[Color_Table.Color_Disabled]
	jz	Break_Color		; Jump if this breakpoint disabled
	mov	ax,ds:[Color_Table.Color_Enabled]
Break_Color:
	call	Set_Foreground		; Call routine to set foreground color
	mov	ax,BREAK_COLUMN 	; Setup starting breakpoint
	call	Set_Column		;			    column value
	lea	bx,ds:[Break_Arg]	; Get pointer to breakpoint arguments
	lea	si,ds:[Break_Format]	; Get pointer to breakpoint format
	call	Printf			; Call routine to print breakpoint
	mov	ax,cx			; Get the original foreground color
	call	Set_Foreground		; Call routine to restore foreground
Mnemonic_Exit:
	ret				; Return to the caller
Print_Mnemonic	Endp			; End of the Print_Mnemonic procedure
	Subttl	Print_Byte	Print Byte Value Routine
	Page	+
;******************************************************************************
;
;	Print_Byte()
;
;		Save the required registers
;		Call routine to print the byte value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Print_Byte	Proc	Near		; Print byte value procedure
	Save	ax,bx,si,ds		; Save the required registers
	mov	ax,cs			; Setup the access
	mov	ds,ax			;		   the data segment
	lea	bx,ds:[Value_Arg]	; Get pointer to value argument list
	lea	si,ds:[Byte_Value]	; Get pointer to byte value format
	call	Printf			; Call routine to print the byte value
	Restore ax,bx,si,ds		; Restore the required registers
	ret				; Return to the caller
Print_Byte	Endp			; End of the Print_Byte procedure
	Subttl	Print_Word	Print Word Value Routine
	Page	+
;******************************************************************************
;
;	Print_Word()
;
;		Save the required registers
;		Call routine to print the word value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Print_Word	Proc	Near		; Print word value procedure
	Save	ax,bx,si,ds		; Save the required registers
	mov	ax,cs			; Setup the access
	mov	ds,ax			;		   the data segment
	lea	bx,ds:[Value_Arg]	; Get pointer to value argument list
	lea	si,ds:[Word_Value]	; Get pointer to word value format
	call	Printf			; Call routine to print the word value
	Restore ax,bx,si,ds		; Restore the required registers
	ret				; Return to the caller
Print_Word	Endp			; End of the Print_Word procedure
	Subttl	Dump_Data	Dump Data Routine
	Page	+
;******************************************************************************
;
;	Dump_Data(Data_Segment)
;
;		Save the required registers
;		Get and save the current window
;		Setup data window as current window
;		Setup the data color values
;		Call routine to clear data window
;		Position to data starting row and column
;		Compute the number of data lines to dump
;		Setup the initial data pointer
;		Update the last data dump address
;		While more data lines to dump
;			Get pointers to ASCII tables
;			Setup the current dump address
;			Call routine to print the address
;			Setup the data foreground color
;			Get the number of bytes per line
;			While there are more bytes
;				Call routine to read the 65C02 memory
;				Move data value to dump buffer
;				Translate data value to ASCII for dump
;			EndWhile
;			Call routine to print this data line
;		EndWhile
;		Restore the original window
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Dump_Data	Proc	Near		; Dump memory data procedure
	Save	ax,bx,cx,dx,si,di,bp,es ; Save the required registers
	call	Get_Window		; Call routine to get current window
	push	si			; Save the current window pointer
	lea	si,ds:[Data_Window]	; Get pointer to debug data window
	call	Set_Window		; Call routine to set current window
	mov	ax,ds:[Color_Table.Color_Address]
	mov	bx,ds:[Color_Table.Color_Back]
	call	Set_Attribute		; Setup the data color values
	call	Clear_Window		; Call routine to clear the window
	xor	ax,ax			; Setup the data
	xor	bx,bx			;		 row and column values
	call	Set_Position		; Call routine to set row and column
	mov	bp,cs:[si.Lower_Right_Row]
	sub	bp,cs:[si.Upper_Left_Row]
	inc	bp			; Compute the number of data lines
	mov	ax,ds			; Setup to access
	mov	es,ax			;		  the data segment
	mov	ds,cs:[RAM_Space]	; Setup initial
	mov	si,cs:[Data_Address]	;		data pointer
	mov	cs:[Last_Data],si	; Update the last data dump address
	jmp	Short Data_Entry	; Go start dumping the data
Data_Loop:
	Save	si,ds			; Save the actual data pointer
	mov	ax,es			; Setup access to
	mov	ds,ax			;		  the data segment
	lea	si,ds:[New_Line_Format] ; Get pointer to new line format string
	call	Printf			; Call routine to print this new line
	Restore si,ds			; Restore the actual data pointer
Data_Entry:
	lea	bx,cs:[ASCII_Table]	; Get pointer to ASCII table
	lea	dx,es:[Value_Table]	; Get pointer to value table
	lea	di,es:[ASCII_String]	; Get pointer to ASCII string area
	mov	cs:[Current_Address],si ; Save the current dump address
	call	Print_Address		; Call routine to print the address
	mov	ax,es:[Color_Table.Color_Data]
	call	Set_Foreground		; Setup the data foreground color
	mov	cx,LINE_COUNT		; Get the number of bytes per line
	push	bp			; Save the data line count value
Byte_Loop:
	xchg	si,di			; Setup the 65C02 effective address
	call	Read_Memory		; Call routine to read 65C02 memory
	xchg	si,di			; Restore the register order
	xchg	dx,di			; Get pointer into value table
	stosb				; Store the actual byte value
	xchg	dx,di			; Restore pointer to ASCII string
	test	al,RANGE_MASK		; Check for character code in range
	jz	Translate		; Jump if character code in range
	xor	al,al			; Use zero code to translate character
Translate:
	xlat	cs:[bx] 		; Translate the character code value
	stosb				; Store the translated character value
	inc	si			; Increment to next memory location
	loop	Byte_Loop		; Loop till all bytes are processed
	pop	bp			; Restore the data line counter value
Print_Data:
	Save	si,ds			; Save the actual data pointer
	mov	ax,es			; Setup access to
	mov	ds,ax			;		  the data segment
	mov	si,ds:[Dump_Format]	; Get the current data dump format
	mov	bx,ds:[Arg_Table + si]	; Get pointers to dump argument/format
	mov	si,ds:[Format_Table + si]
	call	Printf			; Call routine to print this line
	Restore si,ds			; Restore the actual data pointer
	dec	bp			; Decrement the line counter
	jnz	Data_Loop		; Loop until all data is dumped
	pop	si			; Restore the original window pointer
	call	Set_Window		; Call routine to set current window
	mov	ax,es			; Restore original
	mov	ds,ax			;		   data segment value
	Restore ax,bx,cx,dx,si,di,bp,es ; Restore the required registers
	ret				; Return to the caller
Dump_Data	Endp			; End of the Dump_Data procedure
	Subttl	Dump_Code	Dump Code Routine
	Page	+
;******************************************************************************
;
;	Dump_Code(Data_Segment, Stack_Frame)
;
;		Save the required registers
;		Get and save the current window
;		Setup code window as current window
;		Setup the code color values
;		Call routine to clear code window
;		Position to code starting row and column
;		Get the current 65C02 program counter value
;		Compute the number of code lines to dump
;		Setup the initial code pointer
;		Update the last code dump address
;		While more code lines to dump
;			Call routine to print the current address
;			Move opcode bytes to dump buffer
;			Call routine to print opcode bytes
;			Increment the current address value
;			Call routine to print the mnemonic
;		EndWhile
;		Restore the original window
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Dump_Code	Proc	Near		; Print debug title bar procedure
	Save	ax,bx,cx,dx,si,di,bp,es ; Save the required registers
	call	Get_Window		; Call routine to get current window
	push	si			; Save the current window pointer
	lea	si,ds:[Code_Window]	; Get pointer to debug code window
	call	Set_Window		; Call routine to set current window
	mov	ax,ds:[Color_Table.Color_Address]
	mov	bx,ds:[Color_Table.Color_Back]
	call	Set_Attribute		; Setup the code color values
	call	Clear_Window		; Call routine to clear the window
	xor	ax,ax			; Setup the code
	xor	bx,bx			;		 row and column values
	call	Set_Position		; Call routine to set row and column
	mov	ax,ss:[bp.SI_Register]	; Get the
	dec	ax			;	  65C02 program
	mov	ds:[Compare],ax 	;			counter (SI-1)
	mov	bp,cs:[si.Lower_Right_Row]
	sub	bp,cs:[si.Upper_Left_Row]
	inc	bp			; Compute the number of code lines
	mov	ax,ds			; Setup to access
	mov	es,ax			;		  the data segment
	mov	ds,cs:[RAM_Space]	; Setup initial
	mov	si,cs:[Code_Address]	;		code pointer
	mov	cs:[Last_Code],si	; Update the last code dump address
	mov	cs:[Current_Address],si ; Save the current dump address
	jmp	Short Code_Entry	; Go start dumping the code
Code_Loop:
	Save	si,ds			; Save the actual code pointer
	mov	ax,es			; Setup access to
	mov	ds,ax			;		  the data segment
	lea	si,ds:[New_Line_Format] ; Get pointer to new line format string
	call	Printf			; Call routine to print this new line
	Restore si,ds			; Restore the actual data pointer
Code_Entry:
	mov	si,cs:[Current_Address] ; Setup the current address value
	mov	cs:[Code_Address],si	; Update the code address value
	xor	ax,ax			; Default to non-highlight value
	cmp	si,cs:[Compare] 	; Check for current address match
	jne	Print_Code		; Jump if not an address match
	mov	ax,INTENSE		; Setup the highlight color value
Print_Code:
	mov	cs:[Highlight],ax	; Save the color highlight value
	call	Print_Address		; Call routine to print the address
	lea	di,es:[Value_Table]	; Get pointer to value table
	mov	cx,BYTE_COUNT		; Get number of bytes per line (Maximum)
	rep	movsb			; Move bytes into the value table
	push	ds			; Save the 65C02 RAM data segment
	mov	ax,es			; Setup access to
	mov	ds,ax			;		  the data segment
	call	Print_Opcode		; Call routine to print opcode bytes
	add	cs:[Current_Address],ax ; Increment the current address value
	call	Print_Mnemonic		; Call routine to print opcode mnemonic
	pop	ds			; Restore the 65C02 RAM data segment
	dec	bp			; Decrement the line counter
	jnz	Code_Loop		; Loop until all code is dumped
	pop	si			; Restore the original window pointer
	call	Set_Window		; Call routine to set current window
	mov	ax,es			; Restore original
	mov	ds,ax			;		   data segment value
	Restore ax,bx,cx,dx,si,di,bp,es ; Restore the required registers
	ret				; Return to the caller
Dump_Code	Endp			; End of the Dump_Code procedure
	Subttl	Dump_Stack	Dump Stack Routine
	Page	+
;******************************************************************************
;
;	Dump_Stack(Data_Segment, Stack_Frame)
;
;		Save the required registers
;		Get and save the current window
;		Setup stack window as current window
;		Setup the stack color values
;		Call routine to clear stack window
;		Position to stack starting row and column
;		Compute the number of stack lines to dump
;		While more stack lines to dump
;			Get next byte of the stack data
;			Call routine print the stack data
;		EndWhile
;		Restore the original window
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Dump_Stack	Proc	Near		; Dump stack data procedure
	Save	ax,bx,cx,dx,si,di,bp,es ; Save the required registers
	call	Get_Window		; Call routine to get current window
	push	si			; Save the current window pointer
	lea	si,ds:[Stack_Window]	; Get pointer to debug stack window
	call	Set_Window		; Call routine to set current window
	mov	ax,ds:[Color_Table.Color_Stack]
	mov	bx,ds:[Color_Table.Color_Back]
	call	Set_Attribute		; Setup the stack color values
	call	Clear_Window		; Call routine to clear the window
	xor	ax,ax			; Setup the stack
	xor	bx,bx			;		  row and column values
	call	Set_Position		; Call routine to set row and column
	mov	dx,cs:[si.Lower_Right_Row]
	sub	dx,cs:[si.Upper_Left_Row]
	inc	dx			; Compute the number of stack lines
	mov	ax,ds			; Setup to access
	mov	es,ax			;		  the data segment
	mov	ds,cs:[RAM_Space]	; Setup initial
	mov	si,ss:[bp.BX_Register]	;		data pointer
	inc	si			;			     (Stack)
	mov	bp,dx			; Setup the number of lines to dump
	jmp	Short Stack_Entry	; Go start dumping the stack
Stack_Loop:
	Save	si,ds			; Save the actual data pointer
	mov	ax,es			; Setup access to
	mov	ds,ax			;		  the data segment
	lea	si,ds:[New_Line_Format] ; Get pointer to new line format string
	call	Printf			; Call routine to print this new line
	Restore si,ds			; Restore the actual data pointer
Stack_Entry:
	lodsb				; Get a byte of the stack data
	mov	Byte Ptr es:[Value],al	; Store the stack data value
	call	Print_Byte		; Call routine to print the byte value
	dec	bp			; Decrement the line counter
	jnz	Stack_Loop		; Loop until all stack is dumped
	pop	si			; Restore the original window pointer
	call	Set_Window		; Call routine to set current window
	mov	ax,es			; Restore original
	mov	ds,ax			;		   data segment value
	Restore ax,bx,cx,dx,si,di,bp,es ; Restore the required registers
	ret				; Return to the caller
Dump_Stack	Endp			; End of the Dump_Stack procedure
	Subttl	Do_BAD		Bad Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_BAD(Data_Segment)
;
;		Call routine to print bad addressing mode format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_BAD		Proc	Near		; Bad addressing mode procedure
	lea	bx,ds:[BAD_Arg] 	; Get pointer to argument list
	lea	si,ds:[BAD_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_BAD		Endp			; End of the Do_BAD procedure
	Subttl	Do_NONE 	No Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_NONE(Data_Segment)
;
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_NONE 	Proc	Near		; No addressing mode procedure
	ret				; Return to the caller
Do_NONE 	Endp			; End of the Do_NONE procedure
	Subttl	Do_IMM		Immediate Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_IMM(Data_Segment)
;
;		Call routine to print immediate addressing mode format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_IMM		Proc	Near		; Immediate addressing mode procedure
	lea	bx,ds:[IMM_Arg] 	; Get pointer to argument list
	lea	si,ds:[IMM_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_IMM		Endp			; End of the Do_IMM procedure
	Subttl	Do_ABS		Absolute Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_ABS(Data_Segment)
;
;		Call routine to print absolute addressing mode format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_ABS		Proc	Near		; Absolute addressing mode procedure
	lea	bx,ds:[ABS_Arg] 	; Get pointer to argument list
	lea	si,ds:[ABS_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_ABS		Endp			; End of the Do_ABS procedure
	Subttl	Do_DP		Direct Page Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_DP(Data_Segment)
;
;		Call routine to print direct page addressing mode format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_DP		Proc	Near		; Direct page addressing mode procedure
	lea	bx,ds:[DP_Arg]		; Get pointer to argument list
	lea	si,ds:[DP_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_DP		Endp			; End of the Do_DP procedure
	Subttl	Do_DIX		Direct Page Indexed X Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_DIX(Data_Segment)
;
;		Call routine to print direct page indexed X addressing format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_DIX		Proc	Near		; Direct page indexed X procedure
	lea	bx,ds:[DIX_Arg] 	; Get pointer to argument list
	lea	si,ds:[DIX_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_DIX		Endp			; End of the Do_DIX procedure
	Subttl	Do_DIY		Direct Page Indexed Y Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_DIY(Data_Segment)
;
;		Call routine to print direct page indexed Y addressing format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_DIY		Proc	Near		; Direct page indexed Y procedure
	lea	bx,ds:[DIY_Arg] 	; Get pointer to argument list
	lea	si,ds:[DIY_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_DIY		Endp			; End of the Do_DIY procedure
	Subttl	Do_AIX		Absolute Indexed X Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_AIX(Data_Segment)
;
;		Call routine to print absolute indexed X addressing mode format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_AIX		Proc	Near		; Absolute indexed X procedure
	lea	bx,ds:[AIX_Arg] 	; Get pointer to argument list
	lea	si,ds:[AIX_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_AIX		Endp			; End of the Do_AIX procedure
	Subttl	Do_AIY		Absolute Indexed Y Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_AIY(Data_Segment)
;
;		Call routine to print absolute indexed Y addressing mode format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_AIY		Proc	Near		; Absolute indexed Y procedure
	lea	bx,ds:[AIY_Arg] 	; Get pointer to argument list
	lea	si,ds:[AIY_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_AIY		Endp			; End of the Do_AIY procedure
	Subttl	Do_AI		Absolute Indirect Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_AI(Data_Segment)
;
;		Call routine to print absolute indirect addressing mode format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_AI		Proc	Near		; Absolute indirect procedure
	lea	bx,ds:[AI_Arg]		; Get pointer to argument list
	lea	si,ds:[AI_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_AI		Endp			; End of the Do_AI procedure
	Subttl	Do_DI		Direct Page Indirect Addressing Mode Routine
	Page	+
;******************************************************************************
;
;	Do_DI(Data_Segment)
;
;		Call routine to print direct page indirect addressing format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_DI		Proc	Near		; Direct page indirect procedure
	lea	bx,ds:[DI_Arg]		; Get pointer to argument list
	lea	si,ds:[DI_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_DI		Endp			; End of the Do_DI procedure
	Subttl	Do_AIIX 	Absolute Indexed Indirect X Addressing Routine
	Page	+
;******************************************************************************
;
;	Do_AIIX(Data_Segment)
;
;		Call routine to print absolute indexed indirect X addressing
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_AIIX 	Proc	Near		; Absolute indexed indirect X procedure
	lea	bx,ds:[AIIX_Arg]	; Get pointer to argument list
	lea	si,ds:[AIIX_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_AIIX 	Endp			; End of the Do_AIIX procedure
	Subttl	Do_DIIX 	Direct Page Indexed Indirect X Mode Routine
	Page	+
;******************************************************************************
;
;	Do_DIIX(Data_Segment)
;
;		Call routine to print direct page indexed indirect X addressing
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_DIIX 	Proc	Near		; Direct indexed indirect X procedure
	lea	bx,ds:[DIIX_Arg]	; Get pointer to argument list
	lea	si,ds:[DIIX_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_DIIX 	Endp			; End of the Do_DIIX procedure
	Subttl	Do_DIIY 	Direct Page Indexed Indirect Y Mode Routine
	Page	+
;******************************************************************************
;
;	Do_DIIY(Data_Segment)
;
;		Call routine to print direct page indexed indirect Y addressing
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_DIIY 	Proc	Near		; Direct indexed indirect Y procedure
	lea	bx,ds:[DIIY_Arg]	; Get pointer to argument list
	lea	si,ds:[DIIY_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_DIIY 	Endp			; End of the Do_DIIY procedure
	Subttl	Do_PCR		Program Counter Relative Addressing Routine
	Page	+
;******************************************************************************
;
;	Do_PCR(Data_Segment)
;
;		Call routine to print program counter relative addressing format
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Do_PCR		Proc	Near		; PC relative addressing procedure
	mov	al,ds:[Value_1] 	; Get the actual offset value
	cbw				; Convert offset value to a full word
	add	ax,ds:[Current_Address] ; Compute the actual address value
	mov	ds:[Value],ax		; Save the computed address value
	lea	bx,ds:[PCR_Arg] 	; Get pointer to argument list
	lea	si,ds:[PCR_Format]	; Get pointer to format string
	call	Printf			; Call routine to print address
	ret				; Return to the caller
Do_PCR		Endp			; End of the Do_PCR procedure
;******************************************************************************
;
;	Define the 65C02 debugger addressing table
;
;******************************************************************************
	Even				; Force table to an even address
Address_Table	Equ	This Word	; 65C02 Opcode addressing table
	Address BAD,Do_BAD		; Illegal opcode addressing type
	Address NONE,Do_NONE		; No addressing type
	Address IMM,Do_IMM		; Immediate addressing type
	Address ABS,Do_ABS		; Absolute addressing type
	Address DP,Do_DP		; Direct page addressing type
	Address DIX,Do_DIX		; Direct page indexed with X type
	Address DIY,Do_DIY		; Direct page indexed with Y type
	Address AIX,Do_AIX		; Absolute indexed with X type
	Address AIY,Do_AIY		; Absolute indexed with Y type
	Address AI,Do_AI		; Absolute indirect addressing type
	Address DI,Do_DI		; Direct page indirect addressing type
	Address AIIX,Do_AIIX		; Absolute indexed indirect X type
	Address DIIX,Do_DIIX		; Direct page indexed indirect X type
	Address DIIY,Do_DIIY		; Direct page indexed indirect Y type
	Address PCR,Do_PCR		; Program counter relative type
;******************************************************************************
;
;	Define the 65C02 debugger addressing type length table
;
;******************************************************************************
	Even				; Force table to an even address
Length_Table	Equ	This Word	; 65C02 addressing length table
	Dw	1			; Illegal opcode length
	Dw	1			; No address length
	Dw	2			; Immediate address length
	Dw	3			; Absolute addressing length
	Dw	2			; Direct page addressing length
	Dw	2			; Direct page indexed with X length
	Dw	2			; Direct page indexed with Y length
	Dw	3			; Absolute indexed with X length
	Dw	3			; Absolute indexed with Y length
	Dw	3			; Absolute indirect addressing length
	Dw	2			; Direct page indirect addressing length
	Dw	3			; Absolute indexed indirect X length
	Dw	2			; Direct page indexed indirect X length
	Dw	2			; Direct page indexed indirect Y length
	Dw	2			; Program counter relative length
;******************************************************************************
;
;	Define the 65C02 debugger mnemonic table
;
;******************************************************************************
	Even				; Force table to an even address
Mnemonic_Table	Equ	This Byte	; 65C02 Opcode mnemonic table
	Mnemonic	ADC,<"ADC">,READ        ; Add with carry mnemonic
	Mnemonic	AND,<"AND">,READ        ; AND accumulator mnemonic
	Mnemonic	ASL,<"ASL">,READ        ; Arithmetic shift left mnemonic
	Mnemonic	BAD,<"DB ">,NONE        ; Illegal opcode mnemonic
	Mnemonic	BCC,<"BCC">,READ        ; Branch if carry clear mnemonic
	Mnemonic	BCS,<"BCS">,READ        ; Branch if carry set mnemonic
	Mnemonic	BEQ,<"BEQ">,READ        ; Branch if equal mnemonic
	Mnemonic	BIT,<"BIT">,READ        ; Test bits mnemonic
	Mnemonic	BMI,<"BMI">,READ        ; Branch if minus mnemonic
	Mnemonic	BNE,<"BNE">,READ        ; Branch if not equal mnemonic
	Mnemonic	BPL,<"BPL">,READ        ; Branch if plus mnemonic
	Mnemonic	BRA,<"BRA">,READ        ; Branch always mnemonic
	Mnemonic	BRK,<"BRK">,NONE        ; Software break mnemonic
	Mnemonic	BVC,<"BVC">,READ        ; Branch if overflow clear mne.
	Mnemonic	BVS,<"BVS">,READ        ; Branch if overflow set mne.
	Mnemonic	CLC,<"CLC">,READ        ; Branch if carry clear mnemonic
	Mnemonic	CLD,<"CLD">,NONE        ; Clear decimal mode mnemonic
	Mnemonic	CLI,<"CLI">,NONE        ; Clear interrupt disable mne.
	Mnemonic	CLV,<"CLV">,NONE        ; Clear overflow mnemonic
	Mnemonic	CMP,<"CMP">,READ        ; Compare accumulator mnemonic
	Mnemonic	CPX,<"CPX">,READ        ; Compare X register mnemonic
	Mnemonic	CPY,<"CPY">,READ        ; Compare Y register mnemonic
	Mnemonic	DEC,<"DEC">,ACCESS      ; Decrement mnemonic
	Mnemonic	DEX,<"DEX">,NONE        ; Decrement X register mnemonic
	Mnemonic	DEY,<"DEY">,NONE        ; Decrement Y register mnemonic
	Mnemonic	EOR,<"EOR">,READ        ; XOR with accumulator mnemonic
	Mnemonic	INC,<"INC">,ACCESS      ; Increment mnemonic
	Mnemonic	INX,<"INX">,NONE        ; Increment X register mnemonic
	Mnemonic	INY,<"INY">,NONE        ; Increment Y register mnemonic
	Mnemonic	JMP,<"JMP">,READ        ; Jump mnemonic
	Mnemonic	JSR,<"JSR">,READ        ; Jump to subroutine mnemonic
	Mnemonic	LDA,<"LDA">,READ        ; Load accumulator mnemonic
	Mnemonic	LDX,<"LDX">,READ        ; Load X register mnemonic
	Mnemonic	LDY,<"LDY">,READ        ; Load Y register mnemonic
	Mnemonic	LSR,<"LSR">,ACCESS      ; Logical shift right mnemonic
	Mnemonic	NOP,<"NOP">,NONE        ; No operation mnemonic
	Mnemonic	ORA,<"ORA">,READ        ; OR with accumulator mnemonic
	Mnemonic	PHA,<"PHA">,WRITE       ; Push accumulator mnemonic
	Mnemonic	PHP,<"PHP">,WRITE       ; Push processor status mnemonic
	Mnemonic	PHX,<"PHX">,WRITE       ; Push X register mnemonic
	Mnemonic	PHY,<"PHY">,WRITE       ; Push Y register mnemonic
	Mnemonic	PLA,<"PLA">,READ        ; Pull accumulator mnemonic
	Mnemonic	PLP,<"PLP">,READ        ; Pull processor status mnemonic
	Mnemonic	PLX,<"PLX">,READ        ; Pull X register mnemonic
	Mnemonic	PLY,<"PLY">,READ        ; Pull Y register mnemonic
	Mnemonic	ROL,<"ROL">,ACCESS      ; Rotate left mnemonic
	Mnemonic	ROR,<"ROR">,ACCESS      ; Rotate right mnemonic
	Mnemonic	RTI,<"RTI">,READ        ; Return from interrupt mnemonic
	Mnemonic	RTS,<"RTS">,READ        ; Return from sub. mnemonic
	Mnemonic	SBC,<"SBC">,READ        ; Subtract with borrow mnemonic
	Mnemonic	SEC,<"SEC">,NONE        ; Set carry mnemonic
	Mnemonic	SED,<"SED">,NONE        ; Set decimal mode mnemonic
	Mnemonic	SEI,<"SEI">,NONE        ; Set interrupt disable mnemonic
	Mnemonic	STA,<"STA">,WRITE       ; Store accumulator mnemonic
	Mnemonic	STX,<"STX">,WRITE       ; Store X register mnemonic
	Mnemonic	STY,<"STY">,WRITE       ; Store Y register mnemonic
	Mnemonic	STZ,<"STZ">,WRITE       ; Store zero mnemonic
	Mnemonic	TAX,<"TAX">,NONE        ; Transfer acc. to X mnemonic
	Mnemonic	TAY,<"TAY">,NONE        ; Transfer acc. to Y mnemonic
	Mnemonic	TRB,<"TRB">,ACCESS      ; Test/reset bits mnemonic
	Mnemonic	TSB,<"TSB">,ACCESS      ; Test/set bits mnemonic
	Mnemonic	TSX,<"TSX">,NONE        ; Transfer stack to X mnemonic
	Mnemonic	TXA,<"TXA">,NONE        ; Transfer X to acc. mnemonic
	Mnemonic	TXS,<"TXS">,NONE        ; Transfer X to stack mnemonic
	Mnemonic	TYA,<"TYA">,NONE        ; Transfer Y to acc. mnemonic
;******************************************************************************
;
;	Define the 65C02 debugger opcode data table (Opcode, Addressing Mode)
;
;******************************************************************************
	Even				; Force table to an even address
Opcode_Table	Equ	This Byte		; 65C02 Opcode data table
	Opcode		BRK,NONE		; 00h - Software break
	Opcode		ORA,DIIX		; 01h - OR accumulator w/memory
	Opcode		BAD,BAD 		; 02h - Illegal opcode
	Opcode		BAD,BAD 		; 03h - Illegal opcode
	Opcode		TSB,DP			; 04h - Test/set bits w/acc.
	Opcode		ORA,DP			; 05h - OR accumulator w/memory
	Opcode		ASL,DP			; 06h - Shift memory/acc. left
	Opcode		BAD,BAD 		; 07h - Illegal opcode
	Opcode		PHP,NONE		; 08h - Push processor status
	Opcode		ORA,IMM 		; 09h - OR accumulator w/memory
	Opcode		ASL,NONE		; 0Ah - Shift memory/acc. left
	Opcode		BAD,BAD 		; 0Bh - Illegal opcode
	Opcode		TSB,ABS 		; 0Ch - Test/set bits w/acc.
	Opcode		ORA,ABS 		; 0Dh - OR accumulator w/memory
	Opcode		ASL,ABS 		; 0Eh - Shift memory/acc. left
	Opcode		BAD,BAD 		; 0Fh - Illegal opcode
	Opcode		BPL,PCR 		; 10h - Branch if plus
	Opcode		ORA,DIIY		; 11h - OR accumulator w/memory
	Opcode		ORA,DI			; 12h - OR accumulator w/memory
	Opcode		BAD,BAD 		; 13h - Illegal opcode
	Opcode		TRB,DP			; 14h - Test/reset bits w/acc.
	Opcode		ORA,DIX 		; 15h - OR accumulator w/memory
	Opcode		ASL,DIX 		; 16h - Shift memory/acc. left
	Opcode		BAD,BAD 		; 17h - Illegal opcode
	Opcode		CLC,NONE		; 18h - Clear carry
	Opcode		ORA,AIY 		; 19h - OR accumulator w/memory
	Opcode		INC,NONE		; 1Ah - Increment
	Opcode		BAD,BAD 		; 1Bh - Illegal opcode
	Opcode		TRB,ABS 		; 1Ch - Test/reset bits w/acc.
	Opcode		ORA,AIX 		; 1Dh - OR accumulator w/memory
	Opcode		ASL,AIX 		; 1Eh - Shift memory/acc. left
	Opcode		BAD,BAD 		; 1Fh - Illegal opcode
	Opcode		JSR,ABS 		; 20h - Jump to subroutine
	Opcode		AND,DIIX		; 21h - AND accumulator w/memory
	Opcode		BAD,BAD 		; 22h - Illegal opcode
	Opcode		BAD,BAD 		; 23h - Illegal opcode
	Opcode		BIT,DP			; 24h - Test bits w/accumulator
	Opcode		AND,DP			; 25h - AND accumulator w/memory
	Opcode		ROL,DP			; 26h - Rotate memory/acc. left
	Opcode		BAD,BAD 		; 27h - Illegal opcode
	Opcode		PLP,NONE		; 28h - Pull processor status
	Opcode		AND,IMM 		; 29h - AND accumulator w/memory
	Opcode		ROL,NONE		; 2Ah - Rotate memory/acc. left
	Opcode		BAD,BAD 		; 2Bh - Illegal opcode
	Opcode		BIT,ABS 		; 2Ch - Test bits w/accumulator
	Opcode		AND,ABS 		; 2Dh - AND accumulator w/memory
	Opcode		ROL,ABS 		; 2Eh - Rotate memory/acc. left
	Opcode		BAD,BAD 		; 2Fh - Illegal opcode
	Opcode		BMI,PCR 		; 30h - Branch if minus
	Opcode		AND,DIIY		; 31h - AND accumulator w/memory
	Opcode		AND,DI			; 32h - AND accumulator w/memory
	Opcode		BAD,BAD 		; 33h - Illegal opcode
	Opcode		BIT,DIX 		; 34h - Test bits w/accumulator
	Opcode		AND,DIX 		; 35h - AND accumulator w/memory
	Opcode		ROL,DIX 		; 36h - Rotate memory/acc. left
	Opcode		BAD,BAD 		; 37h - Illegal opcode
	Opcode		SEC,NONE		; 38h - Set carry
	Opcode		AND,AIY 		; 39h - AND accumulator w/memory
	Opcode		DEC,NONE		; 3Ah - Decrement
	Opcode		BAD,BAD 		; 3Bh - Illegal opcode
	Opcode		BIT,AIX 		; 3Ch - Test bits w/accumulator
	Opcode		AND,AIX 		; 3Dh - AND accumulator w/memory
	Opcode		ROL,AIX 		; 3Eh - Rotate memory/acc. left
	Opcode		BAD,BAD 		; 3Fh - Illegal opcode
	Opcode		RTI,NONE		; 40h - Return from interrupt
	Opcode		EOR,DIIX		; 41h - XOR accumulator w/memory
	Opcode		BAD,BAD 		; 42h - Illegal opcode
	Opcode		BAD,BAD 		; 43h - Illegal opcode
	Opcode		BAD,BAD 		; 44h - Illegal opcode
	Opcode		EOR,DP			; 45h - XOR accumulator w/memory
	Opcode		LSR,DP			; 46h - Logical shift right
	Opcode		BAD,BAD 		; 47h - Illegal opcode
	Opcode		PHA,NONE		; 48h - Push accumulator
	Opcode		EOR,IMM 		; 49h - XOR accumulator w/memory
	Opcode		LSR,NONE		; 4Ah - Logical shift right
	Opcode		BAD,BAD 		; 4Bh - Illegal opcode
	Opcode		JMP,ABS 		; 4Ch - Jump
	Opcode		EOR,ABS 		; 4Dh - XOR accumulator w/memory
	Opcode		LSR,ABS 		; 4Eh - Logical shift right
	Opcode		BAD,BAD 		; 4Fh - Illegal opcode
	Opcode		BVC,PCR 		; 50h - Branch if overflow clear
	Opcode		EOR,DIIY		; 51h - XOR accumulator w/memory
	Opcode		EOR,DI			; 52h - XOR accumulator w/memory
	Opcode		BAD,BAD 		; 53h - Illegal opcode
	Opcode		BAD,BAD 		; 54h - Illegal opcode
	Opcode		EOR,DIX 		; 55h - XOR accumulator w/memory
	Opcode		LSR,DIX 		; 56h - Logical shift right
	Opcode		BAD,BAD 		; 57h - Illegal opcode
	Opcode		CLI,NONE		; 58h - Clear interrupt disable
	Opcode		EOR,AIY 		; 59h - XOR accumulator w/memory
	Opcode		PHY,NONE		; 5Ah - Push Y index register
	Opcode		BAD,BAD 		; 5Bh - Illegal opcode
	Opcode		BAD,BAD 		; 5Ch - Illegal opcode
	Opcode		EOR,AIX 		; 5Dh - XOR accumulator w/memory
	Opcode		LSR,AIX 		; 5Eh - Logical shift right
	Opcode		BAD,BAD 		; 5Fh - Illegal opcode
	Opcode		RTS,NONE		; 60h - Return from subroutine
	Opcode		ADC,DIIX		; 61h - Add with carry
	Opcode		BAD,BAD 		; 62h - Illegal opcode
	Opcode		BAD,BAD 		; 63h - Illegal opcode
	Opcode		STZ,DP			; 64h - Store zero to memory
	Opcode		ADC,DP			; 65h - Add with carry
	Opcode		ROR,DP			; 66h - Rotate memory/acc. right
	Opcode		BAD,BAD 		; 67h - Illegal opcode
	Opcode		PLA,NONE		; 68h - Pull accumulator
	Opcode		ADC,IMM 		; 69h - Add with carry
	Opcode		ROR,NONE		; 6Ah - Rotate memory/acc. right
	Opcode		BAD,BAD 		; 6Bh - Illegal opcode
	Opcode		JMP,AI			; 6Ch - Jump
	Opcode		ADC,ABS 		; 6Dh - Add with carry
	Opcode		ROR,ABS 		; 6Eh - Rotate memory/acc. right
	Opcode		BAD,BAD 		; 6Fh - Illegal opcode
	Opcode		BVS,PCR 		; 70h - Branch if overflow set
	Opcode		ADC,DIIY		; 71h - Add with carry
	Opcode		ADC,DI			; 72h - Add with carry
	Opcode		BAD,BAD 		; 73h - Illegal opcode
	Opcode		STZ,DIX 		; 74h - Store zero to memory
	Opcode		ADC,DIX 		; 75h - Add with carry
	Opcode		ROR,DIX 		; 76h - Rotate memory/acc. right
	Opcode		BAD,BAD 		; 77h - Illegal opcode
	Opcode		SEI,NONE		; 78h - Set interrupt disable
	Opcode		ADC,AIY 		; 79h - Add with carry
	Opcode		PLY,NONE		; 7Ah - Pull Y index register
	Opcode		BAD,BAD 		; 7Bh - Illegal opcode
	Opcode		JMP,AIIX		; 7Ch - Jump
	Opcode		ADC,AIX 		; 7Dh - Add with carry
	Opcode		ROR,AIX 		; 7Eh - Rotate memory/acc. right
	Opcode		BAD,BAD 		; 7Fh - Illegal opcode
	Opcode		BRA,PCR 		; 80h - Branch always
	Opcode		STA,DIIX		; 81h - Store accumulator
	Opcode		BAD,BAD 		; 82h - Illegal opcode
	Opcode		BAD,BAD 		; 83h - Illegal opcode
	Opcode		STY,DP			; 84h - Store Y index register
	Opcode		STA,DP			; 85h - Store accumulator
	Opcode		STX,DP			; 86h - Store X index register
	Opcode		BAD,BAD 		; 87h - Illegal opcode
	Opcode		DEY,NONE		; 88h - Decrement Y register
	Opcode		BIT,IMM 		; 89h - Test bits w/accumulator
	Opcode		TXA,NONE		; 8Ah - Transfer X reg. to acc.
	Opcode		BAD,BAD 		; 8Bh - Illegal opcode
	Opcode		STY,ABS 		; 8Ch - Store Y index register
	Opcode		STA,ABS 		; 8Dh - Store accumulator
	Opcode		STX,ABS 		; 8Eh - Store X index register
	Opcode		BAD,BAD 		; 8Fh - Illegal opcode
	Opcode		BCC,PCR 		; 90h - Branch if carry clear
	Opcode		STA,DIIY		; 91h - Store accumulator
	Opcode		STA,DI			; 92h - Store accumulator
	Opcode		BAD,BAD 		; 93h - Illegal opcode
	Opcode		STY,DIX 		; 94h - Store Y index register
	Opcode		STA,DIX 		; 95h - Store accumulator
	Opcode		STX,DIX 		; 96h - Store X index register
	Opcode		BAD,BAD 		; 97h - Illegal opcode
	Opcode		TYA,NONE		; 98h - Transfer Y reg. to acc.
	Opcode		STA,AIY 		; 99h - Store accumulator
	Opcode		TXS,NONE		; 9Ah - Transfer X reg. to stack
	Opcode		BAD,BAD 		; 9Bh - Illegal opcode
	Opcode		STZ,ABS 		; 9Ch - Store zero to memory
	Opcode		STA,AIX 		; 9Dh - Store accumulator
	Opcode		STZ,AIX 		; 9Eh - Store zero to memory
	Opcode		BAD,BAD 		; 9Fh - Illegal opcode
	Opcode		LDY,IMM 		; A0h - Load Y register w/memory
	Opcode		LDA,DIIX		; A1h - Load acc. with memory
	Opcode		LDX,IMM 		; A2h - Load X register w/memory
	Opcode		BAD,BAD 		; A3h - Illegal opcode
	Opcode		LDY,DP			; A4h - Load Y register w/memory
	Opcode		LDA,DP			; A5h - Load acc. with memory
	Opcode		LDX,DP			; A6h - Load X register w/memory
	Opcode		BAD,BAD 		; A7h - Illegal opcode
	Opcode		TAY,NONE		; A8h - Transfer acc. to Y reg.
	Opcode		LDA,IMM 		; A9h - Load acc. with memory
	Opcode		TAX,NONE		; AAh - Transfer acc. to X reg.
	Opcode		BAD,BAD 		; ABh - Illegal opcode
	Opcode		LDY,ABS 		; ACh - Load Y register w/memory
	Opcode		LDA,ABS 		; ADh - Load acc. with memory
	Opcode		LDX,ABS 		; AEh - Load X register w/memory
	Opcode		BAD,BAD 		; AFh - Illegal opcode
	Opcode		BCS,PCR 		; B0h - Branch if carry set
	Opcode		LDA,DIIY		; B1h - Load acc. with memory
	Opcode		LDA,DI			; B2h - Load acc. with memory
	Opcode		BAD,BAD 		; B3h - Illegal opcode
	Opcode		LDY,DIX 		; B4h - Load Y register w/memory
	Opcode		LDA,DIX 		; B5h - Load acc. with memory
	Opcode		LDX,DIY 		; B6h - Load X register w/memory
	Opcode		BAD,BAD 		; B7h - Illegal opcode
	Opcode		CLV,NONE		; B8h - Clear overflow
	Opcode		LDA,AIY 		; B9h - Load acc. with memory
	Opcode		TSX,NONE		; BAh - Transfer stack to X reg.
	Opcode		BAD,BAD 		; BBh - Illegal opcode
	Opcode		LDY,AIX 		; BCh - Load Y register w/memory
	Opcode		LDA,AIX 		; BDh - Load acc. with memory
	Opcode		LDX,AIY 		; BEh - Load X register w/memory
	Opcode		BAD,BAD 		; BFh - Illegal opcode
	Opcode		CPY,IMM 		; C0h - Compare Y reg. w/memory
	Opcode		CMP,DIIX		; C1h - Compare memory w/acc.
	Opcode		BAD,BAD 		; C2h - Illegal opcode
	Opcode		BAD,BAD 		; C3h - Illegal opcode
	Opcode		CPY,DP			; C4h - Compare Y reg. w/memory
	Opcode		CMP,DP			; C5h - Compare memory w/acc.
	Opcode		DEC,DP			; C6h - Decrement
	Opcode		BAD,BAD 		; C7h - Illegal opcode
	Opcode		INY,NONE		; C8h - Increment Y register
	Opcode		CMP,IMM 		; C9h - Compare memory w/acc.
	Opcode		DEX,NONE		; CAh - Decrement X register
	Opcode		BAD,BAD 		; CBh - Illegal opcode
	Opcode		CPY,ABS 		; CCh - Compare Y reg. w/memory
	Opcode		CMP,ABS 		; CDh - Compare memory w/acc.
	Opcode		DEC,ABS 		; CEh - Decrement
	Opcode		BAD,BAD 		; CFh - Illegal opcode
	Opcode		BNE,PCR 		; D0h - Branch if not equal
	Opcode		CMP,DIIY		; D1h - Compare memory w/acc.
	Opcode		CMP,DI			; D2h - Compare memory w/acc.
	Opcode		BAD,BAD 		; D3h - Illegal opcode
	Opcode		BAD,BAD 		; D4h - Illegal opcode
	Opcode		CMP,DIX 		; D5h - Compare memory w/acc.
	Opcode		DEC,DIX 		; D6h - Decrement
	Opcode		BAD,BAD 		; D7h - Illegal opcode
	Opcode		CLD,NONE		; D8h - Clear decimal mode
	Opcode		CMP,AIY 		; D9h - Compare memory w/acc.
	Opcode		PHX,NONE		; DAh - Push X index register
	Opcode		BAD,BAD 		; DBh - Illegal opcode
	Opcode		BAD,BAD 		; DCh - Illegal opcode
	Opcode		CMP,AIX 		; DDh - Compare memory w/acc.
	Opcode		DEC,AIX 		; DEh - Decrement
	Opcode		BAD,BAD 		; DFh - Illegal opcode
	Opcode		CPX,IMM 		; E0h - Compare X reg. w/memory
	Opcode		SBC,DIIX		; E1h - Subtract from acc.
	Opcode		BAD,BAD 		; E2h - Illegal opcode
	Opcode		BAD,BAD 		; E3h - Illegal opcode
	Opcode		CPX,DP			; E4h - Compare X reg. w/memory
	Opcode		SBC,DP			; E5h - Subtract from acc.
	Opcode		INC,DP			; E6h - Increment
	Opcode		BAD,BAD 		; E7h - Illegal opcode
	Opcode		INX,NONE		; E8h - Increment X register
	Opcode		SBC,IMM 		; E9h - Subtract from acc.
	Opcode		NOP,NONE		; EAh - No operation
	Opcode		BAD,BAD 		; EBh - Illegal opcode
	Opcode		CPX,ABS 		; ECh - Compare X reg. w/memory
	Opcode		SBC,ABS 		; EDh - Subtract from acc.
	Opcode		INC,ABS 		; EEh - Increment
	Opcode		BAD,BAD 		; EFh - Illegal opcode
	Opcode		BEQ,PCR 		; F0h - Branch if equal
	Opcode		SBC,DIIY		; F1h - Subtract from acc.
	Opcode		SBC,DI			; F2h - Subtract from acc.
	Opcode		BAD,BAD 		; F3h - Illegal opcode
	Opcode		BAD,BAD 		; F4h - Illegal opcode
	Opcode		SBC,DIX 		; F5h - Subtract from acc.
	Opcode		INC,DIX 		; F6h - Increment
	Opcode		BAD,BAD 		; F7h - Illegal opcode
	Opcode		SED,NONE		; F8h - Set decimal mode
	Opcode		SBC,AIY 		; F9h - Subtract from acc.
	Opcode		PLX,NONE		; FAh - Pull X index register
	Opcode		BAD,BAD 		; FBh - Illegal opcode
	Opcode		BAD,BAD 		; FCh - Illegal opcode
	Opcode		SBC,AIX 		; FDh - Subtract from acc.
	Opcode		INC,AIX 		; FEh - Increment
	Opcode		BAD,BAD 		; FFh - Illegal opcode
;******************************************************************************
;
;	Define the ASCII character table
;
;******************************************************************************
ASCII_Table	Equ	This Byte	; Start of ASCII character table
	Db	"....","...."
	Db	"....","...."
	Db	"....","...."
	Db	"....","...."
	Db	' !"#',"$%&'"
	Db	"()*+",",-./"
	Db	"0123","4567"
	Db	"89:;","<=>?"
	Db	"@ABC","DEFG"
	Db	"HIJK","LMNO"
	Db	"PQRS","TUVW"
	Db	"XYZ[","\]^_"
	Db	"`abc","defg"
	Db	"hijk","lmno"
	Db	"pqrs","tuvw"
	Db	"xyz{","|}~ "
;******************************************************************************
;
;	Define the data/code dump argument/format tables
;
;******************************************************************************
Arg_Table	Equ	This Word	; Data dump argument table
	Dw	Offset Byte_Arg
	Dw	Offset Word_Arg
Format_Table	Equ	This Word	; Data dump format table
	Dw	Offset Byte_Format
	Dw	Offset Word_Format
Byte_Table	Equ	This Word	; Code dump format table
	Dw	Offset One_Format
	Dw	Offset Two_Format
	Dw	Offset Three_Format
;******************************************************************************
;
;	Define all the debug message strings
;
;******************************************************************************
New_Line_Format Db	"\n",0          ; New line format string
Debug_Title	Db	"        Apple ][ System Debugger\n"
		Db	"  ADDRESS ---------------------- ASCII",0
Address_Arg	Equ	This Word	; Print address argument list
		Dw	Offset Current_Address
Address_Format	Db	"%04X  ",0
Byte_Arg	Equ	This Word	; Byte data line argument list
		Dw	Offset Value_0
		Dw	Offset Value_1
		Dw	Offset Value_2
		Dw	Offset Value_3
		Dw	Offset Value_4
		Dw	Offset Value_5
		Dw	Offset Value_6
		Dw	Offset Value_7
		Dw	Offset ASCII_String
Byte_Format	Db	"%02tX %02tX %02tX %02tX %02tX %02tX %02tX %02tX  %s",0
Word_Arg	Equ	This Word	; Word data line argument list
		Dw	Offset Value_0
		Dw	Offset Value_2
		Dw	Offset Value_4
		Dw	Offset Value_6
		Dw	Offset ASCII_String
Word_Format	Db	"%04X  %04X  %04X  %04X   %s",0
Code_Arg	Equ	This Word	; Opcode bytes line argument list
		Dw	Offset Value_0
		Dw	Offset Value_1
		Dw	Offset Value_2
One_Format	Db	"%02tX",0
Two_Format	Db	"%02tX %02tX",0
Three_Format	Db	"%02tX %02tX %02tX",0
Value_Arg	Equ	This Word	; Print value argument list (Byte/Word)
		Dw	Offset Value
Byte_Value	Db	"%02tX",0
Word_Value	Db	"%04X",0
PC_Format	Db	"  PC=",0
SP_Format	Db	" -------------------- SP=",0
Mnemonic_Arg	Equ	This Word	; Print mnemonic argument list
Pointer 	Dw	?		; Pointer to mnemonic string
Mnemonic_Format Db	"%s",0
Break_Arg	Equ	This Word	; Print breakpoint argument list
		Dw	Offset Value
Break_Format	Db	"BP%1tu",0
BAD_Arg 	Equ	This Word	; BAD addressing mode argument list
		Dw	Offset Value_0
BAD_Format	Db	"$%02tX",0
IMM_Arg 	Equ	This Word	; IMM addressing mode argument list
		Dw	Offset Value_1
IMM_Format	Db	"#$%02tX",0
ABS_Arg 	Equ	This Word	; ABS addressing mode argument list
		Dw	Offset Value_1
ABS_Format	Db	"$%04X",0
DP_Arg		Equ	This Word	; DP addressing mode argument list
		Dw	Offset Value_1
DP_Format	Db	"$%02tX",0
DIX_Arg Equ	This Word		; DIX addressing mode argument list
		Dw	Offset Value_1
DIX_Format	Db	"$%02tX,X",0
DIY_Arg Equ	This Word		; DIY addressing mode argument list
		Dw	Offset Value_1
DIY_Format	Db	"$%02tX,Y",0
AIX_Arg Equ	This Word		; AIX addressing mode argument list
		Dw	Offset Value_1
AIX_Format	Db	"$%04X,X",0
AIY_Arg Equ	This Word		; AIY addressing mode argument list
		Dw	Offset Value_1
AIY_Format	Db	"$%04X,Y",0
AI_Arg Equ	This Word		; AI addressing mode argument list
		Dw	Offset Value_1
AI_Format	Db	"($%04X)",0
DI_Arg Equ	This Word		; DI addressing mode argument list
		Dw	Offset Value_1
DI_Format	Db	"($%02tX)",0
AIIX_Arg Equ	This Word		; AIIX addressing mode argument list
		Dw	Offset Value_1
AIIX_Format	Db	"($%04X,X)",0
DIIX_Arg Equ	This Word		; DIIX addressing mode argument list
		Dw	Offset Value_1
DIIX_Format	Db	"($%02tX,X)",0
DIIY_Arg Equ	This Word		; DIIY addressing mode argument list
		Dw	Offset Value_1
DIIY_Format	Db	"($%02tX),Y",0
PCR_Arg Equ	This Word		; PCR addressing mode argument list
		Dw	Offset Value
PCR_Format	Db	"$%04X",0
ACC_Format	Db	"  ACC=",0
X_Format	Db	"  X=",0
Y_Format	Db	"  Y=",0
Flags_Format	Db	" -- Flags ",0
Char_Arg	Equ	This Word	; Character argument list
Char_Address	Dw	?
Char_Format	Db	"%c",0
Debug_Prompt	Db	"\n:",0         ; Debug command prompt string
;******************************************************************************
;
;	Define all the debug command tables
;
;******************************************************************************
Debug_Table	Equ	This Byte	; Debug command match table
		Db	3		; Debug command table entry size
		Db	1,"X "          ; Debug execute command            [ X]
		Db	1,"GO"          ; Debug go command                 [ G]
		Db	1,"? "          ; Debug print command (Evaluate)   [ ?]
		Db	2,"EV"          ; Debug evaluate command           [EV]
		Db	1,"DU"          ; Debug dump data command          [ D]
		Db	2,"DB"          ; Debug dump byte data command     [DB]
		Db	2,"DW"          ; Debug dump word data command     [DW]
		Db	1,"UN"          ; Debug unassemble command         [ U]
		Db	1,"RE"          ; Debug register command           [ R]
		Db	2,"RA"          ; Debug radix command              [RA]
		Db	1,"HE"          ; Debug help command               [ H]
		Db	1,"TR"          ; Debug trace command              [ T]
		Db	1,"EN"          ; Debug enter data command         [ E]
		Db	2,"EB"          ; Debug enter byte data command    [EB]
		Db	2,"EW"          ; Debug enter word data command    [EW]
		Db	1,"FI"          ; Debug fill command               [ F]
		Db	1,"MO"          ; Debug move command               [ M]
		Db	1,"CO"          ; Debug compare command            [ C]
		Db	1,"SE"          ; Debug search command             [ S]
		Db	2,"BP"          ; Debug set breakpoint command     [BP]
		Db	2,"BD"          ; Debug disable breakpoint command [BD]
		Db	2,"BE"          ; Debug enable breakpoint command  [BE]
		Db	2,"BC"          ; Debug clear breakpoint command   [BC]
		Db	2,"BL"          ; Debug list breakpoint command    [BL]
		Db	1,"PR"          ; Debug process command            [ P]
		Db	2,"FL"          ; Debug flip command               [FL]
		Db	0		; End of debug match table
Command_Table	Equ	This Word	; Debug command jump table
		Dw	Execute_Command ; Debug execute command routine    [ X]
		Dw	Go_Command	; Debug go command routine	   [ G]
		Dw	Eval_Command	; Debug print command routine	   [ ?]
		Dw	Eval_Command	; Debug evaluate command routine   [EV]
		Dw	Dump_Command	; Debug dump command routine	   [ D]
		Dw	Dbyte_Command	; Debug dump byte command routine  [DB]
		Dw	Dword_Command	; Debug dump word command routine  [DW]
		Dw	Code_Command	; Debug unassemble command routine [ U]
		Dw	Reg_Command	; Debug register command routine   [ R]
		Dw	Radix_Command	; Debug radix command routine	   [RA]
		Dw	Help_Command	; Debug help command routine	   [ H]
		Dw	Trace_Command	; Debug trace command routine	   [ T]
		Dw	Enter_Command	; Debug enter command routine	   [ E]
		Dw	Ebyte_Command	; Debug enter byte command routine [EB]
		Dw	Eword_Command	; Debug enter word command routine [EW]
		Dw	Fill_Command	; Debug fill command routine	   [ F]
		Dw	Move_Command	; Debug move command routine	   [ M]
		Dw	Compare_Command ; Debug compare command routine    [ C]
		Dw	Search_Command	; Debug search command routine	   [ S]
		Dw	Bpoint_Command	; Debug set breakpoint routine	   [BP]
		Dw	Bdsable_Command ; Debug disable breakpoint routine [BD]
		Dw	Benable_Command ; Debug enable breakpoint routine  [BE]
		Dw	Bclear_Command	; Debug clear breakpoint routine   [BC]
		Dw	Blist_Command	; Debug list breakpoint routine    [BL]
		Dw	Proceed_Command ; Debug process command routine    [ P]
		Dw	Flip_Command	; Debug flip command routine	   [FL]
Break_Table	Equ	This Byte	; Start of the debug breakpoint table
		Rept	BREAKPOINTS
		Debug_Break	<>	; Define the breakpoint table entries
		Endm
Local_Break	Debug_Break	<>	; Define the local breakpoint structure
Code_Table	Equ	This Word	; Start of the debug code address table
		Rept	BREAKPOINTS
		Dw	?		; Define the code address table entries
		Endm
Data_Table	Equ	This Word	; Start of the debug data address table
		Rept	BREAKPOINTS
		Dw	?		; Define the data address table entries
		Endm
Type_Table	Equ	This Word	; Start of the debug type table
		Rept	BREAKPOINTS
		Dw	?		; Define the debug type table entries
		Endm
Color_Table	Debug_Color	<>	; Debugger color table
Match_Table	Equ	This Byte	; Start of debug parameter match table
		Db	7		; Match table entry size
		Db	4,"BACKGR"      ; Debugger background color parameter
		Db	5,"TITLE "      ; Debugger title color parameter
		Db	3,"COMMAN"      ; Debugger command color parameter
		Db	4,"ADDRES"      ; Debugger addresses color parameter
		Db	5,"VALUES"      ; Debugger values color parameter
		Db	3,"MNEMON"      ; Debugger mnemonic color parameter
		Db	4,"FLAGS "      ; Debugger flags color parameter
		Db	4,"DATA  "      ; Debugger data color parameter
		Db	2,"STACK "      ; Debugger stack color parameter
		Db	3,"REGIST"      ; Debugger register color parameter
		DB	6,"SCHEME"      ; Debugger color scheme parameter
		Db	0		; End of the debug match table
Jump_Table	Equ	This Word	; Start of parameter jump table
		Dw	Offset Back_Option
		Dw	Offset Title_Option
		Dw	Offset Command_Option
		Dw	Offset Address_Option
		Dw	Offset Value_Option
		Dw	Offset Mnemonic_Option
		Dw	Offset Flags_Option
		Dw	Offset Data_Option
		Dw	Offset Stack_Option
		Dw	Offset Register_Option
		Dw	Offset Scheme_Option
Parm_Table	Equ	This Byte	; Configuration parameter match table
		Db	7		; Match table entry size
		Db	2,"BLACK "      ; Black color value string
		Db	1,"BLUE  "      ; Blue color value string
		Db	1,"GREEN "      ; Green color value string
		Db	1,"CYAN  "      ; Cyan color value string
		Db	2,"RED   "      ; Red color value string
		Db	2,"MAGENT"      ; Magenta color value string
		Db	2,"BROWN "      ; Brown color value string
		Db	2,"WHITE "      ; White color value string
		Db	1,"GRAY  "      ; Gray color value string
		Db	1,"      "      ; Dummy color value string (Int. blue)
		Db	1,"      "      ; Dummy color value string (Int. green)
		Db	1,"      "      ; Dummy color value string (Int. cyan)
		Db	1,"PINK  "      ; Pink color value string
		Db	2,"PURPLE"      ; Purple color value string
		Db	2,"YELLOW"      ; Yellow color value string
		Db	1,"      "      ; Dummy color value string (Int. white)
		Db	3,"INTENS"      ; Intense parameter value string
		Db	0		; End of the parameter match table
;******************************************************************************
;
;	Define all the debug windows
;
;******************************************************************************
Title_Window	Window	<TITLE_UL_ROW,TITLE_UL_COL,TITLE_LR_ROW,TITLE_LR_COL>
Data_Window	Window	<DATA_UL_ROW,DATA_UL_COL,DATA_LR_ROW,DATA_LR_COL>
Status_Window	Window	<STAT_UL_ROW,STAT_UL_COL,STAT_LR_ROW,STAT_LR_COL>
Code_Window	Window	<CODE_UL_ROW,CODE_UL_COL,CODE_LR_ROW,CODE_LR_COL>
Register_Window Window	<REG_UL_ROW,REG_UL_COL,REG_LR_ROW,REG_LR_COL>
Stack_Window	Window	<STACK_UL_ROW,STACK_UL_COL,STACK_LR_ROW,STACK_LR_COL>
Command_Window	Window	<COMM_UL_ROW,COMM_UL_COL,COMM_LR_ROW,COMM_LR_COL>
;******************************************************************************
;
;	Define all other debug data areas
;
;******************************************************************************
Value_Table	Equ	This Byte	; Start of the values table
Value_0 	Db	?		; Byte value 0
Value_1 	Db	?		; Byte value 1
Value_2 	Db	?		; Byte value 2
Value_3 	Db	?		; Byte value 3
Value_4 	Db	?		; Byte value 4
Value_5 	Db	?		; Byte value 5
Value_6 	Db	?		; Byte value 6
Value_7 	Db	?		; Byte value 7
Current_Address Dw	?		; Current address value
Compare_Address Dw	?		; Compare address value
Break_Address	Dw	?		; Breakpoint address value
ASCII_String	Db	"12345678",0    ; ASCII string storage area
Flag_String	Db	"NORBMIZC"      ; 65C02 flag string storage area
Data_Address	Dw	0000h		; Current data dump address
Dump_Format	Dw	FORMAT_BYTE	; Current data dump format (Byte)
Enter_Format	Dw	FORMAT_BYTE	; Current data enter format (Byte)
Code_Address	Dw	0000h		; Current code dump address
Last_Data	Dw	0FFFFh		; Last data dump address
Last_Code	Dw	0FFFFh		; Last code dump address
Value		Dw	?		; Current value storage
Compare 	Dw	?		; Comparsion value storage
Highlight	Dw	?		; Color highlight value
Code_Count	Dw	0		; Code breakpoints count
Data_Count	Dw	0		; Data breakpoints count
Trace_Count	Dw	0		; Trace count value storage
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Debug module
