; This program emulates an APPLE ][ or APPLE ][ PLUS computer on an IBM PC
; or compatible.
;
; References:
; Apple ][ Reference Manual by Christopher Espinosa
; -------------------------
;
; Beneath Apple DOS by Don Worth and Pieter Lechner
; -----------------
;
; Several additional files are needed:
; APPLBIOS.COM .................. BIOS (ROM) for APPLE
; SLOTn.CRD    .................. CONTROLLER CARD for APPLE slot number
; ????????.DSK .................. DISKETTE IMAGES
;
; FUNCTION KEYS:
; F1 ........ RESET
; F2 ........ quit (return to DOS)
; F3 ........ <-
; F4 ........ ->
; F5 ........ change diskette in D1
; F6 ........ change diskette in D2
; F7 ........ change file for slot io
;
;        registers
;     apple       IBM
;     -----       ---
;       A          AL
;       X          DL
;       Y          DH
;       PC         SI
;       S          DI
;          flags
;    NVBBDIZC      P  (apple)
;    NZ.....C      AH (IBM)
;    .VBBDI..      CH (IBM)
;
;CONSTANTS
ALL			EQU	11111111B
NEG			EQU	10000000B
NOT_NEG			EQU	01111111B
ZER			EQU	01000000B
NOT_ZER			EQU	10111111B
CRY			EQU	00000001B
NOT_CRY			EQU	11111110B
OVF			EQU	01000000B
NOT_OVF			EQU	10111111B
DCM			EQU	00001000B
NOT_DCM			EQU	11110111B
INT			EQU	00000100B
NOT_INT			EQU	11111011B
;
APPLE_STACK_LENGTH	EQU	100H
;
BIOS_START		EQU	0D000H
BIOS_LENGTH		EQU	03000H
;IO constants
APPLE_IO_AREA		EQU	0C000H
IO_PAGE			EQU	0C0H
IO_CARD_LENGTH		EQU	0100H
;local IO constants
KEYBOARD_STROBE		EQU	0C010H
SPEAKER			EQU	0C030H
SCN_CHG_LOW		EQU	0C050H
SCN_CHG_HIGH		EQU	0C057H
PUSH_BUTTON_LOW		EQU	0C061H	
PUSH_BUTTON_HI		EQU	0C063H
PADDLE_LOW		EQU	0C064H
PADDLE_HIGH		EQU	0C067H
PADDLE_STROBE		EQU	0C070H
;disk constants
DISK_SLOT		EQU	6
DISK_CARD_ADDRESS	EQU	0C000H+100H*DISK_SLOT
STEPPER_MOVE_LOW 	EQU	0C080H+10H*DISK_SLOT
STEPPER_MOVE_HI		EQU	0C087H+10H*DISK_SLOT
ENABLE_1		EQU	0C08AH+10H*DISK_SLOT
ENABLE_2		EQU	0C08BH+10H*DISK_SLOT
STROBE_LATCH		EQU	0C08CH+10H*DISK_SLOT
LOAD_LATCH		EQU	0C08DH+10H*DISK_SLOT
SET_READ_MODE		EQU	0C08EH+10H*DISK_SLOT
SENSE_WRITE_PROTECT 	EQU	0C08EH+10H*DISK_SLOT
SET_WRITE_MODE		EQU	0C08FH+10H*DISK_SLOT
;disk usage constants
TRACK_LENGTH		EQU	01A00H
READ_MODE		EQU	0
WRITE_MODE		EQU	1
UNINIT_STATUS		EQU	0
READ_STATUS		EQU	1
WRITTEN_STATUS		EQU	2
WRITE_PROTECTED		EQU	0FFH
NOT_WRITE_PROTECTED	EQU	00H
D1			EQU	1
D2			EQU	2
;
;MACROS
IO_CHK	MACRO	DIRECTION
	LOCAL	NO_IO
	CMP	BH,IO_PAGE
	IFIDN	<DIRECTION>,<READ_MEM>
	JNE	NO_IO
	ELSE
	JB	NO_IO
	ENDIF
	CALL	IO_&DIRECTION
NO_IO:
	ENDM
;
CHK_SCN	MACRO	MEMORY_TYPE
	LOCAL	NO_SCN
	IFDIF	<MEMORY_TYPE>,<NO_MEMORY>
	IFDIF	<MEMORY_TYPE>,<ACCUMULATOR>
	IFDIF	<MEMORY_TYPE>,<ZERO_PAGE>
	IFDIF	<MEMORY_TYPE>,<ZERO_PAGE_X>
	IFDIF	<MEMORY_TYPE>,<ZERO_PAGE_Y>
	CMP	BH,CS:SCN_HI
	JA	NO_SCN
	CMP	BH,CS:SCN_LOW
	JB	NO_SCN
	CALL	WORD PTR CS:[SCN_PRC]
NO_SCN:
	ENDIF
	ENDIF
	ENDIF
	ENDIF
	ENDIF
	ENDM
;
INST_SZ	MACRO	N
	IFIDN	<N>,<1>
	INC	SI
	ELSE
	ADD	SI,N
	ENDIF
	ENDM
;
IMMEDIATE	MACRO
	INC	SI
	MOV	BX,SI
	INC	SI
	ENDM
;
ZERO_PAGE	MACRO	DIRECTION
	MOV	BL,[SI] + 1
	XOR	BH,BH
	IO_CHK	DIRECTION
	INST_SZ 2
	ENDM
;
ZERO_PAGE_X	MACRO	DIRECTION
	MOV	BL,[SI] + 1
	XOR	BH,BH
	ADD	BL,DL
	IO_CHK	DIRECTION
	INST_SZ 2
	ENDM
;
ZERO_PAGE_Y	MACRO	DIRECTION
	MOV	BL,[SI] + 1
	XOR	BH,BH
	ADD	BL,DH
	IO_CHK	DIRECTION
	INST_SZ 2
	ENDM
;
ABSOLUTE	MACRO	DIRECTION
	MOV	BX,[SI] + 1
	IO_CHK	DIRECTION
	INST_SZ 3
	ENDM
;
ABSOLUTE_X	MACRO	DIRECTION
	MOV	BX,[SI] + 1
	ADD	BL,DL
	ADC	BH,0
	IO_CHK	DIRECTION
	INST_SZ 3
	ENDM
;
ABSOLUTE_Y	MACRO	DIRECTION
	MOV	BX,[SI] + 1
	ADD	BL,DH
	ADC	BH,0
	IO_CHK	DIRECTION
	INST_SZ 3
	ENDM
;
INDIRECT_X	MACRO	DIRECTION
	MOV	BL,[SI] + 1
	XOR	BH,BH
	ADD	BL,DL
	ADC	BH,0
	MOV	BX,[BX]
	IO_CHK	DIRECTION
	INST_SZ 2
	ENDM
;
INDIRECT_Y	MACRO	DIRECTION
	MOV	BL,[SI] + 1
	XOR	BH,BH
	MOV	BX,[BX]
	ADD	BL,DH
	ADC	BH,0
	IO_CHK	DIRECTION
	INST_SZ 2
	ENDM
;
NO_MEMORY	MACRO
	INST_SZ 1
	ENDM
;
ACCUMULATOR	MACRO
	INST_SZ 1
	ENDM
;
NEXT_INSTRUCTION	MACRO
;process next instruction
	MOV	BL,[SI]
	XOR	BH,BH
	SHL	BX,1
	JMP	WORD PTR CS:JUMP_TAB[BX]
	ENDM
;
PUSH_B	MACRO	REG
	LOCAL	PUSH_B_NOT_0
	MOV	APPLE_STACK[DI],REG
	DEC	DI
	JNS	PUSH_B_NOT_0
	MOV	DI,APPLE_STACK_LENGTH - 1
PUSH_B_NOT_0:
	ENDM
;
PUSH_PC	MACRO
	LOCAL	PUSH_PC_NOT_0,PUSH_PC_EXIT
	DEC	DI
	JNS	PUSH_PC_NOT_0
	MOV	BX,SI
	MOV	APPLE_STACK,BH
	MOV	APPLE_STACK + APPLE_STACK_LENGTH - 1,BL
	MOV	DI,APPLE_STACK_LENGTH - 2
	JMP	SHORT PUSH_PC_EXIT
PUSH_PC_NOT_0:
	MOV	WORD PTR APPLE_STACK[DI],SI
	DEC	DI
	JNS	PUSH_PC_EXIT
	MOV	DI,APPLE_STACK_LENGTH - 1
PUSH_PC_EXIT:
	ENDM
;
PULL_B	MACRO	REG
	LOCAL	PULL_B_NOT_END
	INC	DI
	CMP	DI,APPLE_STACK_LENGTH
	JB	PULL_B_NOT_END
	MOV	DI,0
PULL_B_NOT_END:
	MOV	REG,APPLE_STACK[DI]
	ENDM
;
PULL_PC	MACRO
	LOCAL	PULL_PC_AT_END,PULL_PC_FINISH,PULL_PC_EXIT
	INC	DI
	CMP	DI,APPLE_STACK_LENGTH - 1
	JB	PULL_PC_FINISH
	JA	PULL_PC_AT_END
	MOV	BL,APPLE_STACK + APPLE_STACK_LENGTH - 1
	MOV	BH,APPLE_STACK
	MOV	SI,BX
	MOV	DI,0
	JMP	SHORT PULL_PC_EXIT
PULL_PC_AT_END:
	MOV	DI,0
PULL_PC_FINISH:
	MOV	SI,WORD PTR APPLE_STACK[DI]
	INC	DI
PULL_PC_EXIT:
	ENDM
;
.NOP	MACRO
	NO_MEMORY
	NEXT_INSTRUCTION
	ENDM
;
.ORA	MACRO	MEMORY_TYPE
	LOCAL	ORA_CARRY
	MEMORY_TYPE	READ_MEM
	SAHF
	JC	ORA_CARRY
	OR	AL,[BX]
	LAHF
	NEXT_INSTRUCTION
ORA_CARRY:
	OR	AL,[BX]
	STC
	LAHF
	NEXT_INSTRUCTION
	ENDM
;
.ASL	MACRO	MEMORY_TYPE
	MEMORY_TYPE	WRITE_MEM
	IFIDN	<MEMORY_TYPE>,<ACCUMULATOR>
	SAL	AL,1
	ELSE
	SAL	BYTE PTR [BX],1
	ENDIF
	LAHF
	IFDIF	<MEMORY_TYPE>,<ACCUMULATOR>
	CHK_SCN	MEMORY_TYPE
	ENDIF
	NEXT_INSTRUCTION
	ENDM
;
.AND	MACRO	MEMORY_TYPE
	LOCAL	AND_CARRY
	MEMORY_TYPE	READ_MEM
	SAHF
	JC	AND_CARRY
	AND	AL,[BX]
	LAHF
	NEXT_INSTRUCTION
AND_CARRY:
	AND	AL,[BX]
	STC
	LAHF
	NEXT_INSTRUCTION
	ENDM
;
.EOR	MACRO	MEMORY_TYPE
	LOCAL	EOR_CARRY
	MEMORY_TYPE	READ_MEM
	SAHF
	JC	EOR_CARRY
	XOR	AL,[BX]
	LAHF
	NEXT_INSTRUCTION
EOR_CARRY:
	XOR	AL,[BX]
	STC
	LAHF
	NEXT_INSTRUCTION
	ENDM
;
.ROL	MACRO	MEMORY_TYPE
	LOCAL	ROL_CARRY
	MEMORY_TYPE	WRITE_MEM
	SAHF
	JC	ROL_CARRY
	IFIDN	<MEMORY_TYPE>,<ACCUMULATOR>
	SAL	AL,1
	ELSE
	SAL	BYTE PTR [BX],1
	ENDIF
	LAHF
	IFDIF	<MEMORY_TYPE>,<ACCUMULATOR>
	CHK_SCN	MEMORY_TYPE
	ENDIF
	NEXT_INSTRUCTION
ROL_CARRY:
	IFIDN	<MEMORY_TYPE>,<ACCUMULATOR>
	SAL	AL,1
	ELSE
	SAL	BYTE PTR [BX],1
	ENDIF
	LAHF
	IFIDN	<MEMORY_TYPE>,<ACCUMULATOR>
	OR	AL,00000001B
	ELSE
	OR	BYTE PTR [BX],00000001B
	ENDIF
	AND	AH,NOT_ZER
	IFDIF	<MEMORY_TYPE>,<ACCUMULATOR>
	CHK_SCN	MEMORY_TYPE
	ENDIF
	NEXT_INSTRUCTION
	ENDM
;
.ROR	MACRO	MEMORY_TYPE
	LOCAL	ROR_CARRY
	MEMORY_TYPE	WRITE_MEM
	SAHF
	JC	ROR_CARRY
	IFIDN	<MEMORY_TYPE>,<ACCUMULATOR>
	SHR	AL,1
	ELSE
	SHR	BYTE PTR [BX],1
	ENDIF
	LAHF
	IFDIF	<MEMORY_TYPE>,<ACCUMULATOR>
	CHK_SCN	MEMORY_TYPE
	ENDIF
	NEXT_INSTRUCTION
ROR_CARRY:
	IFIDN	<MEMORY_TYPE>,<ACCUMULATOR>
	SHR	AL,1
	ELSE
	SHR	BYTE PTR [BX],1
	ENDIF
	LAHF
	IFIDN	<MEMORY_TYPE>,<ACCUMULATOR>
	OR	AL,10000000B
	ELSE
	OR	BYTE PTR [BX],10000000B
	ENDIF
	AND	AH,NOT_ZER
	OR	AH,NEG
	IFDIF	<MEMORY_TYPE>,<ACCUMULATOR>
	CHK_SCN	MEMORY_TYPE
	ENDIF
	NEXT_INSTRUCTION
	ENDM
;
.LSR	MACRO	MEMORY_TYPE
	MEMORY_TYPE	WRITE_MEM
	IFIDN	<MEMORY_TYPE>,<ACCUMULATOR>
	SHR	AL,1
	ELSE
	SHR	BYTE PTR [BX],1
	ENDIF
	LAHF
	IFDIF	<MEMORY_TYPE>,<ACCUMULATOR>
	CHK_SCN	MEMORY_TYPE
	ENDIF
	NEXT_INSTRUCTION
	ENDM
;
.ADC	MACRO	MEMORY_TYPE
	LOCAL	ADC_OVERFLOW, ADC_DECIMAL
	MEMORY_TYPE	READ_MEM
	TEST	CH,DCM
	JNZ	ADC_DECIMAL
	SAHF
	ADC	AL,[BX]
	JO	ADC_OVERFLOW
	LAHF
	AND	CH,NOT_OVF
	NEXT_INSTRUCTION
ADC_OVERFLOW:
	LAHF
	OR	CH,OVF
	NEXT_INSTRUCTION
ADC_DECIMAL:
	SAHF
	ADC	AL,[BX]
	DAA
	LAHF
	NEXT_INSTRUCTION
	ENDM
;
.SBC	MACRO	MEMORY_TYPE
	LOCAL	SBC_OVERFLOW, SBC_DECIMAL
	MEMORY_TYPE	READ_MEM
	TEST	CH,DCM
	JNZ	SBC_DECIMAL
	SAHF
	CMC
	SBB	AL,[BX]
	CMC
	JO	SBC_OVERFLOW
	LAHF
	AND	CH,NOT_OVF
	NEXT_INSTRUCTION
SBC_OVERFLOW:
	LAHF
	OR	CH,OVF
	NEXT_INSTRUCTION
SBC_DECIMAL:
	SAHF
	CMC
	SBB	AL,[BX]
	CMC
	DAS
	LAHF
	NEXT_INSTRUCTION
	ENDM
;
.LDA	MACRO	MEMORY_TYPE
	LOAD	AL,MEMORY_TYPE
	ENDM
;
.LDX	MACRO	MEMORY_TYPE
	LOAD	DL,MEMORY_TYPE
	ENDM
;
.LDY	MACRO	MEMORY_TYPE
	LOAD	DH,MEMORY_TYPE
	ENDM
;
LOAD	MACRO	REGISTER,MEMORY_TYPE
	LOCAL	LOAD_CARRY
	MEMORY_TYPE	READ_MEM
	MOV	REGISTER,[BX]
	SAHF
	JC	LOAD_CARRY
	TEST	REGISTER,0FFH
	LAHF
	NEXT_INSTRUCTION
LOAD_CARRY:
	TEST	REGISTER,0FFH
	STC
	LAHF
	NEXT_INSTRUCTION
	ENDM
;
.STA	MACRO	MEMORY_TYPE
	STORE	AL,MEMORY_TYPE
	ENDM
;
.STX	MACRO	MEMORY_TYPE
	STORE	DL,MEMORY_TYPE
	ENDM
;
.STY	MACRO	MEMORY_TYPE
	STORE	DH,MEMORY_TYPE
	ENDM
;
STORE	MACRO	REGISTER,MEMORY_TYPE
	MEMORY_TYPE	WRITE_MEM
	MOV	[BX],REGISTER
	CHK_SCN	MEMORY_TYPE
	NEXT_INSTRUCTION
	ENDM
;
.CMP	MACRO	MEMORY_TYPE
	COMPARE	AL,MEMORY_TYPE
	ENDM
;
.CPX	MACRO	MEMORY_TYPE
	COMPARE	DL,MEMORY_TYPE
	ENDM
;
.CPY	MACRO	MEMORY_TYPE
	COMPARE	DH,MEMORY_TYPE
	ENDM
;
COMPARE	MACRO	REGISTER,MEMORY_TYPE
	MEMORY_TYPE	READ_MEM
	CMP	REGISTER,[BX]
	LAHF
	XOR	AH,CRY
	NEXT_INSTRUCTION
	ENDM
;
.DEC	MACRO	MEMORY_TYPE
	DECRMNT	[BX],MEMORY_TYPE
	ENDM
;
.DEX	MACRO
	DECRMNT	DL,NO_MEMORY
	ENDM
;
.DEY	MACRO
	DECRMNT	DH,NO_MEMORY
	ENDM
;
DECRMNT	MACRO	OPERAND,MEMORY_TYPE
	MEMORY_TYPE	WRITE_MEM
	SAHF
	IFIDN <OPERAND>,<[BX]>
	DEC	BYTE PTR OPERAND
	ELSE
	DEC	OPERAND
	ENDIF
	LAHF
	IFIDN <OPERAND>,<[BX]>
	CHK_SCN	MEMORY_TYPE
	ENDIF
	NEXT_INSTRUCTION
	ENDM
;
.INC	MACRO	MEMORY_TYPE
	INCRMNT	[BX],MEMORY_TYPE
	ENDM
;
.INX	MACRO
	INCRMNT	DL,NO_MEMORY
	ENDM
;
.INY	MACRO
	INCRMNT	DH,NO_MEMORY
	ENDM
;
INCRMNT	MACRO	OPERAND,MEMORY_TYPE
	MEMORY_TYPE	WRITE_MEM
	SAHF
	IFIDN <OPERAND>,<[BX]>
	INC	BYTE PTR OPERAND
	ELSE
	INC	OPERAND
	ENDIF
	LAHF
	IFIDN <OPERAND>,<[BX]>
	CHK_SCN	MEMORY_TYPE
	ENDIF
	NEXT_INSTRUCTION
	ENDM
;
.BIT	MACRO	MEMORY_TYPE
	LOCAL	BIT_ZERO
	MEMORY_TYPE	READ_MEM
	MOV	BL,[BX]
	OR	AH,10000000B
	OR	CH,01000000B
	TEST	AL,BL
	JZ	BIT_ZERO
	OR	BL,00111111B
	AND	AH,BL
	AND	AH,NOT_ZER
	AND	CH,BL
	NEXT_INSTRUCTION
BIT_ZERO:
	OR	BL,00111111B
	AND	AH,BL
	OR	AH,ZER
	AND	CH,BL
	NEXT_INSTRUCTION
	ENDM
;
.BRK	MACRO
	INC	SI
	INC	SI
	PUSH_PC
	MOV	BL,10000001B
	AND	BL,AH
	TEST	AH,ZER
	JZ	BRK_ZERO
	OR	BL,00000010B
BRK_ZERO:
	OR	CH,00110000B
	OR	BL,CH
	PUSH_B	BL
	MOV	SI,DS:[0FFFEH]
	NEXT_INSTRUCTION
	ENDM
;
.PHP	MACRO
	NO_MEMORY
	MOV	BL,10000001B
	AND	BL,AH
	TEST	AH,ZER
	JZ	PHP_ZERO
	OR	BL,00000010B
PHP_ZERO:
	OR	CH,00110000B
	OR	BL,CH
	PUSH_B	BL
	NEXT_INSTRUCTION
	ENDM
;
.JSR	MACRO
	INC	SI
	INC	SI
	PUSH_PC
	MOV	SI,[SI] - 1
	NEXT_INSTRUCTION
	ENDM
;
.PLP	MACRO
	NO_MEMORY
	PULL_B	BL
	MOV	AH,10000001B
	AND	AH,BL
	TEST	BL,00000010B
	JZ	PLP_ZERO
	OR	AH,ZER
PLP_ZERO:
	AND	BL,01111100B
	MOV	CH,BL
	NEXT_INSTRUCTION
	ENDM
;
.RTI	MACRO
	PULL_PC
	INC	SI
	PULL_B	BL
	MOV	AH,10000000B
	AND	AH,BL
	TEST	BL,00000010B
	JZ	RTI_ZERO
	OR	AH,ZER
RTI_ZERO:
	AND	BL,01111100B
	MOV	CH,BL
	NEXT_INSTRUCTION
	ENDM
;
.PHA	MACRO
	NO_MEMORY
	PUSH_B	AL
	NEXT_INSTRUCTION
	ENDM
;
.JMP	MACRO	MEMORY_TYPE
	IFIDN	<MEMORY_TYPE>,<ABSOLUTE>
	MOV	SI,[SI]+1
	ELSE
	MOV	BX,[SI] + 1
	MOV	SI,[BX]
	ENDIF
	NEXT_INSTRUCTION
	ENDM
;
.RTS	MACRO
	PULL_PC
	INC	SI
	NEXT_INSTRUCTION
	ENDM
;
.PLA	MACRO
	NO_MEMORY
	PULL_B	AL
	SAHF
	JC	PLA_CARRY
	TEST	AL,0FFH
	LAHF
	NEXT_INSTRUCTION
PLA_CARRY:
	TEST	AL,0FFH
	STC
	LAHF
	NEXT_INSTRUCTION
	ENDM
;
.TAX	MACRO
	TNSFR	AL,DL
	ENDM
;
.TAY	MACRO
	TNSFR	AL,DH
	ENDM
;
.TSX	MACRO
	MOV	BX,DI
	TNSFR	BL,DL
	ENDM
;
.TXA	MACRO
	TNSFR	DL,AL
	ENDM
;
.TYA	MACRO
	TNSFR	DH,AL
	ENDM
;
TNSFR	MACRO	OPERAND1,OPERAND2
	LOCAL	TNSFR_CARRY
	NO_MEMORY
	MOV	OPERAND2,OPERAND1
	SAHF
	JC	TNSFR_CARRY
	TEST	OPERAND2,0FFH
	LAHF
	NEXT_INSTRUCTION
TNSFR_CARRY:
	TEST	OPERAND2,0FFH
	STC
	LAHF
	NEXT_INSTRUCTION
	ENDM
;
.TXS	MACRO
	NO_MEMORY
	MOV	BL,DL
	XOR	BH,BH
	SAHF
	JC	TXS_CARRY
	TEST	BL,0FFH
	LAHF
	MOV	DI,BX
	NEXT_INSTRUCTION
TXS_CARRY:
	TEST	BL,0FFH
	STC
	LAHF
	MOV	DI,BX
	NEXT_INSTRUCTION
	ENDM
;
SET	MACRO	FLAG,REG
	NO_MEMORY
	OR	REG,FLAG
	NEXT_INSTRUCTION
	ENDM
;
CLEAR	MACRO	FLAG,REG
	NO_MEMORY
	AND	REG,NOT_&FLAG
	NEXT_INSTRUCTION
	ENDM
;
BRANCH	MACRO	SW,FLAG,REG
	LOCAL	BRANCH_TAG
	INST_SZ 2
	TEST	REG,FLAG
	J&SW&Z	BRANCH_TAG
	MOV	BH,[SI] - 1
	SAR	BX,CL
	ADD	SI,BX
BRANCH_TAG:
	NEXT_INSTRUCTION
	ENDM
;
.BPL	MACRO
	BRANCH	N,NEG,AH
	ENDM
;
.CLC	MACRO
	CLEAR	CRY,AH
	ENDM
;
.BMI	MACRO
	BRANCH	,NEG,AH
	ENDM
;
.SEC	MACRO
	SET	CRY,AH
	ENDM
;
.BVC	MACRO
	BRANCH	N,OVF,CH
	ENDM
;
.CLI	MACRO
	CLEAR	INT,CH
	ENDM
;
.BVS	MACRO
	BRANCH	,OVF,CH
	ENDM
;
.SEI	MACRO
	SET	INT,CH
	ENDM
;
.BCC	MACRO
	BRANCH	N,CRY,AH
	ENDM
;
.BCS	MACRO
	BRANCH	,CRY,AH
	ENDM
;
.CLV	MACRO
	CLEAR	OVF,CH
	ENDM
;
.BNE	MACRO
	BRANCH	N,ZER,AH
	ENDM
;
.CLD	MACRO
	CLEAR	DCM,CH
	ENDM
;
.BEQ	MACRO
	BRANCH	,ZER,AH
	ENDM
;
.SED	MACRO
	SET	DCM,CH
	ENDM
;
DSEG	SEGMENT	AT 00H
	ORG	00100H
APPLE_STACK	LABEL	BYTE
	ORG	0C000H
KEYBOARD_DATA	LABEL	BYTE
DSEG	ENDS
;
CSEG	SEGMENT	PARA PUBLIC 'CODE'
	ASSUME	CS:CSEG, DS:DSEG
;
	ORG	0
;
START_COM:
;
	ORG	0100H
;
ENTRY:
	JMP	BEGIN
;
DS_SAV	DW	?
;
TIMER_INT_ADDR	LABEL	DWORD
TIMER_OFF_SAV	DW	?
TIMER_SEG_SAV	DW	?
;
MEM_ER_TXT	DB	'MEMORY ALLOCATION ERROR. PROGRAM TERMINATED.',0DH,0AH,'$'
;
BIOS_FIL_ER_TXT DB	'BIOS FILE ERROR. PROGRAM TERMINATED.',0DH,0AH,'$'
;
BIOS_FILE_NAME	DB	'APPLBIOS.COM',00H
;
IO_CARD_NAME	LABEL	BYTE
	DB	'SLOT'
IO_CARD_NUMBER	DB	'0'
	DB	'.CRD',00H
;
JUMP_TAB DW 100H DUP(?)
;
INST_TAB	LABEL	WORD
	DW	OFFSET L_0
	DW	OFFSET L_1
	DW	OFFSET L_2
	DW	OFFSET L_3
	DW	OFFSET L_4
	DW	OFFSET L_5
	DW	OFFSET L_6
	DW	OFFSET L_7
	DW	OFFSET L_8
	DW	OFFSET L_9
	DW	OFFSET L_A
	DW	OFFSET L_B
	DW	OFFSET L_C
	DW	OFFSET L_D
	DW	OFFSET L_E
	DW	OFFSET L_F
	DW	OFFSET L_10
	DW	OFFSET L_11
	DW	OFFSET L_12
	DW	OFFSET L_13
	DW	OFFSET L_14
	DW	OFFSET L_15
	DW	OFFSET L_16
	DW	OFFSET L_17
	DW	OFFSET L_18
	DW	OFFSET L_19
	DW	OFFSET L_1A
	DW	OFFSET L_1B
	DW	OFFSET L_1C
	DW	OFFSET L_1D
	DW	OFFSET L_1E
	DW	OFFSET L_1F
	DW	OFFSET L_20
	DW	OFFSET L_21
	DW	OFFSET L_22
	DW	OFFSET L_23
	DW	OFFSET L_24
	DW	OFFSET L_25
	DW	OFFSET L_26
	DW	OFFSET L_27
	DW	OFFSET L_28
	DW	OFFSET L_29
	DW	OFFSET L_2A
	DW	OFFSET L_2B
	DW	OFFSET L_2C
	DW	OFFSET L_2D
	DW	OFFSET L_2E
	DW	OFFSET L_2F
	DW	OFFSET L_30
	DW	OFFSET L_31
	DW	OFFSET L_32
	DW	OFFSET L_33
	DW	OFFSET L_34
	DW	OFFSET L_35
	DW	OFFSET L_36
	DW	OFFSET L_37
	DW	OFFSET L_38
	DW	OFFSET L_39
	DW	OFFSET L_3A
	DW	OFFSET L_3B
	DW	OFFSET L_3C
	DW	OFFSET L_3D
	DW	OFFSET L_3E
	DW	OFFSET L_3F
	DW	OFFSET L_40
	DW	OFFSET L_41
	DW	OFFSET L_42
	DW	OFFSET L_43
	DW	OFFSET L_44
	DW	OFFSET L_45
	DW	OFFSET L_46
	DW	OFFSET L_47
	DW	OFFSET L_48
	DW	OFFSET L_49
	DW	OFFSET L_4A
	DW	OFFSET L_4B
	DW	OFFSET L_4C
	DW	OFFSET L_4D
	DW	OFFSET L_4E
	DW	OFFSET L_4F
	DW	OFFSET L_50
	DW	OFFSET L_51
	DW	OFFSET L_52
	DW	OFFSET L_53
	DW	OFFSET L_54
	DW	OFFSET L_55
	DW	OFFSET L_56
	DW	OFFSET L_57
	DW	OFFSET L_58
	DW	OFFSET L_59
	DW	OFFSET L_5A
	DW	OFFSET L_5B
	DW	OFFSET L_5C
	DW	OFFSET L_5D
	DW	OFFSET L_5E
	DW	OFFSET L_5F
	DW	OFFSET L_60
	DW	OFFSET L_61
	DW	OFFSET L_62
	DW	OFFSET L_63
	DW	OFFSET L_64
	DW	OFFSET L_65
	DW	OFFSET L_66
	DW	OFFSET L_67
	DW	OFFSET L_68
	DW	OFFSET L_69
	DW	OFFSET L_6A
	DW	OFFSET L_6B
	DW	OFFSET L_6C
	DW	OFFSET L_6D
	DW	OFFSET L_6E
	DW	OFFSET L_6F
	DW	OFFSET L_70
	DW	OFFSET L_71
	DW	OFFSET L_72
	DW	OFFSET L_73
	DW	OFFSET L_74
	DW	OFFSET L_75
	DW	OFFSET L_76
	DW	OFFSET L_77
	DW	OFFSET L_78
	DW	OFFSET L_79
	DW	OFFSET L_7A
	DW	OFFSET L_7B
	DW	OFFSET L_7C
	DW	OFFSET L_7D
	DW	OFFSET L_7E
	DW	OFFSET L_7F
	DW	OFFSET L_80
	DW	OFFSET L_81
	DW	OFFSET L_82
	DW	OFFSET L_83
	DW	OFFSET L_84
	DW	OFFSET L_85
	DW	OFFSET L_86
	DW	OFFSET L_87
	DW	OFFSET L_88
	DW	OFFSET L_89
	DW	OFFSET L_8A
	DW	OFFSET L_8B
	DW	OFFSET L_8C
	DW	OFFSET L_8D
	DW	OFFSET L_8E
	DW	OFFSET L_8F
	DW	OFFSET L_90
	DW	OFFSET L_91
	DW	OFFSET L_92
	DW	OFFSET L_93
	DW	OFFSET L_94
	DW	OFFSET L_95
	DW	OFFSET L_96
	DW	OFFSET L_97
	DW	OFFSET L_98
	DW	OFFSET L_99
	DW	OFFSET L_9A
	DW	OFFSET L_9B
	DW	OFFSET L_9C
	DW	OFFSET L_9D
	DW	OFFSET L_9E
	DW	OFFSET L_9F
	DW	OFFSET L_A0
	DW	OFFSET L_A1
	DW	OFFSET L_A2
	DW	OFFSET L_A3
	DW	OFFSET L_A4
	DW	OFFSET L_A5
	DW	OFFSET L_A6
	DW	OFFSET L_A7
	DW	OFFSET L_A8
	DW	OFFSET L_A9
	DW	OFFSET L_AA
	DW	OFFSET L_AB
	DW	OFFSET L_AC
	DW	OFFSET L_AD
	DW	OFFSET L_AE
	DW	OFFSET L_AF
	DW	OFFSET L_B0
	DW	OFFSET L_B1
	DW	OFFSET L_B2
	DW	OFFSET L_B3
	DW	OFFSET L_B4
	DW	OFFSET L_B5
	DW	OFFSET L_B6
	DW	OFFSET L_B7
	DW	OFFSET L_B8
	DW	OFFSET L_B9
	DW	OFFSET L_BA
	DW	OFFSET L_BB
	DW	OFFSET L_BC
	DW	OFFSET L_BD
	DW	OFFSET L_BE
	DW	OFFSET L_BF
	DW	OFFSET L_C0
	DW	OFFSET L_C1
	DW	OFFSET L_C2
	DW	OFFSET L_C3
	DW	OFFSET L_C4
	DW	OFFSET L_C5
	DW	OFFSET L_C6
	DW	OFFSET L_C7
	DW	OFFSET L_C8
	DW	OFFSET L_C9
	DW	OFFSET L_CA
	DW	OFFSET L_CB
	DW	OFFSET L_CC
	DW	OFFSET L_CD
	DW	OFFSET L_CE
	DW	OFFSET L_CF
	DW	OFFSET L_D0
	DW	OFFSET L_D1
	DW	OFFSET L_D2
	DW	OFFSET L_D3
	DW	OFFSET L_D4
	DW	OFFSET L_D5
	DW	OFFSET L_D6
	DW	OFFSET L_D7
	DW	OFFSET L_D8
	DW	OFFSET L_D9
	DW	OFFSET L_DA
	DW	OFFSET L_DB
	DW	OFFSET L_DC
	DW	OFFSET L_DD
	DW	OFFSET L_DE
	DW	OFFSET L_DF
	DW	OFFSET L_E0
	DW	OFFSET L_E1
	DW	OFFSET L_E2
	DW	OFFSET L_E3
	DW	OFFSET L_E4
	DW	OFFSET L_E5
	DW	OFFSET L_E6
	DW	OFFSET L_E7
	DW	OFFSET L_E8
	DW	OFFSET L_E9
	DW	OFFSET L_EA
	DW	OFFSET L_EB
	DW	OFFSET L_EC
	DW	OFFSET L_ED
	DW	OFFSET L_EE
	DW	OFFSET L_EF
	DW	OFFSET L_F0
	DW	OFFSET L_F1
	DW	OFFSET L_F2
	DW	OFFSET L_F3
	DW	OFFSET L_F4
	DW	OFFSET L_F5
	DW	OFFSET L_F6
	DW	OFFSET L_F7
	DW	OFFSET L_F8
	DW	OFFSET L_F9
	DW	OFFSET L_FA
	DW	OFFSET L_FB
	DW	OFFSET L_FC
	DW	OFFSET L_FD
	DW	OFFSET L_FE
	DW	OFFSET L_FF
;
FUNC_TAB LABEL WORD
	DW	OFFSET RESET
	DW	OFFSET TERMINATE
	DW	OFFSET LEFT_ARROW
	DW	OFFSET RIGHT_ARROW
	DW	OFFSET D1_CHANGE
	DW	OFFSET D2_CHANGE
	DW	OFFSET SLOT_IO_CHANGE
	DW	OFFSET DEBUG
	DW	2 DUP(OFFSET NOTHING_FUNCTION)
;
COM_LEN = (OFFSET END_COM - OFFSET START_COM)
;
BEGIN:
;setup stack pointer
	MOV	SP,OFFSET END_COM - 2
;free unused memory
	MOV	BX,((COM_LEN + 15) / 16)
	MOV	AH,4AH
	INT	21H
	JNC	NO_RST_MEM_ERR
;error resetting memory size
	MOV	DX,OFFSET MEM_ER_TXT
	MOV	AH,9
	INT	21H
	MOV	AX,4C08H
	INT	21H
NO_RST_MEM_ERR:
;allocate 64K for apple's memory
	MOV	AH,48H
	MOV	BX,1000H
	INT	21H
	JNC	NO_ALLOC_MEM_ERR
;error allocating 64K for apple's memory
	MOV	DX,OFFSET MEM_ER_TXT
	MOV	AH,9
	INT	21H
	MOV	AX,4C08H
	INT	21H
NO_ALLOC_MEM_ERR:
;save segment
	MOV	CS:DS_SAV,AX
;set video mode
	MOV	AX,0
	INT	10H
;hide cursor
	MOV	AH,02H
	XOR	BH,BH
	MOV	DX,0FFFFH
	INT	10H
;save timer routine
	MOV	AX,351CH
	INT	21H
	MOV	CS:TIMER_SEG_SAV,ES
	MOV	CS:TIMER_OFF_SAV,BX
;load apple bios
	MOV	AX,3D00H
	MOV	DX,OFFSET BIOS_FILE_NAME
	INT	21H
	JNC	NO_OPEN_BIOS_FILE_ERROR
	MOV	DX,OFFSET BIOS_FIL_ER_TXT
	MOV	AH,9
	INT	21H
	MOV	AX,4C08H
	INT	21H
NO_OPEN_BIOS_FILE_ERROR:
	MOV	BX,AX
	MOV	AX,CS:DS_SAV
	MOV	ES,AX
	MOV	DS,AX
	MOV	AH,3FH
	MOV	CX,BIOS_LENGTH
	MOV	DX,BIOS_START
	INT	21H
	JNC	NO_READ_BIOS_FILE_ERROR
	MOV	AX,CS
	MOV	DS,AX
	MOV	DX,OFFSET BIOS_FIL_ER_TXT
	MOV	AH,9
	INT	21H
	MOV	AX,4C08H
	INT	21H
NO_READ_BIOS_FILE_ERROR:
	MOV	AH,3EH
	INT	21H
;clear rest of memory
	MOV	AX,CS:DS_SAV
	MOV	DS,AX
	MOV	DI,00H
	MOV	CX,BIOS_START
	MOV	AL,00H
	REP	STOSB
;load io cards
	MOV	SI,1
CARD_LOOP:
	MOV	AX,CS
	MOV	DS,AX
	MOV	DX,OFFSET IO_CARD_NAME
	MOV	AX,SI
	ADD	AL,'0'
	MOV	CS:IO_CARD_NUMBER,AL
	MOV	AX,3D00H
	INT	21H
	JC	IO_CARD_ERROR
	MOV	BX,AX
	MOV	AX,CS:DS_SAV
	MOV	ES,AX
	MOV	DS,AX
	MOV	AH,3FH
	MOV	CL,08H
	MOV	DX,SI
	SHL	DX,CL
	ADD	DX,APPLE_IO_AREA
	MOV	CX,IO_CARD_LENGTH
	INT	21H
	JC	IO_CARD_ERROR
	MOV	AH,3EH
	INT	21H
IO_CARD_ERROR:
	INC	SI
	CMP	SI,7
	JBE	CARD_LOOP
;set up graphics characters
	MOV	AX,0000H
	MOV	DS,AX
	MOV	WORD PTR DS:[7CH],OFFSET GR_CHAR
	MOV	AX,CS
	MOV	DS:[7EH],AX
;
;set JUMP_TAB
	CLD
	MOV	AX,CS
	MOV	DS,AX
	MOV	ES,AX
	MOV	CX,100H
	MOV	SI,OFFSET INST_TAB
	MOV	DI,OFFSET JUMP_TAB
	REP	MOVSW
;find screen type
	MOV	BX,0B800H
	MOV	AX,040H
	MOV	DS,AX
	MOV	AX,[10H]
	AND	AX,30H
	CMP	AX,30H
	JNE	COLOR_SCREEN
	MOV	BX,0B000H
COLOR_SCREEN:
;initialize registers
	MOV	ES,BX
	MOV	AX,CS:DS_SAV
	MOV	DS,AX
	MOV	AX,0
	MOV	CX,0
	MOV	BP,8001H-TRACK_LENGTH
	MOV	DI,000FFH
	MOV	SI,DS:[0FFFCH]
	MOV	CL,8
	MOV	BYTE PTR CS:CURRENT_DRIVE,1
	JMP	D1_CHANGE
;
PROCESS_FUNCTION_KEY:
;restore JUMP_TAB
	PUSH	AX
	PUSH	CX
	PUSH	SI
	PUSH	DI
	PUSH	DS
	PUSH	ES
	CLD
	MOV	AX,CS
	MOV	DS,AX
	MOV	ES,AX
	MOV	CX,100H
	MOV	SI,OFFSET INST_TAB
	MOV	DI,OFFSET JUMP_TAB
	REP	MOVSW
	POP	ES
	POP	DS
	POP	DI
	POP	SI
	POP	CX
	POP	AX
;jump to chosen function
	MOV	BL,CS:FUNCTION_CODE
	XOR	BH,BH
	SHL	BX,1
	JMP	CS:FUNC_TAB-2[BX]
;
;function key procedures
;
RESET:
;simulate reset key pressed
	MOV	SI,DS:[0FFFCH]
	OR	CH,INT
	DEC	DI
	DEC	DI
	DEC	DI
	JNS	RESET_EXIT
	ADD	DI,APPLE_STACK_LENGTH
RESET_EXIT:
	NEXT_INSTRUCTION
;
TERMINATE:
; user terminate program
	CMP	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
	JNE	NO_FINAL_WRITE
	CALL	WRITE_TRACK
NO_FINAL_WRITE:
; close diskettes
	MOV	BX,CS:D1_HANDLE
	MOV	AH,03EH
	INT	21H
	MOV	BX,CS:D2_HANDLE
	MOV	AH,03EH
	INT	21H
; close slot files
	MOV	SI,2
CLOSE_SLOT_LOOP:
	MOV	BX,CS:SLOT_HANDLE[SI]
	MOV	AH,03EH
	INT	21H
	ADD	SI,2
	CMP	SI,0EH
	JBE	CLOSE_SLOT_LOOP
	MOV	AX,CS:TIMER_SEG_SAV
	MOV	DS,AX
	MOV	DX,CS:TIMER_OFF_SAV
	MOV	AX,251CH
	INT	21H
	MOV	AX,2
	INT	10H
	INT	20H
;
LEFT_ARROW:
;simulate left arrow key pressed
	MOV	KEYBOARD_DATA,088H
	NEXT_INSTRUCTION
;
RIGHT_ARROW:
;simulate right arrow key pressed
	MOV	KEYBOARD_DATA,095H
	NEXT_INSTRUCTION
;
BLANK_PROMPT		DB	39 DUP(' '),'$'
;
BEEP	PROC	NEAR
	PUSH	CX
	PUSH	DX
	MOV	CX,200H
BP1:	CALL	CLICK
	MOV	DX,30H
BP2:
	DEC	DX
	JNE	BP2
	LOOP	BP1
	POP	DX
	POP	CX
	RET
BEEP	ENDP
;
D1_CHANGE:
	MOV	BYTE PTR CS:DRIVE_TO_SWITCH,D1
	JMP	DISK_CHANGE
;
D2_CHANGE:
	MOV	BYTE PTR CS:DRIVE_TO_SWITCH,D2
	JMP	DISK_CHANGE
;
DRIVE_TO_SWITCH		DB	?
;
DISK_CHANGE_PROMPT	DB	'D? =                               WP?N$'
;
DISK_INPUT_BUFFER LABEL BYTE
DISK_BUFFER_LENGTH	DB	1DH
DISK_INPUT_LENGTH	DB	?
DISK_FILE_NAME		DB	30 DUP(' ')
DISK_WP			DB	?
;
DISK_CHANGE:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	PUSH	SI
	PUSH	DS
;restore old timer routine
	MOV	AX,CS:TIMER_SEG_SAV
	MOV	DS,AX
	MOV	DX,CS:TIMER_OFF_SAV
	MOV	AX,251CH
	INT	21H
;set up registers
	MOV	AX,CS
	MOV	DS,AX
	MOV	AL,CS:DRIVE_TO_SWITCH
	MOV	AH,0
	DEC	AX
	MOV	DI,AX
	SHL	AX,1
	MOV	SI,AX	
;write track if necessary
	MOV	AL,CS:DRIVE_TO_SWITCH
	CMP	BYTE PTR CS:CURRENT_DRIVE,AL
	JNE	CLOSE_DISK
	CMP	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
	JNE	DISK_NO_WRITE
	CALL	WRITE_TRACK
DISK_NO_WRITE:
	MOV	BYTE PTR CS:TRACK_STATUS,UNINIT_STATUS
CLOSE_DISK:
;close diskette
	MOV	BX,CS:DISK_HANDLE[SI]
	MOV	AH,03EH
	INT	21H
DISK_INPUT:
;put cursor on bottom (unused) line
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1800H
	INT	10H
;display prompt for disk change
	MOV	AL,CS:DRIVE_TO_SWITCH
	ADD	AL,'0'
	MOV	BYTE PTR CS:DISK_CHANGE_PROMPT+1,AL
	MOV	DX,OFFSET CS:DISK_CHANGE_PROMPT
	MOV	AH,09H
	INT	21H
;input file name for DISK
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1805H
	INT	10H
	MOV	DX,OFFSET CS:DISK_INPUT_BUFFER
	MOV	AH,0AH
	INT	21H
	CMP	BYTE PTR CS:DISK_INPUT_LENGTH,0
	JNE	DISK_NAME_ENTERED
	MOV	WORD PTR CS:DISK_HANDLE[SI],0FFFFH
	MOV	BYTE PTR CS:DISK_WRITE_PROTECT_FLAG[DI],NOT_WRITE_PROTECTED
	JMP	DISK_END_INPUT
DISK_NAME_ENTERED:
	MOV	BL,CS:DISK_INPUT_LENGTH
	MOV	BH,0
;use ".DSK" if no extension entered
	PUSH	ES
	PUSH	DI
	MOV	CL,CS:DISK_INPUT_LENGTH
	MOV	CH,0
	MOV	DI,OFFSET DISK_FILE_NAME
	MOV	AX,CS
	MOV	ES,AX
	MOV	AL,'.'
	REPNE	SCASB
	JE	EXTENSION
	MOV	BYTE PTR CS:DISK_FILE_NAME+0[BX],'.'
	MOV	BYTE PTR CS:DISK_FILE_NAME+1[BX],'D'
	MOV	BYTE PTR CS:DISK_FILE_NAME+2[BX],'S'
	MOV	BYTE PTR CS:DISK_FILE_NAME+3[BX],'K'
	ADD	BX,4
EXTENSION:
	MOV	BYTE PTR CS:DISK_FILE_NAME[BX],00H
	POP	DI
	POP	ES
;input WP for DISK
	MOV	BYTE PTR CS:DISK_WP,'N'
WP_LOOP:
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1826H
	INT	10H
	MOV	AH,7
	INT	21H
	AND	AL,11011111B
	CMP	AL,0DH
	JE	WP_CR
	MOV	CS:DISK_WP,AL
	MOV	DL,AL
	MOV	AH,6
	INT	21H
	JMP	WP_LOOP
WP_CR:
	CMP	BYTE PTR CS:DISK_WP,'Y'
	JE	WP_Y
	CMP	BYTE PTR CS:DISK_WP,'N'
	JE	WP_N
	CALL	BEEP
	JMP	WP_LOOP
;
WP_Y:	MOV	BYTE PTR CS:DISK_WRITE_PROTECT_FLAG[DI],WRITE_PROTECTED
	JMP	WP_SKIP
WP_N:	MOV	BYTE PTR CS:DISK_WRITE_PROTECT_FLAG[DI],NOT_WRITE_PROTECTED
WP_SKIP:
;open diskette
	MOV	DX,OFFSET CS:DISK_FILE_NAME
	MOV	AX,03D02H
	INT	21H
	JNC	NO_DISKETTE_OPEN_ERROR
	CALL	BEEP
	JMP	DISK_INPUT
NO_DISKETTE_OPEN_ERROR:
	MOV	CS:DISK_HANDLE[SI],AX
DISK_END_INPUT:
;put cursor on bottom (unused) line
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1800H
	INT	10H
;erase prompt for disk change
	MOV	DX,OFFSET CS:BLANK_PROMPT
	MOV	AH,09H
	INT	21H
;hide cursor
	MOV	AH,02H
	XOR	BH,BH
	MOV	DX,0FFFFH
	INT	10H
;use new timer routine
	MOV	DX,OFFSET INT_ROUT
	MOV	AX,251CH
	INT	21H
;
	MOV	AL,CS:DRIVE_TO_SWITCH
	CMP	BYTE PTR CS:CURRENT_DRIVE,AL
	JNE	NOT_THIS_DRIVE
	MOV	AX,CS:DISK_HANDLE[SI]
	MOV	CS:CURRENT_DRIVE_HANDLE,AX
	MOV	AL,CS:DISK_WRITE_PROTECT_FLAG[DI]
	MOV	DS:SENSE_WRITE_PROTECT,AL
NOT_THIS_DRIVE:
	POP	DS
	POP	SI
	POP	DI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	NEXT_INSTRUCTION
;
SLOT_CHANGE_PROMPT	LABEL	BYTE
	DB	'SLOT '
SLOT_NO_DISPLAY		DB	' '
	DB	' =                               $'
SLOT_INPUT_BUFFER	LABEL BYTE
SLOT_BUFFER_LENGTH	DB	1EH
SLOT_INPUT_LENGTH	DB	?
SLOT_FILE_NAME		DB	30 DUP(' ')
SLOT_NO			DB	?
;
SLOT_IO_CHANGE:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DS
;restore old timer routine
	MOV	AX,CS:TIMER_SEG_SAV
	MOV	DS,AX
	MOV	DX,CS:TIMER_OFF_SAV
	MOV	AX,251CH
	INT	21H
;
	MOV	AX,CS
	MOV	DS,AX
;put cursor on bottom (unused) line
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1800H
	INT	10H
;display prompt for slot change
	MOV	BYTE PTR CS:SLOT_NO_DISPLAY,' '
	MOV	DX,OFFSET CS:SLOT_CHANGE_PROMPT
	MOV	AH,09H
	INT	21H
;input number of SLOT
	JMP	SHORT NO_BEEP3
NO_LOOP:
	CALL	BEEP
NO_BEEP3:
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1805H
	INT	10H
	MOV	AH,7
	INT	21H
	MOV	CS:SLOT_NO,AL
	MOV	DL,AL
	MOV	AH,6
	INT	21H
	CMP	BYTE PTR CS:SLOT_NO,'1'
	JB	NO_LOOP
	CMP	BYTE PTR CS:SLOT_NO,'7'
	JA	NO_LOOP
	CMP	BYTE PTR CS:SLOT_NO,'6'
	JE	NO_LOOP
	MOV	CS:SLOT_NO_DISPLAY,AL
	MOV	AH,0
	SUB	AL,'0'
	MOV	SI,AX
	SHL	SI,1
;close slot io
	MOV	BX,CS:SLOT_HANDLE[SI]
	MOV	AH,03EH
	INT	21H
SLOT_FILE_NAME_INPUT:
;put cursor on bottom (unused) line
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1800H
	INT	10H
;display prompt for slot change
	MOV	DX,OFFSET CS:SLOT_CHANGE_PROMPT
	MOV	AH,09H
	INT	21H
;input file name for SLOT
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1809H
	INT	10H
	MOV	DX,OFFSET CS:SLOT_INPUT_BUFFER
	MOV	AH,0AH
	INT	21H
	CMP	BYTE PTR CS:SLOT_INPUT_LENGTH,0
	JNE	SLOT_NAME_ENTERED
	MOV	WORD PTR CS:SLOT_HANDLE[SI],0FFFFH
	JMP	SLOT_END_INPUT
SLOT_NAME_ENTERED:
	MOV	BL,CS:SLOT_INPUT_LENGTH
	MOV	BH,0
	MOV	BYTE PTR CS:SLOT_FILE_NAME[BX],00H
;open slot io file
	MOV	DX,OFFSET CS:SLOT_FILE_NAME
	MOV	AX,03D02H
	INT	21H
	JNC	NO_SLOT_OPEN_ERROR
;create slot io file
	MOV	DX,OFFSET CS:SLOT_FILE_NAME
	MOV	CX,0
	MOV	AH,03CH
	INT	21H
	JNC	NO_SLOT_OPEN_ERROR
	CALL	BEEP
	JMP	SLOT_FILE_NAME_INPUT
NO_SLOT_OPEN_ERROR:
	MOV	CS:SLOT_HANDLE[SI],AX
SLOT_END_INPUT:
;put cursor on bottom (unused) line
	MOV	AH,02H
	MOV	BH,00H
	MOV	DX,1800H
	INT	10H
;erase prompt for slot io change
	MOV	DX,OFFSET CS:BLANK_PROMPT
	MOV	AH,09H
	INT	21H
;hide cursor
	MOV	AH,02H
	XOR	BH,BH
	MOV	DX,0FFFFH
	INT	10H
;use new timer routine
	MOV	DX,OFFSET INT_ROUT
	MOV	AX,251CH
	INT	21H
;
	POP	DS
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	NEXT_INSTRUCTION
;
DEBUG:
	INT	3
;
NOTHING_FUNCTION:
	NEXT_INSTRUCTION
;
FUNCTION_CODE		DB	?
;
DISK_IO_MODE		DB	READ_MODE
;
TRACK_BUFFER		DB	TRACK_LENGTH DUP(?)
;
TRACK_STATUS		DB	UNINIT_STATUS
;
CURRENT_DRIVE		DB	0
;
CURRENT_TRACK		DB	0
;
CURRENT_STEPPER		DB	0
;
NEW_STEPPER		DB	0
;
DISK_HANDLE		LABEL	WORD
;
D1_HANDLE		DW	0FFFFH
;
D2_HANDLE		DW	0FFFFH
;
D1_TRACK		DB	0
;
D2_TRACK		DB	0
;
D1_STEPPER		DB	0	
;
D2_STEPPER		DB	0	
;
CURRENT_DRIVE_HANDLE	DW	0
;
DISK_WRITE_PROTECT_FLAG	LABEL	BYTE

D1_WRITE_PROTECT_FLAG	DB	0
;
D2_WRITE_PROTECT_FLAG	DB	0
;
DELTA_TRACK		DB	0,0,0,+4,0,0,0,0,0,0,0,0,-4,0,0,0
;
PADDLE_COUNTERS		LABEL	BYTE
;
PADDLE_1_COUNTER	DB	?
;
PADDLE_2_COUNTER	DB	?
;
PADDLE_3_COUNTER	DB	?
;
PADDLE_4_COUNTER	DB	?
;
SLOT_HANDLE		DW	0,4,3,0FFFFH,0FFFFH,0FFFFH,0,0FFFFH
;
READ_TRACK	PROC	NEAR
	CMP	WORD PTR CS:CURRENT_DRIVE_HANDLE,0FFFFH
	JNE	NO_READ_EMPTY_DRIVE
	PUSH	AX
	PUSH	CX
	PUSH	DI
	PUSH	ES
	MOV	CX,TRACK_LENGTH
	MOV	AX,CS
	MOV	ES,AX
	MOV	DI,OFFSET TRACK_BUFFER
	MOV	AL,0FFH
	REP	STOSB
	MOV	BYTE PTR CS:TRACK_STATUS,READ_STATUS
	POP	ES
	POP	DI
	POP	CX
	POP	AX
	RET
NO_READ_EMPTY_DRIVE:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	DS
	MOV	BX,CS:CURRENT_DRIVE_HANDLE
	MOV	AH,00H
	MOV	AL,CS:CURRENT_TRACK
	SHR	AX,1
	MOV	CX,TRACK_LENGTH
	MUL	CX
	MOV	CX,DX
	MOV	DX,AX
	MOV	AX,04200H
	INT	21H
	MOV	BX,CS:CURRENT_DRIVE_HANDLE
	MOV	CX,TRACK_LENGTH
	MOV	AX,CS
	MOV	DS,AX
	MOV	DX,OFFSET TRACK_BUFFER
	MOV	AH,03FH
	INT	21H
	POP	DS
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	MOV	BYTE PTR CS:TRACK_STATUS,READ_STATUS
	RET
READ_TRACK	ENDP
;
WRITE_TRACK	PROC	NEAR
	CMP	WORD PTR CS:CURRENT_DRIVE_HANDLE,0FFFFH
	JNE	NO_WRITE_EMPTY_DRIVE
	MOV	BYTE PTR CS:TRACK_STATUS,READ_STATUS
	RET
NO_WRITE_EMPTY_DRIVE:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	DS
	MOV	BX,CS:CURRENT_DRIVE_HANDLE
	MOV	AH,00H
	MOV	AL,CS:CURRENT_TRACK
	SHR	AX,1
	MOV	CX,TRACK_LENGTH
	MUL	CX
	MOV	CX,DX
	MOV	DX,AX
	MOV	AX,04200H
	INT	21H
	MOV	BX,CS:CURRENT_DRIVE_HANDLE
	MOV	CX,TRACK_LENGTH
	MOV	AX,CS
	MOV	DS,AX
	MOV	DX,OFFSET TRACK_BUFFER
	MOV	AH,040H
	INT	21H
	POP	DS
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	MOV	BYTE PTR CS:TRACK_STATUS,READ_STATUS
	RET
WRITE_TRACK	ENDP
;
STEPPER_MOVE	PROC	NEAR
	TEST	BL,00000001B
	JZ	STEPPER_OFF
	CMP	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
	JNE	NOT_WRITTEN_STEPPER
	CALL	WRITE_TRACK
NOT_WRITTEN_STEPPER:
	PUSH	AX
	PUSH	BX
	AND	BX,0000000000000111B
	SHR	BL,1
	AND	BYTE PTR CS:CURRENT_TRACK,11111100B
	OR	CS:CURRENT_TRACK,BL
	MOV	CS:NEW_STEPPER,BL
	SHL	BL,1
	SHL	BL,1
	OR	BL,CS:CURRENT_STEPPER
	MOV	AL,CS:DELTA_TRACK[BX]
	ADD	BYTE PTR CS:CURRENT_TRACK,AL
	JG	NOT_TRACK_0
	MOV	BYTE PTR CS:CURRENT_TRACK,0
NOT_TRACK_0:
	CMP	BYTE PTR CS:CURRENT_TRACK,69
	JL	NOT_TRACK_69
	MOV	BYTE PTR CS:CURRENT_TRACK,69
NOT_TRACK_69:
	MOV	AL,CS:NEW_STEPPER
	MOV	CS:CURRENT_STEPPER,AL
	MOV	BYTE PTR CS:TRACK_STATUS,UNINIT_STATUS
	POP	BX
	POP	AX
STEPPER_OFF:
	RET
STEPPER_MOVE	ENDP
;
;apple io memory locations
IO_READ_MEM	PROC	NEAR
	CMP	BX,STROBE_LATCH
	JNE	NO_DISK_STROBE_R
	CMP	BYTE PTR CS:TRACK_STATUS,UNINIT_STATUS
	JNE	TRACK_INITIALIZED_R
	CALL	READ_TRACK
TRACK_INITIALIZED_R:
	INC	BP
	JG	NOT_END_OF_TRACK_R
	MOV	BP,8001H-TRACK_LENGTH
NOT_END_OF_TRACK_R:
	CMP	BYTE PTR CS:DISK_IO_MODE,READ_MODE
	JNE	WRITE_BYTE_R
	MOV	CL,CS:TRACK_BUFFER+TRACK_LENGTH-8001H[BP]
	MOV	DS:[STROBE_LATCH],CL
	MOV	CL,8
	RET
WRITE_BYTE_R:
	MOV	CL,DS:[LOAD_LATCH]
	MOV	CS:TRACK_BUFFER+TRACK_LENGTH-8001H[BP],CL
	MOV	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
NO_WRITE_BYTE_R:
	MOV	CL,8
	RET
NO_DISK_STROBE_R:
	CMP	BX,KEYBOARD_STROBE
	JNE	NO_KBD_STRB_R
	AND	KEYBOARD_DATA,01111111B
	RET
NO_KBD_STRB_R:
	CMP	BX,SPEAKER
	JNE	NO_SPEAKER_R
	CALL	CLICK
	RET
NO_SPEAKER_R:
	CMP	BX,SCN_CHG_LOW
	JB	NO_SCN_CHG_R
	CMP	BX,SCN_CHG_HIGH
	JA	NO_SCN_CHG_R
	CALL	SCN_CHG
	RET
NO_SCN_CHG_R:
	CMP	BX,PUSH_BUTTON_LOW
	JB	NO_PUSH_BUTTON_R
	CMP	BX,PUSH_BUTTON_HI
	JA	NO_PUSH_BUTTON_R
	CALL	PUSH_BUTTON
	RET
NO_PUSH_BUTTON_R:
	CMP	BX,SET_READ_MODE
	JNE	NO_CHG_DISK_MODE_R_R
	MOV	BYTE PTR CS:DISK_IO_MODE,READ_MODE
	RET
NO_CHG_DISK_MODE_R_R:
	CMP	BX,SET_WRITE_MODE
	JNE	NO_CHG_DISK_MODE_W_R
	MOV	BYTE PTR CS:DISK_IO_MODE,WRITE_MODE
	RET
NO_CHG_DISK_MODE_W_R:
	CMP	BX,STEPPER_MOVE_LOW
	JB	NO_STEPPER_MOVE_R
	CMP	BX,STEPPER_MOVE_HI
	JA	NO_STEPPER_MOVE_R
	CALL	STEPPER_MOVE
	RET
NO_STEPPER_MOVE_R:
	CMP	BX,ENABLE_1
	JNE	NO_ENABLE_1_R
	CMP	BYTE PTR CS:CURRENT_DRIVE,1
	JNE	NO_SAME_DRIVE_1_R
	RET
NO_SAME_DRIVE_1_R:
	CMP	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
	JNE	NOT_WRITTEN_1_R
	CALL	WRITE_TRACK
NOT_WRITTEN_1_R:
	PUSH	AX
	MOV	AL,CS:CURRENT_TRACK
	MOV	CS:D2_TRACK,AL
	MOV	AL,CS:CURRENT_STEPPER
	MOV	CS:D2_STEPPER,AL
	MOV	BYTE PTR CS:TRACK_STATUS,UNINIT_STATUS
	MOV	BYTE PTR CS:CURRENT_DRIVE,1
	MOV	AX,CS:D1_HANDLE
	MOV	CS:CURRENT_DRIVE_HANDLE,AX
	MOV	AL,CS:D1_WRITE_PROTECT_FLAG
	MOV	DS:SENSE_WRITE_PROTECT,AL
	MOV	AL,CS:D1_TRACK
	MOV	BYTE PTR CS:CURRENT_TRACK,AL
	MOV	AL,CS:D1_STEPPER
	MOV	BYTE PTR CS:CURRENT_STEPPER,AL
	POP	AX
	RET
NO_ENABLE_1_R:
	CMP	BX,ENABLE_2
	JNE	NO_ENABLE_2_R
	CMP	BYTE PTR CS:CURRENT_DRIVE,2
	JNE	NO_SAME_DRIVE_2_R
	RET
NO_SAME_DRIVE_2_R:
	CMP	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
	JNE	NOT_WRITTEN_2_R
	CALL	WRITE_TRACK
NOT_WRITTEN_2_R:
	PUSH	AX
	MOV	AL,CS:CURRENT_TRACK
	MOV	CS:D1_TRACK,AL
	MOV	AL,CS:CURRENT_STEPPER
	MOV	CS:D1_STEPPER,AL
	MOV	BYTE PTR CS:TRACK_STATUS,UNINIT_STATUS
	MOV	BYTE PTR CS:CURRENT_DRIVE,2
	MOV	AX,CS:D2_HANDLE
	MOV	CS:CURRENT_DRIVE_HANDLE,AX
	MOV	AL,CS:D2_WRITE_PROTECT_FLAG
	MOV	DS:SENSE_WRITE_PROTECT,AL
	MOV	AL,CS:D2_TRACK
	MOV	BYTE PTR CS:CURRENT_TRACK,AL
	MOV	AL,CS:D2_STEPPER
	MOV	BYTE PTR CS:CURRENT_STEPPER,AL
	POP	AX
	RET
NO_ENABLE_2_R:
	CMP	BX,PADDLE_STROBE
	JNE	NO_PADDLE_STROBE_R
	CALL	SET_PADDLE_VALUES
	RET
NO_PADDLE_STROBE_R:
	CMP	BX,PADDLE_LOW
	JB	NO_PADDLE_R
	CMP	BX,PADDLE_HIGH
	JA	NO_PADDLE_R
	DEC	BYTE PTR CS:PADDLE_1_COUNTER
	DEC	BYTE PTR CS:PADDLE_2_COUNTER
	DEC	BYTE PTR CS:PADDLE_3_COUNTER
	DEC	BYTE PTR CS:PADDLE_4_COUNTER
	CMP	BYTE PTR CS:PADDLE_COUNTERS-PADDLE_LOW[BX],0
	JNE	PADDLE_NOT_ZERO
	MOV	BYTE PTR DS:[BX],000H
	RET
PADDLE_NOT_ZERO:
	MOV	BYTE PTR DS:[BX],0FFH
	RET
NO_PADDLE_R:
	PUSH	DI
	MOV	DI,0C091H
SLOT_STROBE_LOOP:
	CMP	BX,DI
	JNE	NO_SLOT_STROBE
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	SUB	BX,0C081H
	SHR	BX,1
	SHR	BX,1
	SHR	BX,1
	MOV	BX,CS:SLOT_HANDLE[BX]
	MOV	CX,1
	MOV	DX,DI
	DEC	DX
	MOV	AH,40H
	INT	21H
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	POP	DI
	RET
NO_SLOT_STROBE:
	ADD	DI,10H
	CMP	DI,0C0F1H
	JBE	SLOT_STROBE_LOOP
;
	MOV	DI,0C092H
SLOT_READ_LOOP:
	CMP	BX,DI
	JNE	NO_SLOT_READ
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	SUB	BX,0C082H
	SHR	BX,1
	SHR	BX,1
	SHR	BX,1
	MOV	BX,CS:SLOT_HANDLE[BX]
	MOV	CX,1
	MOV	DX,DI
	MOV	AH,3FH
	INT	21H
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	POP	DI
	RET
NO_SLOT_READ:
	ADD	DI,10H
	CMP	DI,0C0F2H
	JBE	SLOT_READ_LOOP
	POP	DI
	RET
IO_READ_MEM	ENDP
;
IO_WRITE_MEM	PROC	NEAR
	CMP	BX,LOAD_LATCH
	JNE	NO_LOAD_LATCH
	RET
NO_LOAD_LATCH:
	CMP	BX,STROBE_LATCH
	JNE	NO_DISK_STROBE_W
	CMP	BYTE PTR CS:TRACK_STATUS,UNINIT_STATUS
	JNE	TRACK_INITIALIZED_W
	CALL	READ_TRACK
TRACK_INITIALIZED_W:
	INC	BP
	JG	NOT_END_OF_TRACK_W
	MOV	BP,8001H-TRACK_LENGTH
NOT_END_OF_TRACK_W:
	CMP	BYTE PTR CS:DISK_IO_MODE,READ_MODE
	JE	NO_WRITE_BYTE_W
	MOV	CL,DS:[LOAD_LATCH]
	MOV	CS:TRACK_BUFFER+TRACK_LENGTH-8001H[BP],CL
	MOV	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
NO_WRITE_BYTE_W:
	MOV	CL,8
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_DISK_STROBE_W:
	CMP	BX,KEYBOARD_STROBE
	JNE	NO_KBD_STRB_W
	AND	KEYBOARD_DATA,01111111B
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_KBD_STRB_W:
	CMP	BX,SPEAKER
	JNE	NO_SPEAKER_W
	CALL	CLICK
	CALL	CLICK
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_SPEAKER_W:
	CMP	BX,SCN_CHG_LOW
	JB	NO_SCN_CHG_W
	CMP	BX,SCN_CHG_HIGH
	JA	NO_SCN_CHG_W
	CALL	SCN_CHG
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_SCN_CHG_W:
	CMP	BX,SET_READ_MODE
	JNE	NO_CHG_DISK_MODE_R_W
	MOV	BYTE PTR CS:DISK_IO_MODE,READ_MODE
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_CHG_DISK_MODE_R_W:
	CMP	BX,SET_WRITE_MODE
	JNE	NO_CHG_DISK_MODE_W_W
	MOV	BYTE PTR CS:DISK_IO_MODE,WRITE_MODE
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_CHG_DISK_MODE_W_W:
	CMP	BX,STEPPER_MOVE_LOW
	JB	NO_STEPPER_MOVE_W
	CMP	BX,STEPPER_MOVE_HI
	JA	NO_STEPPER_MOVE_W
	CALL	STEPPER_MOVE
NO_STEPPER_MOVE_W:
	CMP	BX,ENABLE_1
	JNE	NO_ENABLE_1_W
	CMP	BYTE PTR CS:CURRENT_DRIVE,1
	JNE	NO_SAME_DRIVE_1_W
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_SAME_DRIVE_1_W:
	CMP	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
	JNE	NOT_WRITTEN_1_W
	CALL	WRITE_TRACK
NOT_WRITTEN_1_W:
	PUSH	AX
	MOV	AL,CS:CURRENT_TRACK
	MOV	CS:D2_TRACK,AL
	MOV	AL,CS:CURRENT_STEPPER
	MOV	CS:D2_STEPPER,AL
	MOV	BYTE PTR CS:TRACK_STATUS,UNINIT_STATUS
	MOV	BYTE PTR CS:CURRENT_DRIVE,1
	MOV	AX,CS:D1_HANDLE
	MOV	CS:CURRENT_DRIVE_HANDLE,AX
	MOV	AL,CS:D1_WRITE_PROTECT_FLAG
	MOV	DS:SENSE_WRITE_PROTECT,AL
	MOV	AL,CS:D1_TRACK
	MOV	BYTE PTR CS:CURRENT_TRACK,AL
	MOV	AL,CS:D1_STEPPER
	MOV	BYTE PTR CS:CURRENT_STEPPER,AL
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	POP	AX
	RET
NO_ENABLE_1_W:
	CMP	BX,ENABLE_2
	JNE	NO_ENABLE_2_W
	CMP	BYTE PTR CS:CURRENT_DRIVE,2
	JNE	NO_SAME_DRIVE_2_W
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_SAME_DRIVE_2_W:
	CMP	BYTE PTR CS:TRACK_STATUS,WRITTEN_STATUS
	JNE	NOT_WRITTEN_2_W
	CALL	WRITE_TRACK
NOT_WRITTEN_2_W:
	PUSH	AX
	MOV	AL,CS:CURRENT_TRACK
	MOV	CS:D1_TRACK,AL
	MOV	AL,CS:CURRENT_STEPPER
	MOV	CS:D1_STEPPER,AL
	MOV	BYTE PTR CS:TRACK_STATUS,UNINIT_STATUS
	MOV	BYTE PTR CS:CURRENT_DRIVE,2
	MOV	AX,CS:D2_HANDLE
	MOV	CS:CURRENT_DRIVE_HANDLE,AX
	MOV	AL,CS:D2_WRITE_PROTECT_FLAG
	MOV	DS:SENSE_WRITE_PROTECT,AL
	MOV	AL,CS:D2_TRACK
	MOV	BYTE PTR CS:CURRENT_TRACK,AL
	MOV	AL,CS:D2_STEPPER
	MOV	BYTE PTR CS:CURRENT_STEPPER,AL
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	POP	AX
	RET
NO_ENABLE_2_W:
	CMP	BX,PADDLE_STROBE
	JNE	NO_PADDLE_STROBE_W
	CALL	SET_PADDLE_VALUES
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
NO_PADDLE_STROBE_W:
	PUSH	DI
	MOV	DI,0C090H
SLOT_WRITE_LOOP:
	CMP	BX,DI
	JNE	NO_SLOT_WRITE
	POP	DI
	RET
NO_SLOT_WRITE:
	ADD	DI,10H
	CMP	DI,0C0F0H
	JBE	SLOT_WRITE_LOOP
	POP	DI
	MOV	BYTE PTR DS:[0CFFFH],00H
	MOV	BX,0CFFFH
	RET
IO_WRITE_MEM	ENDP
;
TEXT_1_LOW		EQU	00400H
TEXT_1_HI 		EQU	007FFH
TEXT_2_LOW		EQU	00800H	
TEXT_2_HI 		EQU	00BFFH
LOW_RES_1_LOW		EQU	00400H
LOW_RES_1_HI 		EQU	007FFH
LOW_RES_2_LOW		EQU	00800H	
LOW_RES_2_HI 		EQU	00BFFH
HI_RES_1_LOW		EQU	02000H
HI_RES_1_HI 		EQU	03FFFH
HI_RES_2_LOW		EQU	04000H	
HI_RES_2_HI 		EQU	05FFFH
;
SCN_MODE	DW	0
;
MODE_ON		DW	0000000000000001B,0000000000000010B,0000000000000100B,0000000000001000B
;
MODE_OFF	DW	1111111111111110B,1111111111111101B,1111111111111011B,1111111111110111B
;
LOW_SCN_ADDR	DW	LOW_RES_1_LOW, TEXT_1_LOW, LOW_RES_1_LOW, TEXT_1_LOW
		DW	LOW_RES_2_LOW, TEXT_2_LOW, LOW_RES_2_LOW, TEXT_2_LOW
		DW	HI_RES_1_LOW, TEXT_1_LOW, TEXT_1_LOW, TEXT_1_LOW
		DW	HI_RES_2_LOW, TEXT_2_LOW, TEXT_2_LOW, TEXT_2_LOW
;
HIGH_SCN_ADDR	DW	LOW_RES_1_HI, TEXT_1_HI, LOW_RES_1_HI, TEXT_1_HI
		DW	LOW_RES_2_HI, TEXT_2_HI, LOW_RES_2_HI, TEXT_2_HI
		DW	HI_RES_1_HI, TEXT_1_HI, HI_RES_1_HI, TEXT_1_HI
		DW	HI_RES_2_HI, TEXT_2_HI, HI_RES_2_HI, TEXT_2_HI
;
SCN_PROC_ADDR	DW	OFFSET LOW_RES_1, OFFSET TEXT_1
		DW	OFFSET LOW_RES_MIX_1, OFFSET TEXT_1
		DW	OFFSET LOW_RES_2, OFFSET TEXT_2
		DW	OFFSET LOW_RES_MIX_2, OFFSET TEXT_2
		DW	OFFSET HI_RES, OFFSET TEXT_1
		DW	OFFSET HI_RES_MIX_1, OFFSET TEXT_1
		DW	OFFSET HI_RES, OFFSET TEXT_2
		DW	OFFSET HI_RES_MIX_2, OFFSET TEXT_2
;
IBM_SCN_MODE	DW	8 DUP (0), 4 DUP (5,0)
;
SCN_LOW		DB	?
;
SCN_HI		DB	?
;
SCN_PRC		DW	?
;
SCN_CHG	PROC	NEAR
	PUSH	AX
	PUSH	BX
	PUSH	DX
	PUSH	SI
	MOV	SI,CS:SCN_MODE
	TEST	BX,0000000000000001B
	JNZ	SCN_CHG_CARRY
	AND	BX,0006H
	AND	SI,CS:MODE_OFF[BX]
	JMP	SCN_CHG_1
SCN_CHG_CARRY:
	AND	BX,0006H
	OR	SI,CS:MODE_ON[BX]
SCN_CHG_1:
	CMP	SI,CS:SCN_MODE
	JE	NO_SCN_CHG
	SHL	SI,1
	MOV	AX,CS:LOW_SCN_ADDR[SI]
	MOV	CS:SCN_LOW,AH
	MOV	AX,CS:HIGH_SCN_ADDR[SI]
	MOV	CS:SCN_HI,AH
	MOV	AX,CS:SCN_PROC_ADDR[SI]
	MOV	CS:SCN_PRC,AX
	MOV	AX,CS:IBM_SCN_MODE[SI]
	INT	10H
	TEST	SI,0012H
	JNZ	A_TEXT_MODE
	MOV	AL,08H
	MOV	DX,03D8H
	OUT	DX,AL
A_TEXT_MODE:
	MOV	AH,02H
	XOR	BH,BH
	MOV	DX,0FFFFH
	INT	10H
	MOV	AX,CS:LOW_SCN_ADDR[SI]
	MOV	DX,CS:HIGH_SCN_ADDR[SI]
SCN_DISP_LOOP:
	MOV	BX,AX
	CALL	WORD PTR CS:SCN_PRC
	INC	AX
	CMP	AX,DX
	JBE	SCN_DISP_LOOP
	SHR	SI,1
	MOV	CS:SCN_MODE,SI
NO_SCN_CHG:
	POP	SI
	POP	DX
	POP	BX
	POP	AX
	RET
SCN_CHG	ENDP
;
SET_PADDLE_VALUES	PROC	NEAR
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	MOV	BL,1
	MOV	CX,0100H
	MOV	DX,201H
	OUT	DX,AL
LOOP_1:
	IN	AL,DX
	TEST	AL,BL
	LOOPNE	LOOP_1
	MOV	CS:PADDLE_1_COUNTER,CL
;
	MOV	BL,2
	MOV	CX,0100H
	MOV	DX,201H
	OUT	DX,AL
LOOP_2:
	IN	AL,DX
	TEST	AL,BL
	LOOPNE	LOOP_2
	MOV	CS:PADDLE_2_COUNTER,CL
;
	MOV	BL,4
	MOV	CX,0100H
	MOV	DX,201H
	OUT	DX,AL
LOOP_3:
	IN	AL,DX
	TEST	AL,BL
	LOOPNE	LOOP_3
	MOV	CS:PADDLE_3_COUNTER,CL
;
	MOV	BL,8
	MOV	CX,0100H
	MOV	DX,201H
	OUT	DX,AL
LOOP_4:
	IN	AL,DX
	TEST	AL,BL
	LOOPNE	LOOP_4
	MOV	CS:PADDLE_4_COUNTER,CL
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET
SET_PADDLE_VALUES	ENDP
;
CLICK	PROC	NEAR
	PUSH	AX
	PUSH	CX
	MOV	AL,10110110B
	OUT	43H,AL
	MOV	AX,533H
	OUT	42H,AL
	MOV	AL,AH
	OUT	42H,AL
	IN	AL,61H
	MOV	AH,AL
	OR	AL,03
	OUT	61H,AL
	MOV	CX,08H
L1:
	LOOP	L1
	MOV	AL,AH
	OUT	61H,AL
	POP	CX
	POP	AX
	RET
CLICK	ENDP
;
PUSH_BUTTON_MASK	DB	00H, 10H, 20H, 40H
;
PUSH_BUTTON	PROC	NEAR
	PUSH	AX
	PUSH	DX
	PUSH	BX
	AND	BX,0003H
	MOV	DX,0201H
	IN	AL,DX
	TEST	AL,CS:PUSH_BUTTON_MASK[BX]
	JZ	BUTTON_ON
	POP	BX
	MOV	BYTE PTR [BX],00H
	POP	DX
	POP	AX
	RET
BUTTON_ON:
	POP	BX
	MOV	BYTE PTR [BX],0FFH
	POP	DX
	POP	AX
	RET
PUSH_BUTTON	ENDP
;
INST_BASE	LABEL	NEAR
L_0:	.BRK
L_1:	.ORA	INDIRECT_X
L_2:;	.NOP
	INT	3	;FOR DEBUGGING
L_3:	.NOP
L_4:	.NOP
L_5:	.ORA	ZERO_PAGE
L_6:	.ASL	ZERO_PAGE
L_7:	.NOP
L_8:	.PHP
L_9:	.ORA	IMMEDIATE
L_A:	.ASL	ACCUMULATOR
L_B:	.NOP
L_C:	.NOP
L_D:	.ORA	ABSOLUTE
L_E:	.ASL	ABSOLUTE
L_F:	.NOP
L_10:	.BPL
L_11:	.ORA	INDIRECT_Y
L_12:	.NOP
L_13:	.NOP
L_14:	.NOP
L_15:	.ORA	ZERO_PAGE_X
L_16:	.ASL	ZERO_PAGE_X
L_17:	.NOP
L_18:	.CLC
L_19:	.ORA	ABSOLUTE_Y
L_1A:	.NOP
L_1B:	.NOP
L_1C:	.NOP
L_1D:	.ORA	ABSOLUTE_X
L_1E:	.ASL	ABSOLUTE_X
L_1F:	.NOP
L_20:	.JSR
L_21:	.AND	INDIRECT_X
L_22:	.NOP
L_23:	.NOP
L_24:	.BIT	ZERO_PAGE
L_25:	.AND	ZERO_PAGE
L_26:	.ROL	ZERO_PAGE
L_27:	.NOP
L_28:	.PLP
L_29:	.AND	IMMEDIATE
L_2A:	.ROL	ACCUMULATOR
L_2B:	.NOP
L_2C:	.BIT	ABSOLUTE
L_2D:	.AND	ABSOLUTE
L_2E:	.ROL	ABSOLUTE
L_2F:	.NOP
L_30:	.BMI
L_31:	.AND	INDIRECT_Y
L_32:	.NOP
L_33:	.NOP
L_34:	.NOP
L_35:	.AND	ZERO_PAGE_X
L_36:	.ROL	ZERO_PAGE_X
L_37:	.NOP
L_38:	.SEC
L_39:	.AND	ABSOLUTE_Y
L_3A:	.NOP
L_3B:	.NOP
L_3C:	.NOP
L_3D:	.AND	ABSOLUTE_X
L_3E:	.ROL	ABSOLUTE_X
L_3F:	.NOP
L_40:	.RTI
L_41:	.EOR	INDIRECT_X
L_42:	.NOP
L_43:	.NOP
L_44:	.NOP
L_45:	.EOR	ZERO_PAGE
L_46:	.LSR	ZERO_PAGE
L_47:	.NOP
L_48:	.PHA
L_49:	.EOR	IMMEDIATE
L_4A:	.LSR	ACCUMULATOR
L_4B:	.NOP
L_4C:	.JMP	ABSOLUTE
L_4D:	.EOR	ABSOLUTE
L_4E:	.LSR	ABSOLUTE
L_4F:	.NOP
L_50:	.BVC
L_51:	.EOR	INDIRECT_Y
L_52:	.NOP
L_53:	.NOP
L_54:	.NOP
L_55:	.EOR	ZERO_PAGE_X
L_56:	.LSR	ZERO_PAGE_X
L_57:	.NOP
L_58:	.CLI
L_59:	.EOR	ABSOLUTE_Y
L_5A:	.NOP
L_5B:	.NOP
L_5C:	.NOP
L_5D:	.EOR	ABSOLUTE_X
L_5E:	.LSR	ABSOLUTE_X
L_5F:	.NOP
L_60:	.RTS
L_61:	.ADC	INDIRECT_X
L_62:	.NOP
L_63:	.NOP
L_64:	.NOP
L_65:	.ADC	ZERO_PAGE
L_66:	.ROR	ZERO_PAGE
L_67:	.NOP
L_68:	.PLA
L_69:	.ADC	IMMEDIATE
L_6A:	.ROR	ACCUMULATOR
L_6B:	.NOP
L_6C:	.JMP	INDIRECT
L_6D:	.ADC	ABSOLUTE
L_6E:	.ROR	ABSOLUTE
L_6F:	.NOP
L_70:	.BVS
L_71:	.ADC	INDIRECT_Y
L_72:	.NOP
L_73:	.NOP
L_74:	.NOP
L_75:	.ADC	ZERO_PAGE_X
L_76:	.ROR	ZERO_PAGE_X
L_77:	.NOP
L_78:	.SEI
L_79:	.ADC	ABSOLUTE_Y
L_7A:	.NOP
L_7B:	.NOP
L_7C:	.NOP
L_7D:	.ADC	ABSOLUTE_X
L_7E:	.ROR	ABSOLUTE_X
L_7F:	.NOP
L_80:	.NOP
L_81:	.STA	INDIRECT_X
L_82:	.NOP
L_83:	.NOP
L_84:	.STY	ZERO_PAGE
L_85:	.STA	ZERO_PAGE
L_86:	.STX	ZERO_PAGE
L_87:	.NOP
L_88:	.DEY
L_89:	.NOP
L_8A:	.TXA
L_8B:	.NOP
L_8C:	.STY	ABSOLUTE
L_8D:	.STA	ABSOLUTE
L_8E:	.STX	ABSOLUTE
L_8F:	.NOP
L_90:	.BCC
L_91:	.STA	INDIRECT_Y
L_92:	.NOP
L_93:	.NOP
L_94:	.STY	ZERO_PAGE_X
L_95:	.STA	ZERO_PAGE_X
L_96:	.STX	ZERO_PAGE_Y
L_97:	.NOP
L_98:	.TYA
L_99:	.STA	ABSOLUTE_Y
L_9A:	.TXS
L_9B:	.NOP
L_9C:	.NOP
L_9D:	.STA	ABSOLUTE_X
L_9E:	.NOP
L_9F:	.NOP
L_A0:	.LDY	IMMEDIATE
L_A1:	.LDA	INDIRECT_X
L_A2:	.LDX	IMMEDIATE
L_A3:	.NOP
L_A4:	.LDY	ZERO_PAGE
L_A5:	.LDA	ZERO_PAGE
L_A6:	.LDX	ZERO_PAGE
L_A7:	.NOP
L_A8:	.TAY
L_A9:	.LDA	IMMEDIATE
L_AA:	.TAX
L_AB:	.NOP
L_AC:	.LDY	ABSOLUTE
L_AD:	.LDA	ABSOLUTE
L_AE:	.LDX	ABSOLUTE
L_AF:	.NOP
L_B0:	.BCS
L_B1:	.LDA	INDIRECT_Y
L_B2:	.NOP
L_B3:	.NOP
L_B4:	.LDY	ZERO_PAGE_X
L_B5:	.LDA	ZERO_PAGE_X
L_B6:	.LDX	ZERO_PAGE_Y
L_B7:	.NOP
L_B8:	.CLV
L_B9:	.LDA	ABSOLUTE_Y
L_BA:	.TSX
L_BB:	.NOP
L_BC:	.LDY	ABSOLUTE_X
L_BD:	.LDA	ABSOLUTE_X
L_BE:	.LDX	ABSOLUTE_Y
L_BF:	.NOP
L_C0:	.CPY	IMMEDIATE
L_C1:	.CMP	INDIRECT_X
L_C2:	.NOP
L_C3:	.NOP
L_C4:	.CPY	ZERO_PAGE
L_C5:	.CMP	ZERO_PAGE
L_C6:	.DEC	ZERO_PAGE
L_C7:	.NOP
L_C8:	.INY
L_C9:	.CMP	IMMEDIATE
L_CA:	.DEX
L_CB:	.NOP
L_CC:	.CPY	ABSOLUTE
L_CD:	.CMP	ABSOLUTE
L_CE:	.DEC	ABSOLUTE
L_CF:	.NOP
L_D0:	.BNE
L_D1:	.CMP	INDIRECT_Y
L_D2:	.NOP
L_D3:	.NOP
L_D4:	.NOP
L_D5:	.CMP	ZERO_PAGE_X
L_D6:	.DEC	ZERO_PAGE_X
L_D7:	.NOP
L_D8:	.CLD
L_D9:	.CMP	ABSOLUTE_Y
L_DA:	.NOP
L_DB:	.NOP
L_DC:	.NOP
L_DD:	.CMP	ABSOLUTE_X
L_DE:	.DEC	ABSOLUTE_X
L_DF:	.NOP
L_E0:	.CPX	IMMEDIATE
L_E1:	.SBC	INDIRECT_X
L_E2:	.NOP
L_E3:	.NOP
L_E4:	.CPX	ZERO_PAGE
L_E5:	.SBC	ZERO_PAGE
L_E6:	.INC	ZERO_PAGE
L_E7:	.NOP
L_E8:	.INX
L_E9:	.SBC	IMMEDIATE
L_EA:	.NOP
L_EB:	.NOP
L_EC:	.CPX	ABSOLUTE
L_ED:	.SBC	ABSOLUTE
L_EE:	.INC	ABSOLUTE
L_EF:	.NOP
L_F0:	.BEQ
L_F1:	.SBC	INDIRECT_Y
L_F2:	.NOP
L_F3:	.NOP
L_F4:	.NOP
L_F5:	.SBC	ZERO_PAGE_X
L_F6:	.INC	ZERO_PAGE_X
L_F7:	.NOP
L_F8:	.SED
L_F9:	.SBC	ABSOLUTE_Y
L_FA:	.NOP
L_FB:	.NOP
L_FC:	.NOP
L_FD:	.SBC	ABSOLUTE_X
L_FE:	.INC	ABSOLUTE_X
L_FF:	.NOP
;
; attribute bytes
INV	EQU	70H; inverse
FLS	EQU	87H; flashing
NRM	EQU	07H; normal
;
CHAR_TABLE	LABEL	WORD
	DB	'@',INV,'A',INV,'B',INV,'C',INV,'D',INV,'E',INV,'F',INV,'G',INV
	DB	'H',INV,'I',INV,'J',INV,'K',INV,'L',INV,'M',INV,'N',INV,'O',INV
	DB	'P',INV,'Q',INV,'R',INV,'S',INV,'T',INV,'U',INV,'V',INV,'W',INV
	DB	'X',INV,'Y',INV,'Z',INV,'[',INV,'\',INV,']',INV,'^',INV,'_',INV
	DB	' ',INV,'!',INV,'"',INV,'#',INV,'$',INV,'%',INV,'&',INV,"'",INV
	DB	'(',INV,')',INV,'*',INV,'+',INV,',',INV,'-',INV,'.',INV,'/',INV
	DB	'0',INV,'1',INV,'2',INV,'3',INV,'4',INV,'5',INV,'6',INV,'7',INV
	DB	'8',INV,'9',INV,':',INV,';',INV,'<',INV,'=',INV,'>',INV,'?',INV
	DB	'@',FLS,'A',FLS,'B',FLS,'C',FLS,'D',FLS,'E',FLS,'F',FLS,'G',FLS
	DB	'H',FLS,'I',FLS,'J',FLS,'K',FLS,'L',FLS,'M',FLS,'N',FLS,'O',FLS
	DB	'P',FLS,'Q',FLS,'R',FLS,'S',FLS,'T',FLS,'U',FLS,'V',FLS,'W',FLS
	DB	'X',FLS,'Y',FLS,'Z',FLS,'[',FLS,'\',FLS,']',FLS,'^',FLS,'_',FLS
	DB	0DBH,FLS,'!',FLS,'"',FLS,'#',FLS,'$',FLS,'%',FLS,'&',FLS,"'",FLS
	DB	'(',FLS,')',FLS,'*',FLS,'+',FLS,',',FLS,'-',FLS,'.',FLS,'/',FLS
	DB	'0',FLS,'1',FLS,'2',FLS,'3',FLS,'4',FLS,'5',FLS,'6',FLS,'7',FLS
	DB	'8',FLS,'9',FLS,':',FLS,';',FLS,'<',FLS,'=',FLS,'>',FLS,'?',FLS
	DB	'@',NRM,'A',NRM,'B',NRM,'C',NRM,'D',NRM,'E',NRM,'F',NRM,'G',NRM
	DB	'H',NRM,'I',NRM,'J',NRM,'K',NRM,'L',NRM,'M',NRM,'N',NRM,'O',NRM
	DB	'P',NRM,'Q',NRM,'R',NRM,'S',NRM,'T',NRM,'U',NRM,'V',NRM,'W',NRM
	DB	'X',NRM,'Y',NRM,'Z',NRM,'[',NRM,'\',NRM,']',NRM,'^',NRM,'_',NRM
	DB	' ',NRM,'!',NRM,'"',NRM,'#',NRM,'$',NRM,'%',NRM,'&',NRM,"'",NRM
	DB	'(',NRM,')',NRM,'*',NRM,'+',NRM,',',NRM,'-',NRM,'.',NRM,'/',NRM
	DB	'0',NRM,'1',NRM,'2',NRM,'3',NRM,'4',NRM,'5',NRM,'6',NRM,'7',NRM
	DB	'8',NRM,'9',NRM,':',NRM,';',NRM,'<',NRM,'=',NRM,'>',NRM,'?',NRM
	DB	'@',NRM,'A',NRM,'B',NRM,'C',NRM,'D',NRM,'E',NRM,'F',NRM,'G',NRM
	DB	'H',NRM,'I',NRM,'J',NRM,'K',NRM,'L',NRM,'M',NRM,'N',NRM,'O',NRM
	DB	'P',NRM,'Q',NRM,'R',NRM,'S',NRM,'T',NRM,'U',NRM,'V',NRM,'W',NRM
	DB	'X',NRM,'Y',NRM,'Z',NRM,'[',NRM,'\',NRM,']',NRM,'^',NRM,'_',NRM
	DB	' ',NRM,'!',NRM,'"',NRM,'#',NRM,'$',NRM,'%',NRM,'&',NRM,"'",NRM
	DB	'(',NRM,')',NRM,'*',NRM,'+',NRM,',',NRM,'-',NRM,'.',NRM,'/',NRM
	DB	'0',NRM,'1',NRM,'2',NRM,'3',NRM,'4',NRM,'5',NRM,'6',NRM,'7',NRM
	DB	'8',NRM,'9',NRM,':',NRM,';',NRM,'<',NRM,'=',NRM,'>',NRM,'?',NRM
;
DEDLIN	MACRO
	DW	8 DUP(0CFFFH)
	ENDM
;
SCNLIN	MACRO	A
	DW	A+00H,A+02H,A+04H,A+06H,A+08H,A+0AH,A+0CH,A+0EH
	DW	A+10H,A+12H,A+14H,A+16H,A+18H,A+1AH,A+1CH,A+1EH
	DW	A+20H,A+22H,A+24H,A+26H,A+28H,A+2AH,A+2CH,A+2EH
	DW	A+30H,A+32H,A+34H,A+36H,A+38H,A+3AH,A+3CH,A+3EH
	DW	A+40H,A+42H,A+44H,A+46H,A+48H,A+4AH,A+4CH,A+4EH
	ENDM
;
POS_TABLE	LABEL	WORD
	SCNLIN	000H
	SCNLIN	280H
	SCNLIN	500H
	DEDLIN
	SCNLIN	050H
	SCNLIN	2D0H
	SCNLIN	550H
	DEDLIN
	SCNLIN	0A0H
	SCNLIN	320H
	SCNLIN	5A0H
	DEDLIN
	SCNLIN	0F0H
	SCNLIN	370H
	SCNLIN	5F0H
	DEDLIN
	SCNLIN	140H
	SCNLIN	3C0H
	SCNLIN	640H
	DEDLIN
	SCNLIN	190H
	SCNLIN	410H
	SCNLIN	690H
	DEDLIN
	SCNLIN	1E0H
	SCNLIN	460H
	SCNLIN	6E0H
	DEDLIN
	SCNLIN	230H
	SCNLIN	4B0H
	SCNLIN	730H
	DEDLIN
;
TEXT_1	PROC	NEAR
	PUSH	BP
	MOV	BP,BX
	SHL	BP,1
	MOV	BP,CS:POS_TABLE-0800H[BP]
	MOV	BL,[BX]
	XOR	BH,BH
	SHL	BX,1
	MOV	BX,CS:CHAR_TABLE[BX]
	MOV	ES:[BP],BX
	POP	BP
	RET
TEXT_1	ENDP
;
TEXT_2	PROC	NEAR
	PUSH	BP
	MOV	BP,BX
	SHL	BP,1
	MOV	BP,CS:POS_TABLE-1000H[BP]
	MOV	BL,[BX]
	XOR	BH,BH
	SHL	BX,1
	MOV	BX,CS:CHAR_TABLE[BX]
	MOV	ES:[BP],BX
	POP	BP
	RET
TEXT_2	ENDP
;
LOW_RES_1 PROC	NEAR
	PUSH	BP
	MOV	BP,BX
	SHL	BP,1
	MOV	BP,CS:POS_TABLE-0800H[BP]
	MOV	BL,[BX]
	MOV	BYTE PTR ES:[BP],0DFH
	MOV	ES:1[BP],BL
	POP	BP
	RET
LOW_RES_1 ENDP
;
LOW_RES_2 PROC	NEAR
	PUSH	BP
	MOV	BP,BX
	SHL	BP,1
	MOV	BP,CS:POS_TABLE-1000H[BP]
	MOV	BL,[BX]
	MOV	BYTE PTR ES:[BP],0DFH
	MOV	ES:1[BP],BL
	POP	BP
	RET
LOW_RES_2 ENDP
;
SCREEN_SEGMENT	DB	250H DUP(0)
		DB	28H DUP(1), 58H DUP(0), 28H DUP(1), 58H DUP(0)
		DB	28H DUP(1), 58H DUP(0), 28H DUP(1), 58H DUP(0)
;
LOW_RES_MIX_1 PROC	NEAR
	PUSH	BP
	MOV	BP,BX
	SHL	BP,1
	MOV	BP,CS:POS_TABLE-0800H[BP]
	TEST	CS:SCREEN_SEGMENT-400H[BX],1
	JNZ	TEXT_A
	MOV	BL,[BX]
	MOV	BYTE PTR ES:[BP],0DFH
	MOV	ES:1[BP],BL
	POP	BP
	RET
TEXT_A:
	MOV	BL,[BX]
	MOV	BH,0
	SHL	BX,1
	MOV	BX,CS:CHAR_TABLE[BX]
	MOV	ES:[BP],BX
	POP	BP
	RET
LOW_RES_MIX_1 ENDP
;
LOW_RES_MIX_2 PROC	NEAR
	PUSH	BP
	MOV	BP,BX
	SHL	BP,1
	MOV	BP,CS:POS_TABLE-1000H[BP]
	TEST	CS:SCREEN_SEGMENT-800H[BX],1
	JNZ	TEXT_B
	MOV	BL,[BX]
	MOV	BYTE PTR ES:[BP],0DFH
	MOV	ES:1[BP],BL
	POP	BP
	RET
TEXT_B:
	MOV	BL,[BX]
	MOV	BH,0
	SHL	BX,1
	MOV	BX,CS:CHAR_TABLE[BX]
	MOV	ES:[BP],BX
	POP	BP
	RET
LOW_RES_MIX_2 ENDP
;
GRPH_OFST	LABEL	WORD
	REPT	2
	REPT	3
	REPT	3
APPLE	=	0
	REPT	40
	DW	((APPLE * 7) MOD 4)
APPLE	=	APPLE + 1
	ENDM
	ENDM
	DW	8 DUP(0)
	ENDM
	ENDM
;
BOX_LIN	MACRO	IBM_ROW
APPLE	=	0
	REPT	40
	DW	IBM_ROW + ((APPLE * 7) / 4)
APPLE	=	APPLE + 1
	ENDM
	ENDM
;
GRPH_POS	LABEL	WORD
	BOX_LIN	0000H
	BOX_LIN	0A00H
	BOX_LIN	1400H
	DEDLIN
	BOX_LIN	0140H
	BOX_LIN	0B40H
	BOX_LIN	1540H
	DEDLIN
;
GRPH_ROWS	DW	0000H, 0280H, 0500H, 0780H, 2000H, 2280H, 2500H, 2780H
		DW	0050H, 02D0H, 0550H, 07D0H, 2050H, 22D0H, 2550H, 27D0H
		DW	00A0H, 0320H, 05A0H, 0820H, 20A0H, 2320H, 25A0H, 2820H
		DW	00F0H, 0370H, 05F0H, 0870H, 20F0H, 2370H, 25F0H, 2870H
		DW	0000H, 0280H, 0500H, 0780H, 2000H, 2280H, 2500H, 2780H
		DW	0050H, 02D0H, 0550H, 07D0H, 2050H, 22D0H, 2550H, 27D0H
		DW	00A0H, 0320H, 05A0H, 0820H, 20A0H, 2320H, 25A0H, 2820H
		DW	00F0H, 0370H, 05F0H, 0870H, 20F0H, 2370H, 25F0H, 2870H
;
DBL0	=	00000000B
DBL1	=	11000000B
DBL2	=	00110000B
DBL3	=	11110000B
DBL4	=	00001100B
DBL5	=	11001100B
DBL6	=	00111100B
DBL7	=	11111100B
DBL8	=	00000011B
DBL9	=	11000011B
DBL10	=	00110011B
DBL11	=	11110011B
DBL12	=	00001111B
DBL13	=	11001111B
DBL14	=	00111111B
DBL15	=	11111111B
;
MASK_OFF_1	DB	00000000B, 11000000B, 11110000B, 11111100B
;
MASK_BYTE	MACRO	A
	DB	DBL&A
ENDM
;
MASK_ON_1	LABEL	BYTE
LP	=	0
	REPT	80H
	MASK_BYTE	%((LP AND 0FH) * 01H)
	MASK_BYTE	%((LP AND 07H) * 02H)
	MASK_BYTE	%((LP AND 03H) * 04H)
	MASK_BYTE	%((LP AND 01H) * 08H)
LP	=	LP + 1
	ENDM
;
MASK_OFF_2	DB	00000011B, 00000000B, 00000000B, 00000000B
;
MASK_ON_2	LABEL	BYTE
LP	=	0
	REPT	80H
	MASK_BYTE	%(LP / 10H)
	MASK_BYTE	%(LP / 08H)
	MASK_BYTE	%((LP AND 3FH) / 04H)
	MASK_BYTE	%((LP AND 1FH) / 02H)
LP	=	LP + 1
	ENDM
;
MASK_OFF_3	DB	11111111B, 11111111B, 00111111B, 00001111B
;
MASK_ON_3	LABEL	BYTE
LP	=	0
	REPT	80H
	DB	0, 0
	MASK_BYTE	%(LP / 40H)
	MASK_BYTE	%(LP / 20H)
LP	=	LP + 1
	ENDM
;
HI_RES	PROC	NEAR
	PUSH	SI
	PUSH	DI
	PUSH	BP
	MOV	BP,BX
	AND	BP,00FFH
	SHL	BP,1
	MOV	SI,CS:GRPH_OFST[BP]
	MOV	DI,CS:GRPH_POS[BP]
	MOV	BP,BX
	SHR	BP,CL
	SHL	BP,1
	ADD	DI,CS:GRPH_ROWS-40H[BP]
	MOV	BP,DS:[BX]
	SHL	BP,1
	AND	BP,00FFH
	SHL	BP,1
	MOV	BL,CS:MASK_OFF_1[SI]
	AND	ES:[DI],BL
	MOV	BL,CS:MASK_ON_1[BP][SI]
	OR	ES:[DI],BL
	MOV	BL,CS:MASK_OFF_2[SI]
	AND	ES:1[DI],BL
	MOV	BL,CS:MASK_ON_2[BP][SI]
	OR	ES:1[DI],BL
	MOV	BL,CS:MASK_OFF_3[SI]
	AND	ES:2[DI],BL
	MOV	BL,CS:MASK_ON_3[BP][SI]
	OR	ES:2[DI],BL
	POP	BP
	POP	DI
	POP	SI
	RET
HI_RES	ENDP
;
ROW	DB	40 DUP(14H), 88 DUP(0), 40 DUP(15H), 88 DUP(0)
	DB	40 DUP(16H), 88 DUP(0), 40 DUP(17H), 88 DUP(0)
;
COL	LABEL	BYTE
	REPT	4
LOOP	=	0
	REPT	40
	DB	LOOP
LOOP	=	LOOP + 1
	ENDM
	DB	88 DUP(0)
	ENDM
;
GR_CHAR	DB	083H, 039H, 021H, 021H, 021H, 03FH, 087H, 0FFH
	DB	0CFH, 087H, 033H, 033H, 003H, 033H, 033H, 0FFH
	DB	003H, 099H, 099H, 083H, 099H, 099H, 003H, 0FFH
	DB	0C3H, 099H, 03FH, 03FH, 03FH, 099H, 0C3H, 0FFH
	DB	007H, 093H, 099H, 099H, 099H, 093H, 007H, 0FFH
	DB	001H, 09DH, 097H, 087H, 097H, 09DH, 001H, 0FFH
	DB	001H, 09DH, 097H, 087H, 097H, 09FH, 00FH, 0FFH
	DB	0C3H, 099H, 03FH, 03FH, 031H, 099H, 0C1H, 0FFH
	DB	033H, 033H, 033H, 003H, 033H, 033H, 033H, 0FFH
	DB	087H, 0CFH, 0CFH, 0CFH, 0CFH, 0CFH, 087H, 0FFH
	DB	0E1H, 0F3H, 0F3H, 0F3H, 033H, 033H, 087H, 0FFH
	DB	019H, 099H, 093H, 087H, 093H, 099H, 019H, 0FFH
	DB	00FH, 09FH, 09FH, 09FH, 09DH, 099H, 001H, 0FFH
	DB	039H, 011H, 001H, 001H, 029H, 039H, 039H, 0FFH
	DB	039H, 019H, 009H, 021H, 031H, 039H, 039H, 0FFH
	DB	0C7H, 093H, 039H, 039H, 039H, 093H, 0C7H, 0FFH
	DB	003H, 099H, 099H, 083H, 09FH, 09FH, 00FH, 0FFH
	DB	087H, 033H, 033H, 033H, 023H, 087H, 0E3H, 0FFH
	DB	003H, 099H, 099H, 083H, 093H, 099H, 019H, 0FFH
	DB	087H, 033H, 01FH, 08FH, 0E3H, 033H, 087H, 0FFH
	DB	003H, 04BH, 0CFH, 0CFH, 0CFH, 0CFH, 087H, 0FFH
	DB	033H, 033H, 033H, 033H, 033H, 033H, 003H, 0FFH
	DB	033H, 033H, 033H, 033H, 033H, 087H, 0CFH, 0FFH
	DB	039H, 039H, 039H, 029H, 001H, 011H, 039H, 0FFH
	DB	039H, 039H, 093H, 0C7H, 0C7H, 093H, 039H, 0FFH
	DB	033H, 033H, 033H, 087H, 0CFH, 0CFH, 087H, 0FFH
	DB	001H, 039H, 073H, 0E7H, 0CDH, 099H, 001H, 0FFH
	DB	087H, 09FH, 09FH, 09FH, 09FH, 09FH, 087H, 0FFH
	DB	03FH, 09FH, 0CFH, 0E7H, 0F3H, 0F9H, 0FDH, 0FFH
	DB	087H, 0E7H, 0E7H, 0E7H, 0E7H, 0E7H, 087H, 0FFH
	DB	0EFH, 0C7H, 093H, 039H, 0FFH, 0FFH, 0FFH, 0FFH
	DB	0FFH, 0FFH, 0FFH, 0FFH, 0FFH, 0FFH, 0FFH, 000H
	DB	0FFH, 0FFH, 0FFH, 0FFH, 0FFH, 0FFH, 0FFH, 0FFH
	DB	0CFH, 087H, 087H, 0CFH, 0CFH, 0FFH, 0CFH, 0FFH
	DB	093H, 093H, 093H, 0FFH, 0FFH, 0FFH, 0FFH, 0FFH
	DB	093H, 093H, 001H, 093H, 001H, 093H, 093H, 0FFH
	DB	0CFH, 083H, 03FH, 087H, 0F3H, 007H, 0CFH, 0FFH
	DB	0FFH, 039H, 033H, 0E7H, 0CFH, 099H, 039H, 0FFH
	DB	0C7H, 093H, 0C7H, 089H, 023H, 033H, 089H, 0FFH
	DB	09FH, 09FH, 03FH, 0FFH, 0FFH, 0FFH, 0FFH, 0FFH
	DB	0E7H, 0CFH, 09FH, 09FH, 09FH, 0CFH, 0E7H, 0FFH
	DB	09FH, 0CFH, 0E7H, 0E7H, 0E7H, 0CFH, 09FH, 0FFH
	DB	0FFH, 099H, 0C3H, 000H, 0C3H, 099H, 0FFH, 0FFH
	DB	0FFH, 0CFH, 0CFH, 003H, 0CFH, 0CFH, 0FFH, 0FFH
	DB	0FFH, 0FFH, 0FFH, 0FFH, 0FFH, 0CFH, 0CFH, 09FH
	DB	0FFH, 0FFH, 0FFH, 003H, 0FFH, 0FFH, 0FFH, 0FFH
	DB	0FFH, 0FFH, 0FFH, 0FFH, 0FFH, 0CFH, 0CFH, 0FFH
	DB	0F9H, 0F3H, 0E7H, 0CFH, 09FH, 03FH, 07FH, 0FFH
	DB	083H, 039H, 031H, 021H, 009H, 019H, 083H, 0FFH
	DB	0CFH, 08FH, 0CFH, 0CFH, 0CFH, 0CFH, 003H, 0FFH
	DB	087H, 033H, 0F3H, 0C7H, 09FH, 033H, 003H, 0FFH
	DB	087H, 033H, 0F3H, 0C7H, 0F3H, 033H, 087H, 0FFH
	DB	0E3H, 0C3H, 093H, 033H, 001H, 0F3H, 0E1H, 0FFH
	DB	003H, 03FH, 007H, 0F3H, 0F3H, 033H, 087H, 0FFH
	DB	0C7H, 09FH, 03FH, 007H, 033H, 033H, 087H, 0FFH
	DB	003H, 033H, 0F3H, 0E7H, 0CFH, 0CFH, 0CFH, 0FFH
	DB	087H, 033H, 033H, 087H, 033H, 033H, 087H, 0FFH
	DB	087H, 033H, 033H, 083H, 0F3H, 0E7H, 08FH, 0FFH
	DB	0FFH, 0CFH, 0CFH, 0FFH, 0FFH, 0CFH, 0CFH, 0FFH
	DB	0FFH, 0CFH, 0CFH, 0FFH, 0FFH, 0CFH, 0CFH, 09FH
	DB	0E7H, 0CFH, 09FH, 03FH, 09FH, 0CFH, 0E7H, 0FFH
	DB	0FFH, 0FFH, 003H, 0FFH, 0FFH, 003H, 0FFH, 0FFH
	DB	09FH, 0CFH, 0E7H, 0F3H, 0E7H, 0CFH, 09FH, 0FFH
	DB	087H, 033H, 0F3H, 0E7H, 0CFH, 0FFH, 0CFH, 0FFH
	DB	07CH, 0C6H, 0DEH, 0DEH, 0DEH, 0C0H, 078H, 000H
	DB	030H, 078H, 0CCH, 0CCH, 0FCH, 0CCH, 0CCH, 000H
	DB	0FCH, 066H, 066H, 07CH, 066H, 066H, 0FCH, 000H
	DB	03CH, 066H, 0C0H, 0C0H, 0C0H, 066H, 03CH, 000H
	DB	0F8H, 06CH, 066H, 066H, 066H, 06CH, 0F8H, 000H
	DB	0FEH, 062H, 068H, 078H, 068H, 062H, 0FEH, 000H
	DB	0FEH, 062H, 068H, 078H, 068H, 060H, 0F0H, 000H
	DB	03CH, 066H, 0C0H, 0C0H, 0CEH, 066H, 03EH, 000H
	DB	0CCH, 0CCH, 0CCH, 0FCH, 0CCH, 0CCH, 0CCH, 000H
	DB	078H, 030H, 030H, 030H, 030H, 030H, 078H, 000H
	DB	01EH, 00CH, 00CH, 00CH, 0CCH, 0CCH, 078H, 000H
	DB	0E6H, 066H, 06CH, 078H, 06CH, 066H, 0E6H, 000H
	DB	0F0H, 060H, 060H, 060H, 062H, 066H, 0FEH, 000H
	DB	0C6H, 0EEH, 0FEH, 0FEH, 0D6H, 0C6H, 0C6H, 000H
	DB	0C6H, 0E6H, 0F6H, 0DEH, 0CEH, 0C6H, 0C6H, 000H
	DB	038H, 06CH, 0C6H, 0C6H, 0C6H, 06CH, 038H, 000H
	DB	0FCH, 066H, 066H, 07CH, 060H, 060H, 0F0H, 000H
	DB	078H, 0CCH, 0CCH, 0CCH, 0DCH, 078H, 01CH, 000H
	DB	0FCH, 066H, 066H, 07CH, 06CH, 066H, 0E6H, 000H
	DB	078H, 0CCH, 0E0H, 070H, 01CH, 0CCH, 078H, 000H
	DB	0FCH, 0B4H, 030H, 030H, 030H, 030H, 078H, 000H
	DB	0CCH, 0CCH, 0CCH, 0CCH, 0CCH, 0CCH, 0FCH, 000H
	DB	0CCH, 0CCH, 0CCH, 0CCH, 0CCH, 078H, 030H, 000H
	DB	0C6H, 0C6H, 0C6H, 0D6H, 0FEH, 0EEH, 0C6H, 000H
	DB	0C6H, 0C6H, 06CH, 038H, 038H, 06CH, 0C6H, 000H
	DB	0CCH, 0CCH, 0CCH, 078H, 030H, 030H, 078H, 000H
	DB	0FEH, 0C6H, 08CH, 018H, 032H, 066H, 0FEH, 000H
	DB	078H, 060H, 060H, 060H, 060H, 060H, 078H, 000H
	DB	0C0H, 060H, 030H, 018H, 00CH, 006H, 002H, 000H
	DB	078H, 018H, 018H, 018H, 018H, 018H, 078H, 000H
	DB	010H, 038H, 06CH, 0C6H, 000H, 000H, 000H, 000H
	DB	000H, 000H, 000H, 000H, 000H, 000H, 000H, 0FFH
	DB	000H, 000H, 000H, 000H, 000H, 000H, 000H, 000H
	DB	030H, 078H, 078H, 030H, 030H, 000H, 030H, 000H
	DB	06CH, 06CH, 06CH, 000H, 000H, 000H, 000H, 000H
	DB	06CH, 06CH, 0FEH, 06CH, 0FEH, 06CH, 06CH, 000H
	DB	030H, 07CH, 0C0H, 078H, 00CH, 0F8H, 030H, 000H
	DB	000H, 0C6H, 0CCH, 018H, 030H, 066H, 0C6H, 000H
	DB	038H, 06CH, 038H, 076H, 0DCH, 0CCH, 076H, 000H
	DB	060H, 060H, 0C0H, 000H, 000H, 000H, 000H, 000H
	DB	018H, 030H, 060H, 060H, 060H, 030H, 018H, 000H
	DB	060H, 030H, 018H, 018H, 018H, 030H, 060H, 000H
	DB	000H, 066H, 03CH, 0FFH, 03CH, 066H, 000H, 000H
	DB	000H, 030H, 030H, 0FCH, 030H, 030H, 000H, 000H
	DB	000H, 000H, 000H, 000H, 000H, 030H, 030H, 060H
	DB	000H, 000H, 000H, 0FCH, 000H, 000H, 000H, 000H
	DB	000H, 000H, 000H, 000H, 000H, 030H, 030H, 000H
	DB	006H, 00CH, 018H, 030H, 060H, 0C0H, 080H, 000H
	DB	07CH, 0C6H, 0CEH, 0DEH, 0F6H, 0E6H, 07CH, 000H
	DB	030H, 070H, 030H, 030H, 030H, 030H, 0FCH, 000H
	DB	078H, 0CCH, 00CH, 038H, 060H, 0CCH, 0FCH, 000H
	DB	078H, 0CCH, 00CH, 038H, 00CH, 0CCH, 078H, 000H
	DB	01CH, 03CH, 06CH, 0CCH, 0FEH, 00CH, 01EH, 000H
	DB	0FCH, 0C0H, 0F8H, 00CH, 00CH, 0CCH, 078H, 000H
	DB	038H, 060H, 0C0H, 0F8H, 0CCH, 0CCH, 078H, 000H
	DB	0FCH, 0CCH, 00CH, 018H, 030H, 030H, 030H, 000H
	DB	078H, 0CCH, 0CCH, 078H, 0CCH, 0CCH, 078H, 000H
	DB	078H, 0CCH, 0CCH, 07CH, 00CH, 018H, 070H, 000H
	DB	000H, 030H, 030H, 000H, 000H, 030H, 030H, 000H
	DB	000H, 030H, 030H, 000H, 000H, 030H, 030H, 060H
	DB	018H, 030H, 060H, 0C0H, 060H, 030H, 018H, 000H
	DB	000H, 000H, 0FCH, 000H, 000H, 0FCH, 000H, 000H
	DB	060H, 030H, 018H, 00CH, 018H, 030H, 060H, 000H
	DB	078H, 0CCH, 00CH, 018H, 030H, 000H, 030H, 000H
;
HI_RES_MIX_1	PROC	NEAR
	CMP	BX,HI_RES_1_LOW
	JAE	HI_RES_MIX_1_HI_RES
	CMP	BX,TEXT_1_HI
	JBE	HI_RES_MIX_1_TEXT
DONE_A:
	RET
HI_RES_MIX_1_TEXT:
	TEST	CS:SCREEN_SEGMENT-400H[BX],1
	JZ	DONE_A
	PUSH	AX
	PUSH	CX
	PUSH	DX
	MOV	AH,02H
	MOV	DH,ROW-650H[BX]
	MOV	DL,COL-650H[BX]
	PUSH	BX
	MOV	BH,00H
	INT	10H
	POP	BX
	MOV	CX,01H
	MOV	AH,01H
	MOV	AL,[BX]
	SHL	AX,1
	SHL	AX,1
	SHR	AH,1
	SHR	AX,1
	SHR	AX,1
	MOV	BX,03H
	MOV	AH,0AH
	INT	10H
	POP	DX
	POP	CX
	POP	AX
	RET
HI_RES_MIX_1_HI_RES:
	PUSH	BP
	MOV	BP,BX
	AND	BP,03FFH
	TEST	CS:SCREEN_SEGMENT[BP],1
	JNZ	DONE_B
	PUSH	SI
	PUSH	DI
	AND	BP,00FFH
	SHL	BP,1
	MOV	SI,CS:GRPH_OFST[BP]
	MOV	DI,CS:GRPH_POS[BP]
	MOV	BP,BX
	SHR	BP,CL
	SHL	BP,1
	ADD	DI,CS:GRPH_ROWS-40H[BP]
	MOV	BP,DS:[BX]
	SHL	BP,1
	AND	BP,00FFH
	SHL	BP,1
	MOV	BL,CS:MASK_OFF_1[SI]
	AND	ES:[DI],BL
	MOV	BL,CS:MASK_ON_1[BP][SI]
	OR	ES:[DI],BL
	MOV	BL,CS:MASK_OFF_2[SI]
	AND	ES:1[DI],BL
	MOV	BL,CS:MASK_ON_2[BP][SI]
	OR	ES:1[DI],BL
	MOV	BL,CS:MASK_OFF_3[SI]
	AND	ES:2[DI],BL
	MOV	BL,CS:MASK_ON_3[BP][SI]
	OR	ES:2[DI],BL
	POP	DI
	POP	SI
DONE_B:
	POP	BP
	RET
HI_RES_MIX_1	ENDP
;
HI_RES_MIX_2	PROC	NEAR
	CMP	BX,HI_RES_2_LOW
	JAE	HI_RES_MIX_2_HI_RES
	CMP	BX,TEXT_2_HI
	JBE	HI_RES_MIX_2_TEXT
DONE_C:
	RET
HI_RES_MIX_2_TEXT:
	TEST	CS:SCREEN_SEGMENT-800H[BX],1
	JZ	DONE_C
	PUSH	AX
	PUSH	CX
	PUSH	DX
	MOV	AH,02H
	MOV	DH,ROW-0A50H[BX]
	MOV	DL,COL-0A50H[BX]
	PUSH	BX
	MOV	BH,00H
	INT	10H
	POP	BX
	MOV	CX,01H
	MOV	AH,01H
	MOV	AL,[BX]
	SHL	AX,1
	SHL	AX,1
	SHR	AH,1
	SHR	AX,1
	SHR	AX,1
	MOV	BX,03H
	MOV	AH,0AH
	INT	10H
	POP	DX
	POP	CX
	POP	AX
	RET
HI_RES_MIX_2_HI_RES:
	PUSH	BP
	MOV	BP,BX
	AND	BP,03FFH
	TEST	CS:SCREEN_SEGMENT[BP],1
	JNZ	DONE_D
	PUSH	SI
	PUSH	DI
	AND	BP,00FFH
	SHL	BP,1
	MOV	SI,CS:GRPH_OFST[BP]
	MOV	DI,CS:GRPH_POS[BP]
	MOV	BP,BX
	SHR	BP,CL
	SHL	BP,1
	ADD	DI,CS:GRPH_ROWS-40H[BP]
	MOV	BP,DS:[BX]
	SHL	BP,1
	AND	BP,00FFH
	SHL	BP,1
	MOV	BL,CS:MASK_OFF_1[SI]
	AND	ES:[DI],BL
	MOV	BL,CS:MASK_ON_1[BP][SI]
	OR	ES:[DI],BL
	MOV	BL,CS:MASK_OFF_2[SI]
	AND	ES:1[DI],BL
	MOV	BL,CS:MASK_ON_2[BP][SI]
	OR	ES:1[DI],BL
	MOV	BL,CS:MASK_OFF_3[SI]
	AND	ES:2[DI],BL
	MOV	BL,CS:MASK_ON_3[BP][SI]
	OR	ES:2[DI],BL
	POP	DI
	POP	SI
DONE_D:
	POP	BP
	RET
HI_RES_MIX_2	ENDP
;
INT_ROUT:
	PUSH	AX
	PUSH	DS
	MOV	AX,CS:DS_SAV
	MOV	DS,AX
	MOV	AH,1
	INT	16H
	JZ	NO_KEY
	MOV	AH,0
	INT	16H
	CMP	AH,59
	JB	NO_FUNCTION
	CMP	AH,68
	JA	NO_FUNCTION
	SUB	AH,58
	MOV	CS:FUNCTION_CODE,AH
	PUSH	CX
	PUSH	DI
	PUSH	ES
	CLD
	MOV	AX,CS
	MOV	ES,AX
	MOV	CX,100H
	MOV	AX,OFFSET PROCESS_FUNCTION_KEY
	MOV	DI,OFFSET JUMP_TAB
	REP	STOSW
	POP	ES
	POP	DI
	POP	CX
	JMP	SHORT NO_KEY
NO_FUNCTION:
	OR	AL,10000000B
	CMP	AL,0E0H
	JB	UPPER_CASE
	AND	AL,11011111B
UPPER_CASE:
	MOV	KEYBOARD_DATA,AL
NO_KEY:
	POP	DS
	POP	AX
	JMP	TIMER_INT_ADDR
;
;program's stack
	DB	100H DUP(?)
;
END_COM:
;
CSEG	ENDS
	END	ENTRY
